#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

from fenrirscreenreader.core.i18n import _
from fenrirscreenreader.utils import char_utils
from fenrirscreenreader.utils import line_utils


class command:
    def __init__(self):
        pass

    def initialize(self, environment):
        self.env = environment

    def shutdown(self):
        pass

    def get_description(self):
        return _("Move Review to the first character on the line")

    def run(self):
        # Check if we're in table mode first
        is_table_mode = self.env["runtime"]["TableManager"].is_table_mode()
        if is_table_mode:
            table_info = self.env["runtime"]["TableManager"].move_to_first_char_in_cell()
            if table_info:
                char_utils.present_char_for_review(
                    self.env,
                    table_info['character'],
                    interrupt=True,
                    announce_capital=True,
                    flush=False,
                )
                self.env["runtime"]["OutputManager"].present_text(
                    _("first character in cell {0}").format(table_info['column_header']),
                    interrupt=False,
                )
            else:
                self.env["runtime"]["OutputManager"].present_text(
                    _("no table data"), interrupt=True, flush=False
                )
            return
        
        # Regular line first character navigation (when not in table mode)
        cursor_pos = self.env["runtime"][
            "CursorManager"
        ].get_review_or_text_cursor()
        x, y, curr_line = line_utils.get_current_line(
            cursor_pos["x"],
            cursor_pos["y"],
            self.env["screen"]["new_content_text"],
        )
        if curr_line.isspace():
            self.env["runtime"]["OutputManager"].present_text(
                _("line is empty"), interrupt=True
            )
            return
        self.env["runtime"]["CursorManager"].set_review_cursor_position(
            (len(curr_line) - len(curr_line.lstrip())), cursor_pos["y"]
        )
        (
            self.env["screen"]["newCursorReview"]["x"],
            self.env["screen"]["newCursorReview"]["y"],
            curr_char,
        ) = char_utils.get_current_char(
            self.env["screen"]["newCursorReview"]["x"],
            self.env["screen"]["newCursorReview"]["y"],
            self.env["screen"]["new_content_text"],
        )

        char_utils.present_char_for_review(
            self.env,
            curr_char,
            interrupt=True,
            announce_capital=True,
            flush=False,
        )
        self.env["runtime"]["OutputManager"].present_text(
            _("first character in line indent {0}").format(
                str(len(curr_line) - len(curr_line.lstrip()))
            ),
            interrupt=False,
        )

    def set_callback(self, callback):
        pass
