//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.ee9.websocket.tests;

import java.net.URI;
import java.util.concurrent.TimeUnit;

import org.eclipse.jetty.ee9.servlet.ServletContextHandler;
import org.eclipse.jetty.ee9.websocket.api.Session;
import org.eclipse.jetty.ee9.websocket.api.StatusCode;
import org.eclipse.jetty.ee9.websocket.api.WebSocketConnectionListener;
import org.eclipse.jetty.ee9.websocket.client.WebSocketClient;
import org.eclipse.jetty.ee9.websocket.server.JettyWebSocketServerContainer;
import org.eclipse.jetty.ee9.websocket.server.config.JettyWebSocketServletContainerInitializer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class CloseInOnOpenTest
{
    private Server server;
    private ServerConnector connector;
    private JettyWebSocketServerContainer serverContainer;
    private WebSocketClient client;

    @BeforeEach
    public void beforeEach() throws Exception
    {
        server = new Server();

        connector = new ServerConnector(server);
        server.addConnector(connector);

        ServletContextHandler context = new ServletContextHandler();
        context.setContextPath("/");
        server.setHandler(context);

        JettyWebSocketServletContainerInitializer.configure(context, (servletContext, wsContainer) ->
            wsContainer.addMapping("/ws", (req, resp) -> new ClosingListener()));
        server.start();

        serverContainer = JettyWebSocketServerContainer.getContainer(context.getServletContext());
        assertNotNull(serverContainer);

        client = new WebSocketClient();
        client.start();
    }

    @AfterEach
    public void afterEach() throws Exception
    {
        client.stop();
        server.stop();
    }

    @Test
    public void testCloseInOnWebSocketConnect() throws Exception
    {
        URI uri = URI.create("ws://localhost:" + connector.getLocalPort() + "/ws");
        EventSocket clientEndpoint = new EventSocket();

        client.connect(clientEndpoint, uri).get(5, TimeUnit.SECONDS);
        assertTrue(clientEndpoint.closeLatch.await(5, TimeUnit.SECONDS));
        assertThat(clientEndpoint.closeCode, is(StatusCode.POLICY_VIOLATION));

        assertThat(serverContainer.getOpenSessions().size(), is(0));
    }

    public static class ClosingListener implements WebSocketConnectionListener
    {
        @Override
        public void onWebSocketConnect(Session session)
        {
            session.close(StatusCode.POLICY_VIOLATION, "I am a WS that closes immediately");
        }
    }
}
