/**********************************************************************
  OrcaInputDialog - ORCA Input Deck Dialog for Avogadro

  Copyright (C) 2014 Dagmar Lenk

  This file is part of the Avogadro molecular editor project.
  For more information, see <http://avogadro.cc/>

  Avogadro is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  Avogadro is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
 **********************************************************************/

#ifndef ORCAINPUTDIALOG_H
#define ORCAINPUTDIALOG_H

#include <QtGui/QDialog>
#include <QtCore/QSettings>
//#include <avogadro/molecule.h>

// Include autogenerated ui code
#include "ui_orcainputdialog.h"

#include "orcadata.h"

// Forward declaration of Avogadro::Molecule
namespace Avogadro {
  class Molecule;
}

namespace Avogadro {

  class OrcaInputDialog : public QDialog
  {
    // The Q_OBJECT macro must be included if a class is to have
    // signals/slots
    Q_OBJECT

  public:
    explicit OrcaInputDialog(QWidget *parent = 0, Qt::WindowFlags f = 0 );
    virtual ~OrcaInputDialog();

        void setMolecule(Molecule *molecule);
        void readSettings(QSettings&);
        void writeSettings(QSettings&) const;


        // Update Setup Widgets

        void updateBasicSetup();
        void updateAdvancedSetup();

        // Update Widgets on Page ...

        void updateBasisPage();
        void updateControlPage();
        void updateSCFPage();
        void updateDFTPage();
        void updateMP2Page();

  public slots:
        void setMode(int mode);
        void updatePreviewText();
  private slots:

        void advancedItemClicked(const QModelIndex &index );

        // Button Slots
        void resetClicked();
        void generateClicked();

        // Preview Slots
        void hideClicked();
        void previewEdited();

        // Basic Slots
        void setBasicComment();
        void setBasicCalculation(int n);

        void setBasicMethod(int n);
        void setBasicBasis(int n);
        void setBasicMultiplicity(int n);
        void setBasicCharge(int n);
        void setBasicCoordsFormat(int n);

        // Advanced Slots

        // Advanced Basis Set Slots
        void setBasisBasisSet(int n);
        void setBasisAuxBasisSet (int n);
        void setBasisAuxCorrBasisSet(int n);
        void setBasisUseEPC (bool value);
        void setBasisUseAuxEPC (bool value);
        void setBasisUseAuxCorrEPC (bool value);

        void setBasisUseRel (bool value);
        void setBasisRel (int n);
        void setBasisDKHOrder (int n);

        // Advanced Control Slots

        void setControlCalculation(int n);
        void setControlMultiplicity(int n);
        void setControlCharge(int n);
        void setControlUseCosX (bool value);
        void setControlUseDFT (bool value);
        void setControlUseMP2 (bool value);
        void setControlUseCCSD (bool value);

        // Advanced SCF Slots

        void setSCFMaxIter (int n);
        void setSCFType(int n);
        void setSCFUseDamping(bool value);
        void setSCFUseLevelShift(bool value);
        void setSCFDampFactor (double f);
        void setSCFDampError (double f);
        void setSCFLevelShift(double f);
        void setSCFLevelError (double f);
        void setSCFAccuracy(int n);
        void setSCFConverger(int n);
        void setSCF2ndConverger(int n);


        // Advanced DFT Slots


        void setDFTGrid( int n);
        void setDFTFinalGrid( int n);
        void setDFTFunctional (int n);

        // Advanced CosX Slots

        void setCosXGrid( int n);
        void setCosXFinalGrid( int n);
        void setCosXSFitting (bool value);

        // Advanced MP2 Slots - not yet implemented

        // Advanced Data Slots
        void setDataFormat(int n);
        void setDataComment();
        void setPrintLevel(int n);
        void setMOPrint(bool value);
        void setBasisPrint(bool value);


  protected:
        // Update the dialog when it is shown
        void showEvent(QShowEvent *event);

        QString saveInputFile(QString inputDeck, QString fileType, QString ext);

        QString readOutputFile();

        void connectModes();
        void connectBasic();
        void connectAdvanced();
        void connectPreview();
        void connectButtons();

  private:
        // This member provides access to all ui elements

        Ui::OrcaInputDialog ui;

        void initComboboxes();

        // Init Data

        void initBasicData();
        void initBasisData();
        void initControlData();
        void initSCFData();
        void initCosXData();
        void initDFTData ();
        void initDataData();

        bool checkDFTforRijCosX();
        void enableAllDFTFunctionals();
        // Internal data structure for the input dialog

        Molecule*           m_molecule;

        QMetaObject*        meta;


        OrcaBasicData*      basicData;

        OrcaBasisData*      basisData;
        OrcaControlData*    controlData;
        OrcaDataData*       dataData;
        OrcaSCFData*        scfData;
        OrcaDFTData*        dftData;
        OrcaCosXData*       cosXData;

        bool m_useDFT;
        bool m_useMP2;
        bool m_useCosX;

        bool m_basic;
        bool m_advanced;

        QButtonGroup *m_scfConvButtons;
        QButtonGroup *m_scfConv2ndButtons;

        QString m_output;
        QString m_savePath;

        bool m_dirty;
        bool m_warned;

        // Generate an input deck as a string
        QString generateInputDeck();
  };
}

#endif
