#!/usr/bin/env perl

use strict;
use warnings;
use 5.010;
use Encode qw(encode);
use Text::CSV;

say <<'EOF';
package Travel::Status::DE::IRIS::Stations;

use strict;
use warnings;
use 5.014;
use utf8;

use Geo::Distance;
use List::Util qw(min);
use List::UtilsBy qw(uniq_by);
use List::MoreUtils qw(firstval pairwise);
use Text::LevenshteinXS qw(distance);

# TODO Geo::Distance is kinda deprecated, it is recommended to use GIS::Distance
# instead. However, since GIS::Distance is not packaged for Debian, I'll stick
# with Geo::Distance for now (which works fine enough here)

# TODO switch to Text::Levenshtein::XS once AUR/Debian packages become available

our $VERSION = '1.09';

my @stations = (
EOF

my $csv = Text::CSV->new(
	{
		binary   => 1,
		sep_char => q{;}
	}
);
while ( my $line = <STDIN> ) {

	#	chomp $line;
	#	$line = decode('UTF-8', $line);

	my $status = $csv->parse($line);
	my @fields = $csv->fields;

	my ( $uic, $station, $name, $mot, $longitute, $latitude ) = @fields;

	if ( $station eq 'DS100' or $station eq q{} ) {
		next;
	}

	$name =~ s{!}{ }g;
	$name =~ s{^\s+}{};
	$name =~ s{\s+$}{};
	$name =~ s{\s+}{ }g;
	$name =~ s{'}{\\'}g;

	$station =~ s{ ^ EBLA     $ }{EBP}x;
	$station =~ s{ ^ EBTHP    $ }{EBTH}x;
	$station =~ s{ ^ EDO \s N $ }{EDO}x;
	$station =~ s{ ^ FBUSS    $ }{FBUS}x;
	$station =~ s{ ^ FH \s\s N$ }{FH}x;
	$station =~ s{ ^ KAREP    $ }{KARE}x;
	$station =~ s{ ^ KBR \s P $ }{KBR}x;
	$station =~ s{ ^ KDDH     $ }{KDD}x;
	$station =~ s{ ^ KDFFH    $ }{KDFF}x;
	$station =~ s{ ^ KDN \s P $ }{KDN}x;
	$station =~ s{ ^ KRY \s P $ }{KRY}x;
	$station =~ s{ ^ TS \s\s T$ }{TST}x;
	$station =~ s{ ^ TSZ \s F $ }{TSFE}x;

	printf(
		"\t['%s','%s',%s,%s,%s],\n",
		encode( 'UTF-8', $station ),
		encode( 'UTF-8', $name ),
		$uic, $longitute, $latitude
	);
}

say <<'EOF';

	[ 'XNTH',  '`t Harde' ],
	[ 'XBAAL', 'Aalter' ],
	[ 'XSA',   'Aarau' ],
	[ 'XSABO', 'Aarburg-Oftring' ],
	[ 'XFAH',  'Achiet' ],
	[ 'XSAA',  'Affoltern Albis' ],
	[ 'XFAG',  'Agde' ],
	[ 'XSAL',  'Aigle' ],
	[ 'XFAM',  'Aime-la-Plagne' ],
	[ 'XIAE',  'Airole' ],
	[ 'XSAI',  'Airolo' ],
	[ 'XFAX',  'Aix-en-Prov TGV' ],
	[ 'XFAI',  'Aix-les-Bains' ],
	[ 'XIAO',  'Alassio' ],
	[ 'XUAI',  'Alba Iulia' ],
	[ 'XIAT',  'Albate Camerlata' ],
	[ 'XIAB',  'Albenga' ],
	[ 'XFAL',  'Albertville' ],
	[ 'XFALN', 'Alencon' ],
	[ 'XFALE', 'Ales' ],
	[ 'XBAE',  'Alken BE' ],
	[ 'XNAL',  'Alkmaar' ],
	[ 'XAAH',  'Allerheiligenh' ],
	[ 'XNAO',  'Almelo' ],
	[ 'XNAOO', 'Almere Oostvaard' ],
	[ 'XNAR',  'Alphen a d Rijn' ],
	[ 'XSADF', 'Altdorf' ],
	[ 'XSAD',  'Altendorf' ],
	[ 'XAAL',  'Altenmarkt i.P.' ],
	[ 'XSAT',  'Altnau' ],
	[ 'XSAS',  'Altstätten SG' ],
	[ 'XFAE',  'Amberieu' ],
	[ 'XNAMV', 'Amersf Vathorst' ],
	[ 'XNAM',  'Amersfoort' ],
	[ 'XNAS',  'Amst Sloterdijk' ],
	[ 'XNAB',  'Amsterd Bijlmer' ],
	[ 'XNAC',  'Amsterdam C' ],
	[ 'XNAZ',  'Amsterdam Zuid' ],
	[ 'XNAA',  'Amsterdam-Amstel' ],
	[ 'XAAS',  'Amstetten' ],
	[ 'XSANF', 'Andelfingen' ],
	[ 'XSAN',  'Andermatt' ],
	[ 'XAAD',  'Andorf' ],
	[ 'XBAL',  'Angleur' ],
	[ 'XFANG', 'Angouleme' ],
	[ 'XBANS', 'Ans' ],
	[ 'XFAN',  'Antibes' ],
	[ 'XBAC',  'Antwerpen-Centr' ],
	[ 'XBAZ',  'Antwerpen-Zuid' ],
	[ 'XFAP',  'Apach' ],
	[ 'XNAE',  'Apeldoorn' ],
	[ 'XNAP',  'Appingedam' ],
	[ 'XUA',   'Arad' ],
	[ 'XSAB',  'Arbon' ],
	[ 'XSABS', 'Arbon S' ],
	[ 'XIAZ',  'Arezzo' ],
	[ 'XFAR',  'Argeles sur Mer' ],
	[ 'XFARL', 'Arles' ],
	[ 'XBAR',  'Arlon' ],
	[ 'XNAH',  'Arnhem' ],
	[ 'XAAR',  'Arnoldstein' ],
	[ 'XSAO',  'Arosa' ],
	[ 'XFAS',  'Arras' ],
	[ 'XFAY',  'Artenay' ],
	[ 'XSAG',  'Arth-Goldau' ],
	[ 'XFARV', 'Arvant' ],
	[ 'XTAS',  'As' ],
	[ 'XKA',   'Ashford' ],
	[ 'XKAI',  'Ashford Int' ],
	[ 'XBASS', 'Asse' ],
	[ 'XAAT',  'Attnang-Puchheim' ],
	[ 'XSAU',  'Au SG' ],
	[ 'XSAZ',  'Au ZH' ],
	[ 'XFAU',  'Aulnoye' ],
	[ 'XFAUM', 'Aumale' ],
	[ 'XFAUN', 'Auneau' ],
	[ 'XBAU',  'auvelais' ],
	[ 'XIAV',  'Aversa' ],
	[ 'XFAVN', 'Avesnes' ],
	[ 'XFAV',  'Avignon Centre' ],
	[ 'XFAVV', 'Avignon-TGV' ],
	[ 'XTBY',  'Babylon' ],
	[ 'XSBAE', 'Bäch' ],
	[ 'XABA',  'Bad Aussee' ],
	[ 'XABB',  'Bad Blumau' ],
	[ 'XABG',  'Bad Gastein' ],
	[ 'XABH',  'Bad Hofgastein' ],
	[ 'XAIS',  'Bad Ischl' ],
	[ 'XSRA',  'Bad Ragaz' ],
	[ 'XASBW', 'Bad Schall-Wall' ],
	[ 'XAVN',  'Bad Vigaun' ],
	[ 'XSZZ',  'Bad Zurzach' ],
	[ 'XSBA',  'Baden CH' ],
	[ 'XFBX',  'Baisieux' ],
	[ 'XSBAL', 'Balerna' ],
	[ 'XRBJ',  'Banova Jaruga' ],
	[ 'XFBH',  'Bantzenheim' ],
	[ 'XFBAY', 'Banyuls sur Mer' ],
	[ 'XEBS',  'Barcelona-S Andr' ],
	[ 'XNBA',  'Barendrecht' ],
	[ 'XFBD',  'Bar-le-Duc' ],
	[ 'XLBA',  'Bascharange-San' ],
	[ 'XSB',   'Basel SBB' ],
	[ 'XSBP',  'Basel SBB Dreisp' ],
	[ 'XSBJ',  'Basel St Johann' ],
	[ 'XSBD',  'Bassersdorf' ],
	[ 'XIBT',  'Battipaglia' ],
	[ 'XDBH',  'Baunhoej' ],
	[ 'XFBY',  'Bayonne' ],
	[ 'XBBE',  'Beernem' ],
	[ 'XMBS',  'Bekescsaba' ],
	[ 'XTBPB', 'Bela p Bezdezem' ],
	[ 'XFBF',  'Belfort' ],
	[ 'XFBA',  'Bellegarde (Ain)' ],
	[ 'XFBV',  'Belleville' ],
	[ 'XSBZ',  'Bellinzona' ],
	[ 'XSBEP', 'Belp' ],
	[ 'XLBUL', 'Belval-Lycee' ],
	[ 'XLBR',  'Belval-Redange' ],
	[ 'XLBUN', 'Belval-Universit' ],
	[ 'XLBS',  'Belvaux-Soleuvre' ],
	[ 'XTBP',  'Benesov n Plouc' ],
	[ 'XFBTR', 'Benestroff' ],
	[ 'XFBEF', 'Benfeld' ],
	[ 'XFBE',  'Bening' ],
	[ 'XJB',   'Beograd' ],
	[ 'XBBC',  'Berchem' ],
	[ 'XLBC',  'Berchem Hp LU' ],
	[ 'XSBRG', 'Berg CH' ],
	[ 'XNBZ',  'Bergen op Zoom' ],
	[ 'XFBER', 'Bergues' ],
	[ 'XSBN',  'Bergün/Bravuogn' ],
	[ 'XSBLG', 'Berlingen' ],
	[ 'XSBLS', 'Berlingen S' ],
	[ 'XSBE',  'Bern' ],
	[ 'XFBEY', 'Bernay' ],
	[ 'XTBU',  'Beroun' ],
	[ 'XFBTM', 'Berthelming' ],
	[ 'XLBG',  'Bertrange-Strass' ],
	[ 'XBBX',  'Bertrix' ],
	[ 'XFB',   'Besancon' ],
	[ 'XFBI',  'Besancon Viotte' ],
	[ 'XLXBB', 'Bettembg Gr FR' ],
	[ 'XLB',   'Bettembourg' ],
	[ 'XSBW',  'Bettwiesen' ],
	[ 'XLBZ',  'Betzdorf (LUX)' ],
	[ 'XIBVA', 'Bevera' ],
	[ 'XNBK',  'Beverwijk' ],
	[ 'XSBX',  'Bex' ],
	[ 'XFBZ',  'Beziers' ],
	[ 'XFBR',  'Biarritz' ],
	[ 'XSBC',  'Biasca' ],
	[ 'XSBBS', 'Biberist RBS' ],
	[ 'XSBB',  'Biberist RM' ],
	[ 'XABIA', 'Bichlbach-Almk' ],
	[ 'XABI',  'Bichlbach-Berw.' ],
	[ 'XSBL',  'Biel/Bienne' ],
	[ 'XBBA',  'Bierset-Awans' ],
	[ 'XTBI',  'Bilina' ],
	[ 'XDBL',  'Billum' ],
	[ 'XSBT',  'Bilten' ],
	[ 'XTBKN', 'Bily Kostel n.N.' ],
	[ 'XSBM',  'Birmensdorf' ],
	[ 'XABO',  'Bischofshofen' ],
	[ 'XFBLV', 'Blainville-D' ],
	[ 'XUBL',  'Blaj' ],
	[ 'XFBLM', 'Blanc Mesnil' ],
	[ 'XSBS',  'Blausee-Mitholz' ],
	[ 'XNBL',  'Blerick' ],
	[ 'XABD',  'Blindenmarkt' ],
	[ 'XABL',  'Bludenz' ],
	[ 'XABS',  'Böckstein' ],
	[ 'XNBD',  'Bodegraven' ],
	[ 'XSBOD', 'Bodio' ],
	[ 'XFBN',  'Boen' ],
	[ 'XABC',  'Böheimkirchen' ],
	[ 'XTBO',  'Bohumin' ],
	[ 'XPBOL', 'Boleslawiec' ],
	[ 'XFBOW', 'Bollwiller' ],
	[ 'XIBC',  'Bologna C' ],
	[ 'XIBZ',  'Bolzano/Bozen' ],
	[ 'XBBDL', 'Boondael/Boondaa' ],
	[ 'XFBJ',  'Bordeaux-St-J' ],
	[ 'XIBD',  'Bordighera' ],
	[ 'XIBOD', 'Borgo S Dalmazzo' ],
	[ 'XNBO',  'Born' ],
	[ 'XDBOP', 'Borup' ],
	[ 'XSBOH', 'Bottighofen' ],
	[ 'XDBV',  'Boulevarden' ],
	[ 'XFBB',  'Bourg-en-Bresse' ],
	[ 'XFBGS', 'Bourges' ],
	[ 'XFBM',  'Bourg-St Maurice' ],
	[ 'XFBZV', 'Bouzonville' ],
	[ 'XDBM',  'Bramming' ],
	[ 'XUBS',  'Brasov' ],
	[ 'XYB',   'Bratislava hl.st' ],
	[ 'XYBZ',  'Bratislava-Petra' ],
	[ 'XABR',  'Braunau a Inn' ],
	[ 'XTBE',  'Breclav' ],
	[ 'XTXBE', 'Breclav Gr AT' ],
	[ 'XNBR',  'Breda' ],
	[ 'XDBR',  'Bredebro' ],
	[ 'XAB',   'Bregenz' ],
	[ 'XABHF', 'Bregenz See' ],
	[ 'XFBRY', 'Breil sur Roya' ],
	[ 'XIBN',  'Brennero/Brenner' ],
	[ 'XIBCA', 'Brescia' ],
	[ 'XIBX',  'Bressan./Brixen' ],
	[ 'XBBS',  'Bressoux' ],
	[ 'XFBRT', 'Brest FR' ],
	[ 'XTBD',  'Breziny u Decina' ],
	[ 'XSBG',  'Brig' ],
	[ 'XABT',  'Brixen im Thale' ],
	[ 'XABX',  'Brixlegg' ],
	[ 'XTBR',  'Brno hl.n.' ],
	[ 'XDBN',  'Broens' ],
	[ 'XSBRH', 'Bronschhofen' ],
	[ 'XABM',  'Bruck a. d. Mur' ],
	[ 'XABK',  'Bruck a.d.Leitha' ],
	[ 'XABF',  'Bruck-Fusch' ],
	[ 'XSBRU', 'Brugg AG' ],
	[ 'XBBR',  'Brugge' ],
	[ 'XFBAH', 'Brumath' ],
	[ 'XIBB',  'Brunico/Bruneck' ],
	[ 'XSBNN', 'Brunnen' ],
	[ 'XBBME', 'Brux M Eurostar' ],
	[ 'XBBL',  'Bruxelles-Centr' ],
	[ 'XBBLX', 'Bruxelles-Lux' ],
	[ 'XBB',   'Bruxelles-Midi' ],
	[ 'XBBNL', 'Bruxelles-N-Aero' ],
	[ 'XBBN',  'Bruxelles-Nord' ],
	[ 'XSBU',  'Buchs SG' ],
	[ 'XUBN',  'Bucuresti Nord' ],
	[ 'XMBK',  'Budapest-Keleti' ],
	[ 'XMBN',  'Budapest-Nyugati' ],
	[ 'XNBP',  'Buitenpost' ],
	[ 'XSBH',  'Bülach' ],
	[ 'XFBG',  'Bully-Grenay' ],
	[ 'XDBU',  'Bur' ],
	[ 'XABU',  'Burgfried' ],
	[ 'XSBGL', 'Bürglen' ],
	[ 'XEBU',  'Burgos' ],
	[ 'XFBS',  'Busigny' ],
	[ 'XSBUS', 'Bussnang' ],
	[ 'XIBU',  'Busto Arsizio' ],
	[ 'XPBY',  'Bydgoszcz Gl' ],
	[ 'XICIG', 'C.Isarco/Gossens' ],
	[ 'XSCZ',  'Cadenazzo' ],
	[ 'XFCK',  'Calais Ville' ],
	[ 'XFFE',  'Calais-Frethun' ],
	[ 'XECM',  'Caldas de M.' ],
	[ 'XICTF', 'Campo d Tr/Freif' ],
	[ 'XFCA',  'Cannes' ],
	[ 'XFCB',  'Cannes la Bocca' ],
	[ 'XLCP',  'Capellen' ],
	[ 'XSCR',  'Capolago-Riva S' ],
	[ 'XFCAB', 'Carbonne' ],
	[ 'XFCN',  'Carcassonne' ],
	[ 'XICAR', 'Carimate' ],
	[ 'XFCNO', 'Carnoules' ],
	[ 'XICB',  'Casteld./Ehrenb.' ],
	[ 'XFCAN', 'Castelnaudary' ],
	[ 'XSCA',  'Castione-Arbedo' ],
	[ 'XFCAV', 'Cavaillon' ],
	[ 'XICTL', 'Centallo' ],
	[ 'XLCS',  'Cents-Hamm' ],
	[ 'XFCE',  'Cerbere' ],
	[ 'XTCC',  'Cernovice u Ch' ],
	[ 'XTCKA', 'Ceska Kamenice' ],
	[ 'XTC',   'Ceska Kubice' ],
	[ 'XTCL',  'Ceska Lipa hl.n.' ],
	[ 'XTCLS', 'Ceska Lipa Strel' ],
	[ 'XTTR',  'Ceska Trebova' ],
	[ 'XTCB',  'Ceske Budejovice' ],
	[ 'XTCV',  'Ceske Velenice' ],
	[ 'XFCM',  'Chalons en Champ' ],
	[ 'XFCS',  'Chalon-sur-Saone' ],
	[ 'XFCY',  'Chambery' ],
	[ 'XFCG',  'Champigneulles' ],
	[ 'XBCH',  'Charleroi-Sud' ],
	[ 'XFCAR', 'Chat Arnoux St A' ],
	[ 'XFCD',  'Chateau-du-Loir' ],
	[ 'XFCX',  'Chateauroux' ],
	[ 'XFCT',  'Chateau-Thierry' ],
	[ 'XBCL',  'Chatelet' ],
	[ 'XFCH',  'Chauny' ],
	[ 'XTCH',  'Cheb' ],
	[ 'XTCS',  'Cheb-Skalka' ],
	[ 'XFCSG', 'Chelles-Gournay' ],
	[ 'XFCGG', 'Chenay-Gagny' ],
	[ 'XBCE',  'Chenee' ],
	[ 'XFCHB', 'Cherbourg' ],
	[ 'XSCH',  'Chiasso' ],
	[ 'XICK',  'Chiusa/Klausen' ],
	[ 'XICU',  'Chiusi-Chianci.T' ],
	[ 'XTCD',  'Chodov' ],
	[ 'XPCJ',  'Chojnow' ],
	[ 'XTCO',  'Chomutov' ],
	[ 'XTCM',  'Chomutov Mesto' ],
	[ 'XTCY',  'Chotyne' ],
	[ 'XTCR',  'Chrastava' ],
	[ 'XTCA',  'Chrastava-And H' ],
	[ 'XTCHR', 'Chribska' ],
	[ 'XSC',   'Chur' ],
	[ 'XFCIG', 'Cintegabelle' ],
	[ 'XFCLE', 'Clermont-Ferrand' ],
	[ 'XFCL',  'Clerval' ],
	[ 'XLCV',  'Clervaux' ],
	[ 'XNCV',  'Coevorden' ],
	[ 'XFCLL', 'Collioure' ],
	[ 'XFC',   'Colmar' ],
	[ 'XFCQ',  'Combs la Ville-Q' ],
	[ 'XIC',   'Como S.G.' ],
	[ 'XFCP',  'Compiegne' ],
	[ 'XICGL', 'Conegliano' ],
	[ 'XFCF',  'Conflans-Jarny' ],
	[ 'XFCOM', 'Corbehem' ],
	[ 'XFCOE', 'Corbeil-Essones' ],
	[ 'XSCO',  'Cornaux' ],
	[ 'XFCOS', 'Cosne' ],
	[ 'XSCS',  'Cossonay' ],
	[ 'XFCOL', 'Coulommiers' ],
	[ 'XFCSN', 'Courcelles s N' ],
	[ 'XFCOU', 'Coutras' ],
	[ 'XFCR',  'Creil' ],
	[ 'XFCU',  'Culmont-Chalind' ],
	[ 'XFCZ',  'Culoz' ],
	[ 'XICUN', 'Cuneo' ],
	[ 'XUCU',  'Curtici' ],
	[ 'XPCD',  'Czechowice-Dzied' ],
	[ 'XPCW',  'Czerwiensk' ],
	[ 'XSDA',  'Dachsen' ],
	[ 'XSDS',  'Dagmersellen' ],
	[ 'XSDN',  'Däniken' ],
	[ 'XSDD',  'Davos Dorf' ],
	[ 'XSDP',  'Davos Platz' ],
	[ 'XFDX',  'Dax' ],
	[ 'XNDE',  'De Eschmarke' ],
	[ 'XMD',   'Debrecen' ],
	[ 'XTD',   'Decin hl.n.' ],
	[ 'XTDV',  'Decin vychod' ],
	[ 'XTD Z', 'Decin zapad.n.' ],
	[ 'XTDC',  'Decin-Certova V' ],
	[ 'XTDPR', 'Decin-Priper' ],
	[ 'XTDP',  'Decin-Prost Zleb' ],
	[ 'XNDD',  'Delden' ],
	[ 'XSDE',  'Delemont' ],
	[ 'XNDT',  'Delft' ],
	[ 'XNDZ',  'Delfzijl' ],
	[ 'XNDH',  'Den Haag C' ],
	[ 'XNDHH', 'Den Haag HS' ],
	[ 'XNDL',  'Den Helder' ],
	[ 'XBDL',  'Denderleeuw' ],
	[ 'XTDS',  'Desenice' ],
	[ 'XIDS',  'Desenzano G S' ],
	[ 'XIDE',  'Desio' ],
	[ 'XFDEE', 'Dettwiller' ],
	[ 'XUD',   'Deva' ],
	[ 'XNDV',  'Deventer' ],
	[ 'XYDN',  'DevinskaNovaVes' ],
	[ 'XIDM',  'Diano Marina' ],
	[ 'XFDM',  'Diemeringen' ],
	[ 'XBDBK', 'Diepenbeek' ],
	[ 'XSDH',  'Diessenhofen' ],
	[ 'XSDHS', 'Diessenhofen S' ],
	[ 'XSD',   'Dietlikon' ],
	[ 'XFDI',  'Dieulouard' ],
	[ 'XLDI',  'Differdange' ],
	[ 'XFD',   'Dijon' ],
	[ 'XFDN',  'Dijon PN' ],
	[ 'XLDR',  'Dippach-Reckange' ],
	[ 'XIDT',  'Dobbiaco/Toblach' ],
	[ 'XZD',   'Dobova' ],
	[ 'XZXDO', 'Dobova Gr HR' ],
	[ 'XDDS',  'Doestrup Sdrj' ],
	[ 'XTDK',  'Doksy' ],
	[ 'XFDO',  'Dole' ],
	[ 'XBDG',  'Dolhain-Gileppe' ],
	[ 'XTDH',  'Dolni Habartice' ],
	[ 'XTDD',  'Dolni Podluzi' ],
	[ 'XTDA',  'Dolni Poustevna' ],
	[ 'XTDL',  'Dolni Zleb' ],
	[ 'XTDZ',  'Dolni Zleb zast' ],
	[ 'XTDO',  'Domazlice' ],
	[ 'XTDI',  'Domina' ],
	[ 'XLDM',  'Dommeldange' ],
	[ 'XID',   'Domodossola' ],
	[ 'XNDO',  'Dordrecht' ],
	[ 'XADG',  'Dorfgastein' ],
	[ 'XADO',  'Dornbirn' ],
	[ 'XADS',  'Dornbirn-Schoren' ],
	[ 'XSDO',  'Dottikon-Din' ],
	[ 'XFDR',  'Dourges' ],
	[ 'XLDF',  'Drauffelt' ],
	[ 'XNDR',  'Dronten' ],
	[ 'XTDU',  'Duchcov' ],
	[ 'XRDS',  'Dugo Selo' ],
	[ 'XNDU',  'Duivendrecht' ],
	[ 'XFDK',  'Dunkerque' ],
	[ 'XDDY',  'Dyreby' ],
	[ 'XKEI',  'Ebbsfleet Int' ],
	[ 'XAEB',  'Eben im Pongau' ],
	[ 'XAEF',  'Ebenfurth' ],
	[ 'XFEH',  'Ebersheim' ],
	[ 'XBED',  'Ede BE' ],
	[ 'XNE',   'Ede Centrum' ],
	[ 'XNEW',  'Ede-Wageningen' ],
	[ 'XSEK',  'Effretikon' ],
	[ 'XSEG',  'Eglisau' ],
	[ 'XSEN',  'Egnach' ],
	[ 'XAEZ',  'Ehrwald(Zugsp.B)' ],
	[ 'XNEI',  'Eindhoven' ],
	[ 'XSES',  'Einsiedeln' ],
	[ 'XDEY',  'Ejby' ],
	[ 'XFEL',  'Elne' ],
	[ 'XAEL',  'Elsbethen' ],
	[ 'XSEB',  'Emmenbrücke' ],
	[ 'XBEN',  'Engis' ],
	[ 'XAEN',  'Enns' ],
	[ 'XNES',  'Enschede' ],
	[ 'XFE',   'Epernay' ],
	[ 'XFEI',  'Epierre-St L' ],
	[ 'XFEN',  'Epinal' ],
	[ 'XSE',   'Erlen' ],
	[ 'XSEM',  'Ermatingen' ],
	[ 'XSEMS', 'Ermatingen S' ],
	[ 'XBEQ',  'Erquelinnes' ],
	[ 'XFES',  'Erstein' ],
	[ 'XSEF',  'Erstfeld' ],
	[ 'XDE',   'Esbjerg' ],
	[ 'XAEU',  'Eschenau AT' ],
	[ 'XLEA',  'Esch-Sur-Alzette' ],
	[ 'XDES',  'Eskilstrup' ],
	[ 'XBES',  'Essen BE' ],
	[ 'XFETP', 'Etampes' ],
	[ 'XLET',  'Ettelbruck' ],
	[ 'XSEW',  'Etzwilen' ],
	[ 'XBEP',  'Eupen' ],
	[ 'XFEV',  'Evreux' ],
	[ 'XNEM',  'Eygelshov Markt' ],
	[ 'XAFS',  'Faak am See' ],
	[ 'XSFD',  'Faido' ],
	[ 'XFF',   'Faulquemont' ],
	[ 'XFFHL', 'Fegersheim Lipsh' ],
	[ 'XSFE',  'Fehraltorf' ],
	[ 'XAFK',  'Feldkirch' ],
	[ 'XAFT',  'Feldkirchen(Kär)' ],
	[ 'XAFX',  'Felixdorf' ],
	[ 'XMBF',  'Ferencvaros' ],
	[ 'XIFE',  'Ferrara' ],
	[ 'XAFB',  'Fieberbrunn' ],
	[ 'XEFI',  'Figueras' ],
	[ 'XSFI',  'Filisur' ],
	[ 'XIFLM', 'Finale Ligure M' ],
	[ 'XIFRS', 'Firenze S.M.N' ],
	[ 'XSFM',  'Flamatt' ],
	[ 'XEFL',  'Flassa' ],
	[ 'XAFG',  'Flaurling' ],
	[ 'XFFL',  'Flers' ],
	[ 'XAWFL', 'Floridsdorf' ],
	[ 'XSFU',  'Flüelen' ],
	[ 'XSFL',  'Flums' ],
	[ 'XAFC',  'Föderlach' ],
	[ 'XFFT',  'Fontaine' ],
	[ 'XFFA',  'Fontainebleau' ],
	[ 'XFFS',  'Fontan-Saorge' ],
	[ 'XFFB',  'Forbach' ],
	[ 'XBFM',  'Forest-Midi' ],
	[ 'XIFF',  'Fort./Franzensf.' ],
	[ 'XIF',   'Fossano' ],
	[ 'XFFLT', 'Fourchambault' ],
	[ 'XBFR',  'Fraipont' ],
	[ 'XAFA',  'Frankenmarkt' ],
	[ 'XTFA',  'Frant Lazne Auqa' ],
	[ 'XTF',   'Frantiskovy Lazn' ],
	[ 'XAFR',  'Frastanz' ],
	[ 'XSFA',  'Frauenfeld' ],
	[ 'XDF',   'Fredericia' ],
	[ 'XSFB',  'Freienbach CH' ],
	[ 'XSFF',  'Frenkendorf-Füll' ],
	[ 'XSF',   'Fribourg' ],
	[ 'XAF',   'Friesach' ],
	[ 'XDFV',  'Frisvardvej' ],
	[ 'XAFW',  'Fritzens-Wattens' ],
	[ 'XFFX',  'Frontenex' ],
	[ 'XFFD',  'Frouard' ],
	[ 'XSFR',  'Frutigen' ],
	[ 'XAFU',  'Fürnitz' ],
	[ 'XFFU',  'Futuroscope' ],
	[ 'XDGR',  'Gaarde' ],
	[ 'XFGN',  'Gagny' ],
	[ 'XFGAA', 'Gaillon-Aubevoye' ],
	[ 'XIGA',  'Gallarate' ],
	[ 'XSGM',  'Gampel-Steg' ],
	[ 'XFGD',  'Gandrange' ],
	[ 'XFGAN', 'Gannat' ],
	[ 'XAGD',  'Gänserndorf' ],
	[ 'XFGA',  'Gardanne' ],
	[ 'XPG',   'Gdansk Glowny' ],
	[ 'XPGOL', 'Gdansk Oliwa' ],
	[ 'XPGWR', 'Gdansk Wrzeszcz' ],
	[ 'XPGD',  'Gdynia Gl Os' ],
	[ 'XAGEI', 'Geinberg' ],
	[ 'XNLU',  'Geleen-Lutterade' ],
	[ 'XSGK',  'Gelterkinden' ],
	[ 'XIGF',  'Gemona Friuli' ],
	[ 'XSGE',  'Geneve' ],
	[ 'XSGEA', 'Geneve-Aeroport' ],
	[ 'XBGK',  'Genk' ],
	[ 'XIGP',  'Genova PP' ],
	[ 'XBGB',  'Gentbrugge' ],
	[ 'XBGD',  'Gent-Dampoort' ],
	[ 'XBGP',  'Gent-St-Pieters' ],
	[ 'XSGF',  'Gerlafingen' ],
	[ 'XAGE',  'Gerling/Pinzgau' ],
	[ 'XEG',   'Gerona(Viajeros)' ],
	[ 'XFGE',  'Gevrey' ],
	[ 'XAGB',  'Gießenbach' ],
	[ 'XSGNR', 'Gisikon-Root' ],
	[ 'XFGI',  'Gisors' ],
	[ 'XSGI',  'Giubiasco' ],
	[ 'XDGJ',  'Gjesing' ],
	[ 'XNGB',  'Glanerbrug' ],
	[ 'XSG',   'Glattbrugg' ],
	[ 'XSGA',  'Glattfelden' ],
	[ 'XAGL',  'Gloggnitz' ],
	[ 'XDGU',  'Glumsoe' ],
	[ 'XAGN',  'Gmünd NÖ' ],
	[ 'XPGN',  'Gniezno' ],
	[ 'XLGB',  'Goebelsmuhle' ],
	[ 'XAGA',  'Golling-Abtenau' ],
	[ 'XSGO',  'Goppenstein' ],
	[ 'XPGW',  'Gorzow Wlkp' ],
	[ 'XSGS',  'Göschenen' ],
	[ 'XSGG',  'Gossau SG' ],
	[ 'XSGL',  'Gottlieben' ],
	[ 'XAGT',  'Götzendorf' ],
	[ 'XAGZ',  'Götzis' ],
	[ 'XNG',   'Gouda' ],
	[ 'XBGY',  'Gouvy' ],
	[ 'XFGR',  'Graffenstaden' ],
	[ 'XAGR',  'Gramatneusiedl' ],
	[ 'XEGR',  'Granollers-Centr' ],
	[ 'XAGG',  'Gratw.-Gratkorn' ],
	[ 'XAG',   'Graz Hbf' ],
	[ 'XAGO',  'Graz Ost' ],
	[ 'XDGB',  'Gredstedbro' ],
	[ 'XSGRN', 'Grenchen Nord' ],
	[ 'XFG',   'Grenoble' ],
	[ 'XFGS',  'Gresy sur Isere' ],
	[ 'XFGZ',  'Gretz-Armainvill' ],
	[ 'XAGRB', 'Gries' ],
	[ 'XAGI',  'Gries/Pinzgau' ],
	[ 'XAGP',  'Griesk-Gallspach' ],
	[ 'XAGM',  'Gröbming' ],
	[ 'XBGR',  'Groenendaal' ],
	[ 'XNGR',  'Groningen' ],
	[ 'XAGS',  'Groß Schwechat' ],
	[ 'XDGA',  'Guldager' ],
	[ 'XAGK',  'Guntramsd-Kaiser' ],
	[ 'XAGUR', 'Gurten' ],
	[ 'XSGU',  'Güttingen' ],
	[ 'XMG',   'Györ' ],
	[ 'XBHA',  'Haaltert' ],
	[ 'XNHE',  'Haarlem' ],
	[ 'XNHLS', 'Haarl-Spaarnwoud' ],
	[ 'XSHW',  'Haeggenschwil-W' ],
	[ 'XAHU',  'Hagenau im Innkr' ],
	[ 'XFHD',  'Hagondange' ],
	[ 'XFHG',  'Haguenau' ],
	[ 'XAHM',  'Haiming' ],
	[ 'XAHT',  'Hall in Tirol' ],
	[ 'XAHL',  'Hallein' ],
	[ 'XAHE',  'Hallwang-Elixh.' ],
	[ 'XAHA',  'Hard-Fussach' ],
	[ 'XBHT',  'Hasselt' ],
	[ 'XAHI',  'Hatting' ],
	[ 'XSHL',  'Hauptwil' ],
	[ 'XAHAS', 'Haus' ],
	[ 'XFHA',  'Hayange' ],
	[ 'XFHZ',  'Hazebrouck' ],
	[ 'XDHED', 'Hedehusene' ],
	[ 'XDHEE', 'Hee' ],
	[ 'XSHR',  'Heerbrugg' ],
	[ 'XNHR',  'Heerenveen' ],
	[ 'XNH',   'Heerlen' ],
	[ 'XNHK',  'Heerlen d Kissel' ],
	[ 'XMXHY', 'Hegyeshal Gr AT' ],
	[ 'XMHY',  'Hegyeshalom' ],
	[ 'XAHP',  'Heiterw-Plansee' ],
	[ 'XNHM',  'Helmond' ],
	[ 'XFHE',  'Hendaye' ],
	[ 'XSHE',  'Hendschiken' ],
	[ 'XNHL',  'Hengelo' ],
	[ 'XFHB',  'Henin-Beaumont' ],
	[ 'XDHEN', 'Henne' ],
	[ 'XBHR',  'Herentals' ],
	[ 'XBHE',  'Hergenrath' ],
	[ 'XAHN',  'Hernals' ],
	[ 'XFHN',  'Herny' ],
	[ 'XFHH',  'Herrlisheim Colm' ],
	[ 'XBHL',  'Herstal' ],
	[ 'XBHZ',  'Herzele' ],
	[ 'XSHZ',  'Herzogenbuchsee' ],
	[ 'XAHB',  'Herzogenburg' ],
	[ 'XNHO',  'Hillegom' ],
	[ 'XNHI',  'Hilversum' ],
	[ 'XNHIS', 'Hilversum Sp' ],
	[ 'XDHM',  'Hjerm' ],
	[ 'XTHN',  'Hnevice' ],
	[ 'XFHC',  'Hochfelden' ],
	[ 'XAHF',  'Hochfilzen' ],
	[ 'XAHO',  'Hochzirl' ],
	[ 'XBHO',  'Hoeilaart' ],
	[ 'XDHT',  'Hoeje Taastrup' ],
	[ 'XNHH',  'Hoek v. Holland' ],
	[ 'XAHH',  'Hohenau' ],
	[ 'XAHS',  'Hohenems' ],
	[ 'XDHO',  'Holstebro' ],
	[ 'XFHM',  'Hombourg Ht' ],
	[ 'XNHD',  'Hoofddorp' ],
	[ 'XNHG',  'Hoogeveen' ],
	[ 'XNHS',  'Hoogezand-Sapp' ],
	[ 'XAHG',  'Hopfgarten' ],
	[ 'XAHGB', 'Hopfgarten / B' ],
	[ 'XSHG',  'Horgen' ],
	[ 'XSHB',  'Horn (Bodensee)' ],
	[ 'XSHBS', 'Horn(Bodensee) S' ],
	[ 'XTHB',  'Horni Blatna' ],
	[ 'XTHD',  'Horni Dvoriste' ],
	[ 'XTHK',  'Horni Kamenice' ],
	[ 'XTHP',  'Horni Podluzi' ],
	[ 'XTHPT', 'Horni Poustevna' ],
	[ 'XTHV',  'Horovice' ],
	[ 'XAHOG', 'Hörsching' ],
	[ 'XTHR',  'Hradek n. N.' ],
	[ 'XTHM',  'Hranice n Morave' ],
	[ 'XTHY',  'Hrebeny' ],
	[ 'XSHU',  'Hüntwangen-Wil' ],
	[ 'XAHUT', 'Hüttau' ],
	[ 'XBHU',  'Huy' ],
	[ 'XDHVD', 'Hviding' ],
	[ 'XDHYL', 'Hyllerslev' ],
	[ 'XPIL',  'Ilawa Glowna' ],
	[ 'XSIM',  'Immensee' ],
	[ 'XIIM',  'Imperia PM' ],
	[ 'XAIB',  'Imsterberg' ],
	[ 'XAIP',  'Imst-Pitztal' ],
	[ 'XBIM',  'Ingelmunster' ],
	[ 'XFIW',  'Ingwiller' ],
	[ 'XAI',   'Innsbruck Hbf' ],
	[ 'XAIH',  'Innsbruck Höttg' ],
	[ 'XAIW',  'Innsbruck Westbf' ],
	[ 'XPIW',  'Inowroclaw' ],
	[ 'XSIO',  'Interlaken Ost' ],
	[ 'XSI',   'Interlaken West' ],
	[ 'XAIZ',  'Inzing' ],
	[ 'XEIR',  'Irun' ],
	[ 'XFIS',  'Is sur Tille' ],
	[ 'XIIT',  'Iselle di Trasqu' ],
	[ 'XIXIT', 'Iselle Grenze CH' ],
	[ 'XRIG',  'Ivanic Grad' ],
	[ 'XTJP',  'Jablonne / P' ],
	[ 'XDJP',  'Janderup' ],
	[ 'XPJZ',  'Jankowa Zaganska' ],
	[ 'XTJU',  'Janovice n Uh' ],
	[ 'XAJD',  'Jedlersdorf' ],
	[ 'XTJL',  'Jedlova' ],
	[ 'XDJE',  'Jegum' ],
	[ 'XPJG',  'Jelenia Gora' ],
	[ 'XBJ',   'Jemelle' ],
	[ 'XAJB',  'Jenbach' ],
	[ 'XAJBZ', 'Jenbach Zillert' ],
	[ 'XZJ',   'Jesenice' ],
	[ 'XSJE',  'Jestetten' ],
	[ 'XFJM',  'Jeumont' ],
	[ 'XTJD',  'Jiretin pod Jedl' ],
	[ 'XTJI',  'Jirkov z.' ],
	[ 'XFJF',  'Joeuf' ],
	[ 'XFJY',  'Joigny' ],
	[ 'XAJ',   'Judenburg' ],
	[ 'XFJU',  'Juvisy' ],
	[ 'XTKA',  'Kadan' ],
	[ 'XSKI',  'Kaiseraugust' ],
	[ 'XFKH',  'Kalhausen' ],
	[ 'XAKAL', 'Kalsdorf' ],
	[ 'XAK',   'Kalwang' ],
	[ 'XNKM',  'Kampen Zuid' ],
	[ 'XSKA',  'Kandersteg' ],
	[ 'XAKP',  'Kapfenberg' ],
	[ 'XTKVD', 'Karlovy V dol.n.' ],
	[ 'XTKV',  'Karlovy Vary' ],
	[ 'XTKVV', 'Karlovy V-Dvory' ],
	[ 'XPKA',  'Katowice Osobowa' ],
	[ 'XLKT',  'Kautenbach' ],
	[ 'XSKH',  'Kehlhof' ],
	[ 'XMK',   'Kelebia' ],
	[ 'XMBL',  'Kelenföld' ],
	[ 'XAKM',  'Kematen/Tirol' ],
	[ 'XSKW',  'Kesswil' ],
	[ 'XNKE',  'Kesteren' ],
	[ 'XSKB',  'Kilchberg' ],
	[ 'XSKS',  'Killwangen-S' ],
	[ 'XAKB',  'Kindberg' ],
	[ 'XAKG',  'Kirchberg i.T.' ],
	[ 'XAKL',  'Kirchbichl' ],
	[ 'XAKIT', 'Kittsee' ],
	[ 'XAKIH', 'Kitz / Hahnenk' ],
	[ 'XAKI',  'Kitzbühel' ],
	[ 'XAKT',  'Klagenfurt Hbf' ],
	[ 'XTKY',  'Klatovy' ],
	[ 'XAKE',  'Kledering' ],
	[ 'XAKS',  'Klein Schwechat' ],
	[ 'XLKB',  'Kleinbettingen' ],
	[ 'XSKT',  'Kloten' ],
	[ 'XAKF',  'Knittelfeld' ],
	[ 'XSKO',  'Koblenz CH' ],
	[ 'XDKH',  'Koebenhavn H' ],
	[ 'XDKHL', 'Koebenhvs Lufthv' ],
	[ 'XDKG',  'Koege' ],
	[ 'XFKO',  'Kogenheim' ],
	[ 'XAKO',  'Kolbnitz' ],
	[ 'XDKO',  'Kolding' ],
	[ 'XTKI',  'Kolin' ],
	[ 'XMKO',  'Komarom' ],
	[ 'XPKN',  'Konin' ],
	[ 'XNKB',  'Koog Bloemwijk' ],
	[ 'XNKZ',  'Koog-Zaandijk' ],
	[ 'XMKR',  'Körmend' ],
	[ 'XDKS',  'Korsoer' ],
	[ 'XBKO',  'Kortenberg' ],
	[ 'XBKK',  'Kortrijk' ],
	[ 'XPK',   'Kostrzyn' ],
	[ 'XTKO',  'Kovarska' ],
	[ 'XPKR',  'Krakow Gl Osob' ],
	[ 'XTKRP', 'Kralovske Porici' ],
	[ 'XTKR',  'Kralupy n.Vl.' ],
	[ 'XAKA',  'Kranebitten' ],
	[ 'XZKJ',  'Kranj' ],
	[ 'XTKC',  'Kraslice' ],
	[ 'XTKP',  'Kraslice p Vlek' ],
	[ 'XTKCP', 'Kraslice Predm' ],
	[ 'XTKLP', 'Krasna Lipa' ],
	[ 'XTKLM', 'Krasna Lipa mest' ],
	[ 'XAKD',  'Krems(Donau)' ],
	[ 'XSKRB', 'Kreuzl.Bernrain' ],
	[ 'XSKR',  'Kreuzlingen' ],
	[ 'XSKL',  'Kreuzlingen Haf.' ],
	[ 'XAKK',  'Krichdorf a d Kr' ],
	[ 'XFKM',  'Krimmeri-Meinau' ],
	[ 'XTKM',  'Krimov' ],
	[ 'XNKA',  'Krommenie-As' ],
	[ 'XNKR',  'Kropswolde' ],
	[ 'XZKR',  'Krsko' ],
	[ 'XAKR',  'Krumpendorf' ],
	[ 'XPKZG', 'Krzewina Zgorzel' ],
	[ 'XPKY',  'Krzyz' ],
	[ 'XAKU',  'Kuchl' ],
	[ 'XAKN',  'Kufstein' ],
	[ 'XAKUN', 'Kundl' ],
	[ 'XPKU',  'Kunowice' ],
	[ 'XSKN',  'Küssnacht a R' ],
	[ 'XRKU',  'Kutina' ],
	[ 'XPKT',  'Kutno' ],
	[ 'XYKU',  'Kuty' ],
	[ 'XTKT',  'Kytlice' ],
	[ 'XFLBS', 'La Bastide S L' ],
	[ 'XFWBI', 'La Brigue' ],
	[ 'XFLAT', 'La Charite' ],
	[ 'XFGPL', 'La Plaine' ],
	[ 'XFLRY', 'La Roche s Yon' ],
	[ 'XFLST', 'La Souterraine' ],
	[ 'XALAA', 'Laa a d Thaya' ],
	[ 'XSLH',  'Lachen' ],
	[ 'XNZLW', 'Lage Zwaluwe' ],
	[ 'XALN',  'Lähn' ],
	[ 'XFLAE', 'L\'Aigle' ],
	[ 'XLLM',  'Lamadelaine' ],
	[ 'XALB',  'Lambach' ],
	[ 'XSLMC', 'Lamone-Cadempino' ],
	[ 'XALE',  'Landeck' ],
	[ 'XBLD',  'Landen' ],
	[ 'XNLG',  'Landgraaf' ],
	[ 'XSLQ',  'Landquart' ],
	[ 'XFLA',  'Landry' ],
	[ 'XBLO',  'Langdorp' ],
	[ 'XALA',  'Langen a.Arlberg' ],
	[ 'XSLT',  'Langenthal' ],
	[ 'XALK',  'Langkampfen' ],
	[ 'XSLL',  'Läufelfingen' ],
	[ 'XSLF',  'Laufen' ],
	[ 'XSLN',  'Laufenburg' ],
	[ 'XSLA',  'Lausanne' ],
	[ 'XSLAF', 'Lausanne-Flon' ],
	[ 'XSLS',  'Lausen' ],
	[ 'XALAH', 'Lauterach' ],
	[ 'XFLTG', 'Lauterbourg' ],
	[ 'XILVM', 'Laveno Mombello' ],
	[ 'XFLB',  'Le Bourget' ],
	[ 'XFLC',  'Le Creusot TGV' ],
	[ 'XFHV',  'Le Havre' ],
	[ 'XFLM',  'Le Mans' ],
	[ 'XFRVM', 'Le Raincy' ],
	[ 'XFTP',  'Le Treport Mers' ],
	[ 'XBLK',  'Lebbeke' ],
	[ 'XNLW',  'Leeuwarden' ],
	[ 'XPLE',  'Legnica' ],
	[ 'XALZ',  'Leibnitz' ],
	[ 'XNL',   'Leiden' ],
	[ 'XNLL',  'Leiden Lamme' ],
	[ 'XALT',  'Leithen' ],
	[ 'XNLY',  'Lelystad' ],
	[ 'XDLM',  'Lem' ],
	[ 'XALND', 'Lend' ],
	[ 'XSLW',  'Lengwil' ],
	[ 'XFLN',  'Lens' ],
	[ 'XSLB',  'Lenzburg' ],
	[ 'XALEZ', 'Lenzing' ],
	[ 'XALO',  'Leoben Hbf' ],
	[ 'XALG',  'Leogang' ],
	[ 'XBLB',  'Leopoldsburg' ],
	[ 'XALM',  'Lermoos' ],
	[ 'XFLE',  'Lerouville' ],
	[ 'XFLAC', 'Les Arcs' ],
	[ 'XFAO',  'Les-Aubrais-Orl' ],
	[ 'XZLB',  'Lesce-Bled' ],
	[ 'XLLD',  'Leudelange' ],
	[ 'XSLE',  'Leuk' ],
	[ 'XBLE',  'Leuven' ],
	[ 'XFLZ',  'Lezignan (Aude)' ],
	[ 'XTL',   'Liberec' ],
	[ 'XBLIG', 'Liege (G)' ],
	[ 'XALI',  'Lienz' ],
	[ 'XBLI',  'Liers' ],
	[ 'XSLI',  'Liestal' ],
	[ 'XALZN', 'Liezen' ],
	[ 'XFLI',  'Lille' ],
	[ 'XFLIE', 'Lille Europa' ],
	[ 'XILMN', 'Limone' ],
	[ 'XAL',   'Linz Hbf' ],
	[ 'XALW',  'Linz Wegscheid' ],
	[ 'XPLIL', 'Lipinki Luzyckie' ],
	[ 'XTLL',  'Lipova Sluknova' ],
	[ 'XZLJ',  'Litija' ],
	[ 'XTLM',  'Litomerice Mesto' ],
	[ 'XILI',  'Livorno Centrale' ],
	[ 'XZL',   'Ljubljana' ],
	[ 'XSLC',  'Locarno' ],
	[ 'XALH',  'Lochau-Hoerbranz' ],
	[ 'XDLF',  'Loeftgard' ],
	[ 'XMLO',  'Lököshaza' ],
	[ 'XFLNG', 'Longueau' ],
	[ 'XFLY',  'Longwy' ],
	[ 'XFLS',  'Lons-Le-Saunier' ],
	[ 'XALDF', 'Loosdorf' ],
	[ 'XFLR',  'Lorraine-Louv' ],
	[ 'XSLO',  'Lottstetten' ],
	[ 'XFLO',  'Lourches' ],
	[ 'XFL',   'Lourdes' ],
	[ 'XTLO',  'Lovosice' ],
	[ 'XPLU',  'Luban Slaski' ],
	[ 'XALU',  'Ludesch' ],
	[ 'XSL',   'Lugano' ],
	[ 'XTLS',  'Luh n Svatavou' ],
	[ 'XIL',   'Luino' ],
	[ 'XFLUQ', 'Lumes' ],
	[ 'XDLUN', 'Lundby' ],
	[ 'XDLD',  'Lunde DK' ],
	[ 'XDLU',  'Lunderskov' ],
	[ 'XFLUN', 'Lunel' ],
	[ 'XFLU',  'Luneville' ],
	[ 'XSLP',  'Lupfig' ],
	[ 'XALUU', 'Lustenau' ],
	[ 'XSLZ',  'Luterbach-Attish' ],
	[ 'XFLUB', 'Lutterbach' ],
	[ 'XFLT',  'Lutzelbourg' ],
	[ 'XLL',   'Luxembourg' ],
	[ 'XSLU',  'Luzern' ],
	[ 'XFLPD', 'Lyon Part-Dieu' ],
	[ 'XAL K', 'Lz-Kleinmünchen' ],
	[ 'XNMA',  'Maassluis' ],
	[ 'XNMT',  'Maastricht' ],
	[ 'XTMC',  'Machnin' ],
	[ 'XTMR',  'Machnin hrad' ],
	[ 'XFMCV', 'Macon Ville' ],
	[ 'XFMAL', 'Macon-Loche-TGV' ],
	[ 'XSMF',  'Maienfeld' ],
	[ 'XAMS',  'Maishfn-Saalbach' ],
	[ 'XTMV',  'Mala Velen' ],
	[ 'XPMA',  'Malbork' ],
	[ 'XPMAL', 'Malczyce' ],
	[ 'XAMO',  'Mallnitz-Oberv.' ],
	[ 'XVM',   'Malmö' ],
	[ 'XLMR',  'Mamer' ],
	[ 'XLMY',  'Mamer Lycee' ],
	[ 'XSMM',  'Mammern(Bodens)' ],
	[ 'XSMMS', 'Mammern(Bodens)S' ],
	[ 'XBMA',  'Manage' ],
	[ 'XSMB',  'Mannenbach' ],
	[ 'XSMBS', 'Mannenbach S' ],
	[ 'XLMN',  'Manternach' ],
	[ 'XAMG',  'Marchegg' ],
	[ 'XBMD',  'Marche-les-Dames' ],
	[ 'XBMP',  'Marchienne-au-P' ],
	[ 'XAMR',  'Marchtrenk' ],
	[ 'XZM',   'Maribor' ],
	[ 'XTMVE', 'Markvartice' ],
	[ 'XFMSS', 'Marle-sur-Serre' ],
	[ 'XBMR',  'Marloie' ],
	[ 'XFMVC', 'Marne la Vallee' ],
	[ 'XFMBL', 'Marseille Blanc' ],
	[ 'XFM',   'Marseille-St-Ch' ],
	[ 'XNMH',  'Martenshoek' ],
	[ 'XSMA',  'Martigny' ],
	[ 'XFMAG', 'Martigues' ],
	[ 'XSMW',  'Märwil' ],
	[ 'XAM',   'Matrei' ],
	[ 'XFMB',  'Maubeuge' ],
	[ 'XAMT',  'Mautern' ],
	[ 'XAMY',  'Mayrhofen' ],
	[ 'XBMCH', 'Mechelen' ],
	[ 'XUM',   'Medias' ],
	[ 'XSM',   'Meiringen' ],
	[ 'XAME',  'Melk' ],
	[ 'XSMS',  'Mels' ],
	[ 'XFMEL', 'Melun' ],
	[ 'XSME',  'Mendrisio' ],
	[ 'XFME',  'Menton' ],
	[ 'XSMEN', 'Menznau' ],
	[ 'XIMM',  'Merano/Meran' ],
	[ 'XBMER', 'Merchtem' ],
	[ 'XBME',  'Merelbeke' ],
	[ 'XLM',   'Mersch' ],
	[ 'XLMT',  'Mertert' ],
	[ 'XFMEH', 'Merxheim' ],
	[ 'XAMD',  'Messendorf' ],
	[ 'XFMZV', 'Metz-Ville' ],
	[ 'XFMU',  'Meuse' ],
	[ 'XLMC',  'Michelau' ],
	[ 'XAMI',  'Micheldorf' ],
	[ 'XDMF',  'Middelfart' ],
	[ 'XPMIE', 'Miekinia' ],
	[ 'XTMD',  'Mikulasovice d n' ],
	[ 'XIMB',  'Milano C' ],
	[ 'XIMG',  'Milano Greco P' ],
	[ 'XIMP',  'Milano PG' ],
	[ 'XBMM',  'Milmort' ],
	[ 'XTMM',  'Mimon' ],
	[ 'XAMN',  'Mining' ],
	[ 'XFMI',  'Miramas' ],
	[ 'XAMH',  'Mitterberghütten' ],
	[ 'XAMV',  'Mitterdf-Veitsch' ],
	[ 'XAMIX', 'Mixnitz-Bärensch' ],
	[ 'XTMB',  'Ml Boleslav hl n' ],
	[ 'XTMY',  'Mlyny (CZ)' ],
	[ 'XFMOD', 'Modane' ],
	[ 'XSMI',  'Möhlin' ],
	[ 'XBML',  'Mol' ],
	[ 'XSMOS', 'Mols' ],
	[ 'XFMH',  'Mommenheim' ],
	[ 'XFMC',  'Monaco-M.Carlo' ],
	[ 'XIMW',  'Monguelfo/Welsbg' ],
	[ 'XBMS',  'Mons' ],
	[ 'XFMD',  'Montbeliard' ],
	[ 'XFMN',  'Montelimar' ],
	[ 'XSMY',  'Monthey' ],
	[ 'XFMLV', 'Montlucon Ville' ],
	[ 'XFMOM', 'Montmelian' ],
	[ 'XFMND', 'Mont-Notre-Dame' ],
	[ 'XFMO',  'Montpellier' ],
	[ 'XSMO',  'Montreux' ],
	[ 'XIMOZ', 'Monza' ],
	[ 'XAMB',  'Moosbierbaum/H' ],
	[ 'XFMOR', 'Moret-Veneux' ],
	[ 'XSMG',  'Morges' ],
	[ 'XFMOH', 'Morhange' ],
	[ 'XCM',   'Moskva' ],
	[ 'XMMO',  'Mosonmagyarovar' ],
	[ 'XTMO',  'Most' ],
	[ 'XAMTZ', 'Mötz' ],
	[ 'XFMOU', 'Mouchard' ],
	[ 'XFMSA', 'Moulins-s-Allier' ],
	[ 'XBMN',  'Mouscron' ],
	[ 'XBMT',  'Moustier' ],
	[ 'XSMU',  'Moutier' ],
	[ 'XFMT',  'Moutiers-Salins' ],
	[ 'XFMG',  'Moyeuvre Grande' ],
	[ 'XSMH',  'Mühlehorn' ],
	[ 'XAMHM', 'Mühlheim' ],
	[ 'XAMM',  'Mühl-Möllbrücke' ],
	[ 'XBMU',  'Muizen' ],
	[ 'XFMVD', 'Mulhouse Dornach' ],
	[ 'XFMV',  'Mulhouse Ville' ],
	[ 'XSMBU', 'Münchenbuchsee' ],
	[ 'XFMUM', 'Mundolsheim' ],
	[ 'XBMZ',  'Munkzwalm' ],
	[ 'XLMB',  'Munsbach' ],
	[ 'XSMUS', 'Münsingen CH' ],
	[ 'XFMST', 'Munster FR' ],
	[ 'XAMW',  'Münster-Wiesing' ],
	[ 'XSMN',  'Münstlg-Scherz' ],
	[ 'XSMUO', 'Muolen' ],
	[ 'XSMR',  'Murg' ],
	[ 'XAMZ',  'Mürzzuschlag' ],
	[ 'XAMU',  'Musau' ],
	[ 'XFMUS', 'Mussidan' ],
	[ 'XSMT',  'Muttenz' ],
	[ 'XDNV',  'Naestved' ],
	[ 'XMNV',  'Nagymaros-Visegr' ],
	[ 'XBNA',  'Namur' ],
	[ 'XFNT',  'Nancois Tronvill' ],
	[ 'XFN',   'Nancy' ],
	[ 'XFNA',  'Nantes' ],
	[ 'XINAC', 'Napoli C' ],
	[ 'XFNB',  'Narbonne' ],
	[ 'XSN',   'Nebikon' ],
	[ 'XBN',   'Neerpelt' ],
	[ 'XTNJ',  'Nejdek' ],
	[ 'XTNJZ', 'Nejdek zast' ],
	[ 'XFNP',  'Nemours-St-Pierr' ],
	[ 'XAND',  'Nendeln' ],
	[ 'XANE',  'Nenzing' ],
	[ 'XBNE',  'Nessonvaux' ],
	[ 'XTNT',  'Nestedice' ],
	[ 'XTNM',  'Nestemice' ],
	[ 'XANF',  'Nettingsdorf' ],
	[ 'XSNC',  'Neuchatel' ],
	[ 'XBNC',  'Neufchateau' ],
	[ 'XFNE',  'Neufchateau' ],
	[ 'XSNE',  'Neuhausen CH' ],
	[ 'XSNK',  'Neukirch-Egnach' ],
	[ 'XANK',  'Neumarkt-Kallham' ],
	[ 'XANKD', 'Neumarkt-Köstend' ],
	[ 'XANG',  'Neuratting' ],
	[ 'XFNC',  'Nice Ville' ],
	[ 'XSNB',  'Niederbipp' ],
	[ 'XSNI',  'Niederglatt' ],
	[ 'XLNK',  'Niederkorn' ],
	[ 'XNNA',  'Nieuw Amsterdam' ],
	[ 'XNNS',  'Nieuwe Schans' ],
	[ 'XNNM',  'Nijmegen' ],
	[ 'XFNI',  'Nimes' ],
	[ 'XDNA',  'Noerre Alslev' ],
	[ 'XDNN',  'Noerre-Nebel' ],
	[ 'XDNP',  'Noerreport' ],
	[ 'XLNZ',  'Noertzange' ],
	[ 'XFNR',  'Nogent le Retrou' ],
	[ 'XFNO',  'Noisy-le-Sec' ],
	[ 'XFNDB', 'Notre Dame B' ],
	[ 'XRNG',  'Nova Gradiska' ],
	[ 'XRNK',  'Nova Kapela' ],
	[ 'XTNR',  'Nova Role' ],
	[ 'XTNRZ', 'Nova Role zast' ],
	[ 'XINV',  'Novara' ],
	[ 'XTNH',  'Nove Hamry' ],
	[ 'XTNHZ', 'Nove Hamry zast' ],
	[ 'XTNS',  'Nove Sedlo u Lok' ],
	[ 'XYNZ',  'Nove Zamky' ],
	[ 'XFNV',  'Noveant' ],
	[ 'XJNB',  'Novi Beograd' ],
	[ 'XJNS',  'Novi Sad' ],
	[ 'XRNV',  'Novska' ],
	[ 'XTNB',  'Novy Bor' ],
	[ 'XPND',  'Nowe Drezdenko' ],
	[ 'XFNY',  'Noyon' ],
	[ 'XDNE',  'Ny Ellebjerg' ],
	[ 'XDNY',  'Nyborg' ],
	[ 'XMNH',  'Nyiregyhaza' ],
	[ 'XDNK',  'Nykoebing F' ],
	[ 'XTN',   'Nymburk hl.n.' ],
	[ 'XAOA',  'Oberalm' ],
	[ 'XSOG',  'Oberburg' ],
	[ 'XSOB',  'Oberglatt' ],
	[ 'XAOB',  'Oberhofen' ],
	[ 'XLOK',  'Oberkorn' ],
	[ 'XFOM',  'Obermodern' ],
	[ 'XAOM',  'Obernberg-Alth' ],
	[ 'XSOR',  'Oberrieden' ],
	[ 'XSOT',  'Oberriet' ],
	[ 'XBOB',  'Obourg' ],
	[ 'XDOD',  'Odense' ],
	[ 'XDOG',  'Oelgod' ],
	[ 'XDKHO', 'Oerestad' ],
	[ 'XFOE',  'Oermingen' ],
	[ 'XDOP',  'Oesterport' ],
	[ 'XLOE',  'Oetrange' ],
	[ 'XDOB',  'Oksboel' ],
	[ 'XROK',  'Okucani' ],
	[ 'XNOD',  'Oldenzaal' ],
	[ 'XTOL',  'Oldrichov u N' ],
	[ 'XBOL',  'Olen' ],
	[ 'XIOM',  'Olivetta S Mich' ],
	[ 'XTO',   'Olomouc hl.n.' ],
	[ 'XTOV',  'Olovi' ],
	[ 'XSOL',  'Olten' ],
	[ 'XNOM',  'Ommen' ],
	[ 'XFO',   'Onville' ],
	[ 'XBO',   'Oostende' ],
	[ 'XPO',   'Opole Glowne' ],
	[ 'XSOP',  'Oppikon' ],
	[ 'XBOW',  'Opwijk' ],
	[ 'XFOR',  'Orange' ],
	[ 'XFOC',  'Orchies' ],
	[ 'XDOH',  'Orehoved' ],
	[ 'XFOL',  'Orleans' ],
	[ 'XIOV',  'Orvieto' ],
	[ 'XRO',   'Osijek' ],
	[ 'XNO',   'Oss' ],
	[ 'XSOM',  'Ostermundingen' ],
	[ 'XTOH',  'Ostrava hl.n.' ],
	[ 'XTOI',  'Ostrava-Svinov' ],
	[ 'XPOM',  'Oswiecim' ],
	[ 'XSOS',  'Othmarsingen' ],
	[ 'XAOE',  'Ötztal' ],
	[ 'XDOU',  'Outrup' ],
	[ 'XNOV',  'Overveen' ],
	[ 'XIPW',  'P.Gardena/Waidbr' ],
	[ 'XDPA',  'Padborg' ],
	[ 'XIPP',  'Padova' ],
	[ 'XFPS',  'Pagny-s-Moselle' ],
	[ 'XFPAN', 'Pantin' ],
	[ 'XTPAR', 'Pardubice' ],
	[ 'XFPMP', 'Paris Montparnas' ],
	[ 'XFPN',  'Paris Nord' ],
	[ 'XFPA',  'Paris-Austerlitz' ],
	[ 'XFPO',  'Paris-Est' ],
	[ 'XFPG',  'Paris-G-d-Lyon' ],
	[ 'XFPSL', 'Paris-St-Lazare' ],
	[ 'XAPD',  'Parndorf' ],
	[ 'XAPT',  'Paternion-Feistr' ],
	[ 'XAPA',  'Patsch' ],
	[ 'XFPU',  'Pau' ],
	[ 'XIPVI', 'Pavia' ],
	[ 'XAPG',  'Peggau-Deutschf' ],
	[ 'XFPTE', 'Peltre' ],
	[ 'XBP',   'Pepinster' ],
	[ 'XFPGX', 'Perigueux' ],
	[ 'XTPEK', 'Pernink' ],
	[ 'XFPE',  'Perpignan' ],
	[ 'XIPDG', 'Peschiera d Gard' ],
	[ 'XLP',   'Petange' ],
	[ 'XAPK',  'Peterskirchen' ],
	[ 'XFPTC', 'Petit Croix' ],
	[ 'XSPF',  'Pfäffikon SZ' ],
	[ 'XAPFW', 'Pfarrwerfen' ],
	[ 'XAPL',  'Pflach' ],
	[ 'XAPC',  'Pichl' ],
	[ 'XPPN',  'Piensk' ],
	[ 'XAPV',  'Pill-Vomperbach' ],
	[ 'XSXPT', 'Pino Grenze IT' ],
	[ 'XIPI',  'Pisa C' ],
	[ 'XTPLA', 'Plesna' ],
	[ 'XUPV',  'Ploiesti Vest' ],
	[ 'XTPZ',  'Plzen hl.n.' ],
	[ 'XAPN',  'Pöchlarn' ],
	[ 'XFPT',  'Poitiers' ],
	[ 'XTPI',  'Pomezi n O' ],
	[ 'XFPM',  'Pont-a-Mousson' ],
	[ 'XSPT',  'Pontresina' ],
	[ 'XFPXE', 'Pont-Ste-Maxence' ],
	[ 'XFPV',  'Pont-St-Vincent' ],
	[ 'XIPD',  'Pordenone' ],
	[ 'XFPR',  'Port Vendres' ],
	[ 'XEPB',  'Port-Bou' ],
	[ 'XAPW',  'Pörtschach a.W.' ],
	[ 'XTPO',  'Potucky' ],
	[ 'XTPOZ', 'Potucky zastavka' ],
	[ 'XFPSU', 'Pougues-les-Eaux' ],
	[ 'XPPG',  'Poznan Glowny' ],
	[ 'XTP',   'Praha hl.n.' ],
	[ 'XTPH',  'Praha-Holesovice' ],
	[ 'XTPM',  'Praha-Smichov' ],
	[ 'XTPV',  'Praha-Vrsovice' ],
	[ 'XAPM',  'Pram-Haag' ],
	[ 'XSPR',  'Pratteln' ],
	[ 'XUP',   'Predeal' ],
	[ 'XAPR',  'Pregarten' ],
	[ 'XTPR',  'Prerov' ],
	[ 'XAPRD', 'Prinzersdorf' ],
	[ 'XBPO',  'Profondsart' ],
	[ 'XPPRY', 'Przylep' ],
	[ 'XZP',   'Ptuj' ],
	[ 'XAPO',  'Puch-Oberalm' ],
	[ 'XAPZ',  'Pusarnitz' ],
	[ 'XBQ',   'Quevy' ],
	[ 'XFQ',   'Quimper' ],
	[ 'XARA',  'Radstadt' ],
	[ 'XSRF',  'Rafz' ],
	[ 'XARW',  'Rankweil' ],
	[ 'XSRZ',  'Ranzo-S-Abbondio' ],
	[ 'XARK',  'Rattenbg-Krams' ],
	[ 'XIREC', 'Re' ],
	[ 'XSRM',  'Rebstein-Marbach' ],
	[ 'XFRD',  'Reding' ],
	[ 'XERG',  'Redondela de Gal' ],
	[ 'XSRC',  'Reichenbach Kt' ],
	[ 'XSRE',  'Reichenburg' ],
	[ 'XFRS',  'Reims' ],
	[ 'XARE',  'Reith' ],
	[ 'XDRE',  'Rejsby' ],
	[ 'XARWL', 'Rekawinkel' ],
	[ 'XSRN',  'Rekingen AG' ],
	[ 'XFRF',  'Remelfing' ],
	[ 'XFRM',  'Remilly' ],
	[ 'XFRT',  'Remiremont' ],
	[ 'XFR',   'Rennes' ],
	[ 'XTRE',  'Retenice' ],
	[ 'XAREZ', 'Retz' ],
	[ 'XART',  'Reutte in Tirol' ],
	[ 'XARTS', 'Reutte Schulz.' ],
	[ 'XSRH',  'Rheineck' ],
	[ 'XSRD',  'Rheinfelden' ],
	[ 'XDRB',  'Ribe' ],
	[ 'XDRBN', 'Ribe Noerremark' ],
	[ 'XSRW',  'Richterswil' ],
	[ 'XARI',  'Ried i Innkreis' ],
	[ 'XARD',  'Riedenburg' ],
	[ 'XARZ',  'Rietz' ],
	[ 'XIRI',  'Rimini' ],
	[ 'XDRG',  'Ringkoebing' ],
	[ 'XDRI',  'Ringsted' ],
	[ 'XIRP',  'Rio di Pusteria' ],
	[ 'XSRB',  'Rivera-Bironico' ],
	[ 'XFRV',  'Rivesaltes' ],
	[ 'XFROA', 'Roanne' ],
	[ 'XIRL',  'Robilante' ],
	[ 'XIROV', 'Roccavione' ],
	[ 'XLRD',  'Rodange' ],
	[ 'XDR',   'Roedby F' ],
	[ 'XDRO',  'Roedby Faerge' ],
	[ 'XDRD',  'Roedekro' ],
	[ 'XDRN',  'Roenne' ],
	[ 'XFRSW', 'Roeschwoog' ],
	[ 'XSRO',  'Roggwil-Berg' ],
	[ 'XSRY',  'Roggwil-Wynau' ],
	[ 'XARH',  'Rohr-Bad Hall' ],
	[ 'XTRO',  'Rokycany' ],
	[ 'XIRM',  'Roma Termini' ],
	[ 'XSR',   'Romanshorn' ],
	[ 'XSRHF', 'Romanshorn (See)' ],
	[ 'XFRO',  'Rombas' ],
	[ 'XBRO',  'Ronet' ],
	[ 'XLRT',  'Roodt' ],
	[ 'XNRD',  'Roosendaal' ],
	[ 'XARP',  'Roppen' ],
	[ 'XSRS',  'Rorschach' ],
	[ 'XSRSH', 'Rorschach Hafen' ],
	[ 'XSRSS', 'Rorschach Hf S' ],
	[ 'XARB',  'Rosenbach' ],
	[ 'XAXRB', 'Rosenbach Gr SI' ],
	[ 'XDRK',  'Roskilde' ],
	[ 'XTRV',  'Rotava' ],
	[ 'XSRTB', 'Rothenburg' ],
	[ 'XSRTT', 'Rothenthurm' ],
	[ 'XSRR',  'Rothrist' ],
	[ 'XSRK',  'Rotkreuz' ],
	[ 'XNRL',  'Rotterd Lombardi' ],
	[ 'XNRN',  'Rotterd Noord G' ],
	[ 'XNRC',  'Rotterdam C' ],
	[ 'XNRS',  'RotterdamStadion' ],
	[ 'XTRN',  'Roudnice n Labem' ],
	[ 'XFROU', 'Rouffach' ],
	[ 'XIROT', 'Rovato' ],
	[ 'XIRR',  'Rovereto' ],
	[ 'XIRG',  'Rovigo' ],
	[ 'XARU',  'Rum' ],
	[ 'XJRU',  'Ruma' ],
	[ 'XTRU',  'Rumburk' ],
	[ 'XSRU',  'Rümlang' ],
	[ 'XSRP',  'Rupperswil' ],
	[ 'XSRL',  'Rüschlikon' ],
	[ 'XTRS',  'Rusova' ],
	[ 'XSRT',  'Rüthi SG' ],
	[ 'XTRY',  'Rybniste' ],
	[ 'XPR',   'Rzepin' ],
	[ 'XFSAC', 'S Avre l Chambre' ],
	[ 'XFSMV', 'S Michel Valloir' ],
	[ 'XASPS', 'S Peter-Seitenst' ],
	[ 'XISCI', 'S.Cand./Innichen' ],
	[ 'XASA',  'Saalfelden' ],
	[ 'XSSA',  'Safenwil' ],
	[ 'XFSZE', 'Saincaize' ],
	[ 'XFSAS', 'Saintes' ],
	[ 'XBSG',  'Saint-Ghislain' ],
	[ 'XSSZ',  'Salez-Sennwald' ],
	[ 'XAAI',  'Salzb Aiglhof' ],
	[ 'XASLF', 'Salzb Liefering' ],
	[ 'XASBA', 'Salzburg Aigen' ],
	[ 'XASB',  'Salzburg Hbf' ],
	[ 'XASBP', 'Salzburg Parsch' ],
	[ 'XASSA', 'Salzburg Sam' ],
	[ 'XASBS', 'Salzburg Süd' ],
	[ 'XASG',  'Salzburg-Gnigl' ],
	[ 'XSSD',  'Samedan' ],
	[ 'XSSMT', 'Samstagern' ],
	[ 'XISAR', 'San Remo' ],
	[ 'XLSC',  'Sandweiler-Cont' ],
	[ 'XFSN',  'Sanry sur Nied' ],
	[ 'XRXSJ', 'Sapjane Gr SI' ],
	[ 'XNSO',  'Sappemeer Oost' ],
	[ 'XSSR',  'Sargans' ],
	[ 'XFSB',  'Sarrebourg' ],
	[ 'XFSM',  'Sarreguemines' ],
	[ 'XFSI',  'Sarreinsming' ],
	[ 'XFSRI', 'Sathonay-Rilieux' ],
	[ 'XFSUD', 'Saumur' ],
	[ 'XFSV',  'Saverne' ],
	[ 'XISAV', 'Savona Letimbro' ],
	[ 'XAML',  'Sb Mülln-Altstdt' ],
	[ 'XASVZ', 'Schaan-Vaduz' ],
	[ 'XBS',   'Schaerbeek' ],
	[ 'XSS',   'Schaffhsn CH' ],
	[ 'XASFU', 'Schaftenau' ],
	[ 'XASH',  'Schärding' ],
	[ 'XAA',   'Scharnitz' ],
	[ 'XNSE',  'Scheemda' ],
	[ 'XNSC',  'Schiedam-Rott.W.' ],
	[ 'XLSL',  'Schifflange' ],
	[ 'XSSF',  'Schindellegi-F.' ],
	[ 'XNSP',  'Schiphol' ],
	[ 'XASL',  'Schladming' ],
	[ 'XASBL', 'Schönbichl' ],
	[ 'XASCH', 'Schönwies' ],
	[ 'XLSW',  'Schouweiler' ],
	[ 'XASN',  'Schruns' ],
	[ 'XSSBU', 'Schübelb-Buttiko' ],
	[ 'XBSU',  'Schulen' ],
	[ 'XASW',  'Schwarzach-St.V.' ],
	[ 'XASC',  'Schwaz' ],
	[ 'XFSWM', 'Schwindratzheim' ],
	[ 'XSSY',  'Schwyz' ],
	[ 'XSST',  'Scuol-Tarasp' ],
	[ 'XTSBZ', 'Sebuzin' ],
	[ 'XAS',   'Seefeld in Tirol' ],
	[ 'XASK',  'Seekirchen a W' ],
	[ 'XDSE',  'Sejstrup' ],
	[ 'XFSL',  'Selestat' ],
	[ 'XASZ',  'Selzthal' ],
	[ 'XISER', 'Seregno' ],
	[ 'XISGI', 'Sesto S Giovanni' ],
	[ 'XFSE',  'Sete' ],
	[ 'XSSV',  'Sevelen' ],
	[ 'XZSN',  'Sevnica' ],
	[ 'XNSH',  's-Hertogenbosch' ],
	[ 'XJSI',  'Sid' ],
	[ 'XSSW',  'Siebnen-Wangen' ],
	[ 'XSSH',  'Siegershausen' ],
	[ 'XPSIZ', 'Sieniawa Zarska' ],
	[ 'XFSIK', 'Sierck-les-Bains' ],
	[ 'XSSI',  'Sierre/Siders' ],
	[ 'XDSIG', 'Sig' ],
	[ 'XUSG',  'Sighisoara' ],
	[ 'XASIL', 'Sillian' ],
	[ 'XASLZ', 'Silz' ],
	[ 'XUSI',  'Simeria' ],
	[ 'XUSN',  'Sinaia' ],
	[ 'XSSN',  'Sion' ],
	[ 'XSSC',  'Sissach' ],
	[ 'XNSI',  'Sittard' ],
	[ 'XDSB',  'Skaerbaek' ],
	[ 'XDSJ',  'Skjern' ],
	[ 'XZSL',  'Skofja Loka' ],
	[ 'XDSGE', 'Slagelse' ],
	[ 'XRSL',  'Slavonski Brod' ],
	[ 'XPSLB', 'Slubice' ],
	[ 'XTSLN', 'Sluknov' ],
	[ 'XTSLZ', 'Sluknov Zastavka' ],
	[ 'XCSM',  'Smolensk' ],
	[ 'XNSK',  'Sneek' ],
	[ 'XTSO',  'Sokolov' ],
	[ 'XSSO',  'Solothurn' ],
	[ 'XFSO',  'Somain' ],
	[ 'XPSO',  'Sopot' ],
	[ 'XMS',   'Sopron' ],
	[ 'XDSOR', 'Soroe' ],
	[ 'XPSI',  'Sosnowiec Glowny' ],
	[ 'XBSP',  'Spa' ],
	[ 'XDSP',  'Spangsbjerg' ],
	[ 'XTSP',  'Spicak' ],
	[ 'XASS',  'Spielfeld-Straß' ],
	[ 'XSSP',  'Spiez' ],
	[ 'XASPP', 'Spital am Pyhrn' ],
	[ 'XASP',  'Spittal(Millst.)' ],
	[ 'XRS',   'Split' ],
	[ 'XJSM',  'Sr Mitrovica' ],
	[ 'XFSDD', 'St Dalmas Tende' ],
	[ 'XSSG',  'St Gallen' ],
	[ 'XSSGW', 'St Gallen Winkel' ],
	[ 'XSSGH', 'St Gallen-Haggen' ],
	[ 'XFJ',   'St Jory' ],
	[ 'XFSMA', 'St Malo' ],
	[ 'XAJO',  'St. Jodok' ],
	[ 'XSMZ',  'St. Moritz' ],
	[ 'XAAB',  'St.Anton a.Arlbg' ],
	[ 'XAAM',  'St.Anton i.M.' ],
	[ 'XASJ',  'St.Johann i.P.' ],
	[ 'XAJT',  'St.Johann i.T.' ],
	[ 'XSSM',  'St.Margrethen' ],
	[ 'XASE',  'St.Michael' ],
	[ 'XAP',   'St.Pölten Hbf' ],
	[ 'XASV',  'St.Valentin' ],
	[ 'XAVG',  'St.Veit a.d.Glan' ],
	[ 'XSSTA', 'Staad' ],
	[ 'XASU',  'Stadlau' ],
	[ 'XASRM', 'Stadt Rottenmann' ],
	[ 'XFSTA', 'Staffelfelden' ],
	[ 'XASI',  'Stainach-Irdning' ],
	[ 'XFSAR', 'St-Amour' ],
	[ 'XASTM', 'Stams' ],
	[ 'XASTS', 'Stans(b Schwaz)' ],
	[ 'XJSP',  'Stara Pazova' ],
	[ 'XTSR',  'Stara Role' ],
	[ 'XBSA',  'Statte' ],
	[ 'XFSA',  'St-Avold' ],
	[ 'XBSD',  'St-Denijs-B' ],
	[ 'XSSB',  'Steckborn' ],
	[ 'XSSBS', 'Steckborn S' ],
	[ 'XNSW',  'Steenwijk' ],
	[ 'XAST',  'Steinach i Tirol' ],
	[ 'XFSNG', 'Steinbourg' ],
	[ 'XASF',  'Steindorf-Straßw' ],
	[ 'XSSU',  'Steinebrunn' ],
	[ 'XSSK',  'Stein-Säckingen' ],
	[ 'XFSF',  'Stephansfeld' ],
	[ 'XSSE',  'Stettbach' ],
	[ 'XASY',  'Steyr' ],
	[ 'XFSGF', 'St-Germ-d-Fosses' ],
	[ 'XFSG',  'St-Gervais' ],
	[ 'XFSHT', 'St-Hilaire-au-T' ],
	[ 'XFJL',  'St-Jean-de-Luz' ],
	[ 'XFJMA', 'St-Jean-d-Maur' ],
	[ 'XFSTL', 'St-Louis' ],
	[ 'XSMC',  'St-Maurice' ],
	[ 'XASRA', 'Stockerau' ],
	[ 'XFPY',  'St-Pierre-d\'Alb' ],
	[ 'XFPDC', 'St-Pierre-d-C' ],
	[ 'XFPL',  'St-Pol-de-Leon' ],
	[ 'XFSP',  'St-Priest' ],
	[ 'XFSQ',  'St-Quentin' ],
	[ 'XFSR',  'St-Raphael-Vales' ],
	[ 'XFSTG', 'Strasbourg' ],
	[ 'XASTH', 'Straßhof' ],
	[ 'XASR',  'Straßwalchen' ],
	[ 'XISTA', 'Stresa' ],
	[ 'XRSV',  'Striziv Vrpolje' ],
	[ 'XDS',   'Struer' ],
	[ 'XPSKW', 'Strzelce Kraj W' ],
	[ 'XFSSL', 'St-Sulpice-Laur' ],
	[ 'XYST',  'Sturovo' ],
	[ 'XJST',  'Subotica' ],
	[ 'XTS',   'Sucha' ],
	[ 'XASUM', 'Summerau' ],
	[ 'XSSUS', 'Sursee' ],
	[ 'XFSVF', 'Survilliers Foss' ],
	[ 'XASBN', 'Süssenbrunn' ],
	[ 'XNSU',  'Susteren' ],
	[ 'XTSV',  'Svatava' ],
	[ 'XTSVZ', 'Svatava zast' ],
	[ 'XTSVO', 'Svor' ],
	[ 'XPSWB', 'Swiebodzin' ],
	[ 'XPSM',  'Swinoujscie C' ],
	[ 'XPSG',  'Szczecin Glowny' ],
	[ 'XPSU',  'Szczecin Gu.' ],
	[ 'XMSG',  'Szentgotthard' ],
	[ 'XMSB',  'Szob' ],
	[ 'XMXSB', 'Szob Gr SK' ],
	[ 'XMSN',  'Szolnok' ],
	[ 'XDKT',  'Taarnby' ],
	[ 'XTT',   'Tabor' ],
	[ 'XITG',  'Taggia' ],
	[ 'XBTA',  'Tamines' ],
	[ 'XFTA',  'Tarascon' ],
	[ 'XDTAM', 'Tarm' ],
	[ 'XITB',  'Tarvisio Boscov' ],
	[ 'XIXTB', 'Tarvisio Gr AT' ],
	[ 'XMTA',  'Tata' ],
	[ 'XMTB',  'Tatabanya' ],
	[ 'XATF',  'Taufkirchen Pram' ],
	[ 'XDTA',  'Taulov' ],
	[ 'XST',   'Taverne-Torric' ],
	[ 'XATR',  'Taxenbach-Rauris' ],
	[ 'XATX',  'Taxham' ],
	[ 'XPTC',  'Tczew' ],
	[ 'XSTE',  'Tecknau' ],
	[ 'XATP',  'Telfs-Pfaffenhfn' ],
	[ 'XFTE',  'Templeuve' ],
	[ 'XFTND', 'Tende' ],
	[ 'XATN',  'Tenneck' ],
	[ 'XTTV',  'Teplic v Cechach' ],
	[ 'XPTE',  'Terespol' ],
	[ 'XPXTE', 'Terespol Gr BY' ],
	[ 'XATW',  'Terfens-Weer' ],
	[ 'XFTG',  'Tergnier' ],
	[ 'XBTE',  'Testelt' ],
	[ 'XFTT',  'Teting' ],
	[ 'XFHP',  'TGV Hte Picardie' ],
	[ 'XSTW',  'Thalwil' ],
	[ 'XFTHV', 'Thionville' ],
	[ 'XSTH',  'Thun' ],
	[ 'XSTU',  'Thusis' ],
	[ 'XFTB',  'Tieffenbach' ],
	[ 'XBTI',  'Tienen' ],
	[ 'XNT',   'Tilburg' ],
	[ 'XDTM',  'Tim' ],
	[ 'XDTI',  'Tinglev' ],
	[ 'XATI',  'Tisis' ],
	[ 'XTTI',  'Tisova u N' ],
	[ 'XDTR',  'Tistrup' ],
	[ 'XDTB',  'Tjaereborg' ],
	[ 'XSTOA', 'Tobel-Affeltrang' ],
	[ 'XDTO',  'Toender' ],
	[ 'XDTON', 'Toender Nord' ],
	[ 'XDTP',  'Tommerup' ],
	[ 'XBTG',  'Tongeren' ],
	[ 'XITUP', 'Torino P Susa' ],
	[ 'XFTU',  'Toul' ],
	[ 'XFTO',  'Toulon' ],
	[ 'XFTM',  'Toulouse-Matabi' ],
	[ 'XFTC',  'Tourcoing' ],
	[ 'XBTO',  'Tournai' ],
	[ 'XFTOU', 'Tournan' ],
	[ 'XRT',   'Tovarnik' ],
	[ 'XATRN', 'Traun' ],
	[ 'XZT',   'Trbovlje' ],
	[ 'XTTRB', 'Trebusice' ],
	[ 'XATA',  'Treibach-Althf' ],
	[ 'XDTRK', 'Trekroner' ],
	[ 'XVTF',  'Trelleborg F' ],
	[ 'XITR',  'Trento' ],
	[ 'XITS',  'Treviso Centrale' ],
	[ 'XAT',   'Trieben' ],
	[ 'XATK',  'Trimmelkam-Ost.' ],
	[ 'XLTV',  'Troisvierges' ],
	[ 'XBT',   'Trooz' ],
	[ 'XATS',  'Tschagguns' ],
	[ 'XATU',  'Tulln' ],
	[ 'XATD',  'Tullnerfeld' ],
	[ 'XPTU',  'Tuplice' ],
	[ 'XPTUD', 'Tuplice Debinka' ],
	[ 'XSTG',  'Turgi' ],
	[ 'XPTY',  'Tychy' ],
	[ 'XFUC',  'Uckange' ],
	[ 'XIU',   'Udine' ],
	[ 'XNUI',  'Uitgeest' ],
	[ 'XDUF',  'Ulfborg' ],
	[ 'XAUH',  'Ulmerfeld-Hausme' ],
	[ 'XAU',   'Ulrichsbr-Füssen' ],
	[ 'XAUS',  'Unterberg-Stefbr' ],
	[ 'XSUT',  'Unterterzen' ],
	[ 'XAUM',  'Unzmarkt' ],
	[ 'XTUS',  'Usti n L-Strekov' ],
	[ 'XTU',   'Usti n.L. hl.n.' ],
	[ 'XTUZ',  'Usti n.L. zapad' ],
	[ 'XNU',   'Utrecht C' ],
	[ 'XSUW',  'Uttwil' ],
	[ 'XMV',   'Vac' ],
	[ 'XFVAR', 'Vaires' ],
	[ 'XDVAL', 'Valby' ],
	[ 'XIVO',  'Valdaora-Anters' ],
	[ 'XTVA',  'Valdek' ],
	[ 'XFVDR', 'Val-de-Reuil' ],
	[ 'XFVC',  'Valence' ],
	[ 'XFVCV', 'Valence-TGV' ],
	[ 'XFVA',  'Valenciennes' ],
	[ 'XNVB',  'Valkenburg (NL)' ],
	[ 'XDVA',  'Vamdrup' ],
	[ 'XAVA',  'Vandans' ],
	[ 'XFVN',  'Varangeville' ],
	[ 'XDVR',  'Varde' ],
	[ 'XDVRK', 'Varde Kaserne' ],
	[ 'XDVRN', 'Varde Nord' ],
	[ 'XDVRW', 'Varde Vest' ],
	[ 'XTSVD', 'Varns stare' ],
	[ 'XTVD',  'Varnsdorf' ],
	[ 'XTVK',  'Varnsdorf piv Ko' ],
	[ 'XNVK',  'Veenendaal K' ],
	[ 'XTVY',  'Vejprty' ],
	[ 'XAVE',  'Velden a.W.' ],
	[ 'XTVZ',  'Velke Zernoseky' ],
	[ 'XTVV',  'Velky Senov' ],
	[ 'XTVVZ', 'Velky Senov zast' ],
	[ 'XDVE',  'Vemb' ],
	[ 'XFVH',  'Vendenheim' ],
	[ 'XIVNM', 'Venezia Mestre' ],
	[ 'XIVNS', 'Venezia S.Lucia' ],
	[ 'XFVE',  'Venissieux' ],
	[ 'XNVL',  'Venlo' ],
	[ 'XIVT',  'Ventimiglia' ],
	[ 'XIVE',  'Vercelli' ],
	[ 'XIVNN', 'Vernante' ],
	[ 'XFVEA', 'Verneuil-sur-Avr' ],
	[ 'XFVEE', 'Vernieul l\'Etang' ],
	[ 'XFVNO', 'Vernon' ],
	[ 'XIVP',  'Verona PN' ],
	[ 'XBVC',  'Verviers-Central' ],
	[ 'XBVP',  'Verviers-Palais' ],
	[ 'XTVER', 'Vesele Rabstejn' ],
	[ 'XCV',   'Viazma' ],
	[ 'XDVSJ', 'Viby Sjaelland' ],
	[ 'XIVC',  'Vicenza' ],
	[ 'XFVI',  'Vienne' ],
	[ 'XFVVA', 'Vievola' ],
	[ 'XTVL',  'Vilemov Sluknova' ],
	[ 'XIVL',  'Villa Literno' ],
	[ 'XIVB',  'Villab./Niederdf' ],
	[ 'XAVH',  'Villach Hbf' ],
	[ 'XAVW',  'Villach Westbf' ],
	[ 'XFVD',  'Villars l Dombes' ],
	[ 'XFVDS', 'Villedieu-les-Po' ],
	[ 'XFVR',  'Villers-Cotter' ],
	[ 'XFVG',  'Villiers Bel Gon' ],
	[ 'XAV',   'Vils' ],
	[ 'XBVV',  'Vilvoorde' ],
	[ 'XRVI',  'Vinkovci' ],
	[ 'XIVS',  'Vipit./Sterzing' ],
	[ 'XDVY',  'Visby' ],
	[ 'XBV',   'Vise' ],
	[ 'XSVI',  'Visp' ],
	[ 'XFVV',  'Vitry-La-Ville' ],
	[ 'XFVF',  'Vitry-L-Francois' ],
	[ 'XFVT',  'Vittel' ],
	[ 'XNVD',  'Vlaardingen Cent' ],
	[ 'XNVT',  'Vleuten' ],
	[ 'XNVI',  'Vlissingen' ],
	[ 'XAVB',  'Vöcklabruck' ],
	[ 'XAVM',  'Vöcklamarkt' ],
	[ 'XIVOG', 'Voghera' ],
	[ 'XDV',   'Vojens' ],
	[ 'XTV',   'Vojtanov' ],
	[ 'XAVO',  'Volders-Baumk' ],
	[ 'XAVL',  'Völs' ],
	[ 'XDVB',  'Vordingborg' ],
	[ 'XDVM',  'Vroegum' ],
	[ 'XTVP',  'Vysoka Pec' ],
	[ 'XSWW',  'Wädenswil' ],
	[ 'XAWA',  'Wald Schoberpaß' ],
	[ 'XSWA',  'Walenstadt' ],
	[ 'XLWF',  'Walferdange' ],
	[ 'XSWS',  'Wallisellen' ],
	[ 'XAWV',  'Warmbad Villach' ],
	[ 'XPWC',  'Warszawa C' ],
	[ 'XPWW',  'Warszawa Wschod.' ],
	[ 'XPWZ',  'Warszawa Zachod' ],
	[ 'XAWBK', 'Wartberg a d Kr' ],
	[ 'XAWB',  'Wartberg i.Mürz.' ],
	[ 'XLW',   'Wasserbillig' ],
	[ 'XBWA',  'Waterloo' ],
	[ 'XBWM',  'Watermael' ],
	[ 'XLWK',  'Wecker' ],
	[ 'XNWT',  'Weert' ],
	[ 'XNWS',  'Weesp' ],
	[ 'XPWE',  'Wegliniec' ],
	[ 'XSW',   'Weinfelden' ],
	[ 'XBW',   'Welkenraedt' ],
	[ 'XAWE',  'Wels Hbf' ],
	[ 'XAWEN', 'Wendling' ],
	[ 'XAWRF', 'Werfen' ],
	[ 'XAWR',  'Wernstein' ],
	[ 'XBWT',  'Wespelaar-Tildon' ],
	[ 'XAWD',  'Westendorf AT' ],
	[ 'XSWE',  'Wettingen' ],
	[ 'XAWF',  'Wien F-J Bf' ],
	[ 'XAWIF', 'Wien Flughafen' ],
	[ 'XAWIE', 'Wien Hbf' ],
	[ 'XAWIO', 'Wien Hbf ARZ' ],
	[ 'XAWH',  'Wien Hütteldorf' ],
	[ 'XAWG',  'Wien Meidling' ],
	[ 'XAWMI', 'Wien Mitte' ],
	[ 'XAWNP', 'Wien Praterstern' ],
	[ 'XAWSG', 'Wien Simmering' ],
	[ 'XAWW',  'Wien Westbf' ],
	[ 'XAWNS', 'Wiener Neustadt' ],
	[ 'XAWP',  'Wien-Penzing' ],
	[ 'XSWI',  'Wil' ],
	[ 'XAWI',  'Wildon' ],
	[ 'XLWW',  'Wilwerwiltz' ],
	[ 'XFWI',  'Wilwisheim' ],
	[ 'XAWIG', 'Windischgarsten' ],
	[ 'XFWM',  'Wingen-sur-Moder' ],
	[ 'XNWI',  'Winschoten' ],
	[ 'XNWW',  'Winterswijk' ],
	[ 'XSWT',  'Winterthur' ],
	[ 'XFWBG', 'Wissembourg' ],
	[ 'XPWIT', 'Witnica' ],
	[ 'XSWB',  'Wittenbach' ],
	[ 'XFWT',  'Wittring' ],
	[ 'XNWO',  'Woerden' ],
	[ 'XSWO',  'Wohlen' ],
	[ 'XFWY',  'Woippy' ],
	[ 'XBWO',  'Wolfstee' ],
	[ 'XAWT',  'Wolfurt' ],
	[ 'XAWO',  'Wolkersdorf' ],
	[ 'XAWL',  'Wörgl' ],
	[ 'XNWM',  'Wormerveer' ],
	[ 'XPWR',  'Wroclaw Glowny' ],
	[ 'XPWRL', 'Wroclaw Lesnica' ],
	[ 'XPWRD', 'Wroclaw Now Dwor' ],
	[ 'XAY',   'Ybbs a.d.Donau' ],
	[ 'XSY',   'Yverdon' ],
	[ 'XBYG',  'Yves-Gomezee' ],
	[ 'XRZD',  'Zadar' ],
	[ 'XPZ',   'Zagan' ],
	[ 'XZZG',  'Zagorje' ],
	[ 'XRZ',   'Zagreb Gl.kol.' ],
	[ 'XNZA',  'Zandvoort a.Zee' ],
	[ 'XPZY',  'Zary' ],
	[ 'XPZS',  'Zasieki' ],
	[ 'XPZAW', 'Zawiercie' ],
	[ 'XPZA',  'Zbaszynek' ],
	[ 'XPZE',  'Zebrzydowa' ],
	[ 'XPZD',  'Zebrzydowice' ],
	[ 'XBZD',  'Zedelgem' ],
	[ 'XBZ',   'Zeebrugge' ],
	[ 'XTZRC', 'Zelezna Ruda C' ],
	[ 'XTZR',  'Zelezna Ruda M' ],
	[ 'XAZ',   'Zell am See' ],
	[ 'XAZZ',  'Zell am Ziller' ],
	[ 'XAZW',  'Zeltweg' ],
	[ 'XSZE',  'Zermatt' ],
	[ 'XFZ',   'Zetting' ],
	[ 'XNZV',  'Zevenaar' ],
	[ 'XPZG',  'Zgorzelec' ],
	[ 'XPZGM', 'Zgorzelec Miasto' ],
	[ 'XBZI',  'Zichem' ],
	[ 'XZZM',  'Zidani most' ],
	[ 'XSZB',  'Ziegelbrücke' ],
	[ 'XPZIG', 'Zielona Gora' ],
	[ 'XAZL',  'Zirl' ],
	[ 'XTFS',  'Zirovice-Seniky' ],
	[ 'XSZI',  'Zizers' ],
	[ 'XSZG',  'Zofingen' ],
	[ 'XBZO',  'Zolder' ],
	[ 'XSZU',  'Zug CH' ],
	[ 'XNZB',  'Zuidbroek' ],
	[ 'XEZM',  'Zumarraga' ],
	[ 'XSZN',  'Zürich Enge' ],
	[ 'XSZF',  'Zürich Flughafen' ],
	[ 'XSZH',  'Zürich HB' ],
	[ 'XSZW',  'Zürich Wiedikon' ],
	[ 'XSZWO', 'Zürich Wollishof' ],
	[ 'XSZA',  'Zürich-Altst.' ],
	[ 'XSZR',  'Zürich-Hardbr' ],
	[ 'XSZO',  'Zürich-Oerlikon' ],
	[ 'XSZP',  'Zürich-Opfikon' ],
	[ 'XSZS',  'Zürich-Stadelhof' ],
	[ 'XNZW',  'Zwijndrecht' ],
	[ 'XNZL',  'Zwolle' ],

);

sub get_stations {
	return @stations;
}

sub normalize {
	my ($val) = @_;

	$val =~ s{Ä}{Ae}g;
	$val =~ s{Ö}{Oe}g;
	$val =~ s{Ü}{Ue}g;
	$val =~ s{ä}{ae}g;
	$val =~ s{ö}{oe}g;
	$val =~ s{ß}{sz}g;
	$val =~ s{ü}{ue}g;

	return $val;
}

sub get_station {
	my ($name) = @_;

	my $ds100_match = firstval { $name eq $_->[0] } @stations;

	if ($ds100_match) {
		return ($ds100_match);
	}

	return get_station_by_name($name);
}

sub get_station_by_location {
	my ( $lon, $lat, $num_matches ) = @_;

	$num_matches //= 10;

	my $geo = Geo::Distance->new();

	# we only use geolocations inside germany.
	# For these, this fast preprocessing step will let through all
	# coordinates inside a 60km radius (and a bunch which are farther out)
	my @candidates = grep {
			   $#{$_} >= 4
		   and abs( $_->[3] - $lon ) < 1
		   and abs( $_->[4] - $lat )
		   < 1
	} @stations;
	my @distances
	  = map { $geo->distance( 'kilometer', $lon, $lat, $_->[3], $_->[4] ) }
	  @candidates;
	my @station_map = pairwise { [ $a, $b ] } @candidates, @distances;

	@station_map = sort { $a->[1] <=> $b->[1] } @station_map;
	splice( @station_map, 10 );

	return @station_map;
}


sub get_station_by_name {
	my ($name) = @_;

	my $nname = lc($name);
	my $actual_match = firstval { $nname eq lc( $_->[1] ) } @stations;

	if ($actual_match) {
		return ($actual_match);
	}

	$nname = normalize($nname);
	$actual_match = firstval { $nname eq normalize( lc( $_->[1] ) ) } @stations;
	if ($actual_match) {
		return ($actual_match);
	}

	my @distances   = map { distance( $nname, $_->[1] ) } @stations;
	my $min_dist    = min(@distances);
	my @station_map = pairwise { [ $a, $b ] } @stations, @distances;

	my @substring_matches = grep { $_->[1] =~ m{$name}i } @stations;
	my @levenshtein_matches
	  = map { $_->[0] } grep { $_->[1] == $min_dist } @station_map;

	return uniq_by { $_->[0] } ( @substring_matches, @levenshtein_matches );
}

1;

__END__

=head1 NAME

Travel::Status::DE::IRIS::Stations - Station name to station code mapping

=head1 SYNOPSIS

    use Travel::Status::DE::IRIS::Stations;

    my $name = 'Essen Hbf';
    my @stations = Travel::Status::DE::IRIS::Stations::get_station_by_name(
      $name);

    if (@stations < 1) {
      # no matching stations
    }
    elsif (@stations > 1) {
      # too many matches
    }
    else {
      printf("Input '%s' matched station code %s (as '%s')\n",
        $name, @{$stations[0]});
    }

=head1 VERSION

version 1.09

=head1 DESCRIPTION

This module contains a list of Deutsche Bahn stations, and also some stations
outside of Germany which are served by Deutsche Bahn trains. It offers
several accessors to look up stations based on names or geolocation data and
can also simply dump all known stations.

Each of the following methods returns a list of array references. Each
array reference describes a single station and contains either two or
five elements:

=over

=item * The station code (also known as DS100 / "Druckschrift 100" /
"Richtlinie 100"). A short string used exclusively by Deutsche Bahn APIs. Note
that it may contain space characters.

=item * The station name

=item * The international station number (IBNR, "Internationale Bahnhofsnummer")

=item * The station's longitude

=item * The station's latitude

=back

=head1 METHODS

=over

=item Travel::Status::DE::IRIS::Stations::get_stations

Returns a list of all known stations, lexically sorted by station name.

=item Travel::Status::DE::IRIS::Stations::get_station(I<$in>)

Returns a list of stations matching I<$in>.

If a I<$in> is a valid station code, only one element ([I<$in>, related name])
is returned. Otherwise, it is passed to get_station_by_name(I<$in>) (see
below).

Note that station codes matching is case sensitive and must be exact.

=item Travel::Status::DE::IRIS::Stations::get_station_by_location(I<$lon>, I<$lat>, I<$num_matches>)

Looks for stations which are close to longitude/latitude I<$lon>/I<$lat> and
returns the closest I<$num_matches> (defaults to 10) matches. Note that
stations which are located more than 70 kilometers away from I<$lon>/I<$lat>
may be ignored when computing the closest matches.

Note that location-based lookup is only supported for stations inside Germany,
since the station list data source does not provide geolocation data for
non-german stations.

=item Travel::Status::DE::IRIS::Stations::get_station_by_name(I<$name>)

Returns a list of stations where the station name matches I<$name>.

Matching happens in two steps: If a case-insensitive exact match exists, only
this one is returned. Otherwise, all stations whose name contains I<$name> as
a substring (also case-insensitive) and all stations whose name has a low
Levenshtein distance to I<$name> are returned.

This two-step behaviour makes sure that not-prefix-free stations can still be
matched directly. For instance, both "Essen-Steele" and "Essen-Steele Ost"
are valid station names, but "essen-steele" will only return "Essen-Steele".

=back

=head1 DIAGNOSTICS

None.

=head1 DEPENDENCIES

=over

=item * Geo::Distance(3pm)

=item * List::MoreUtils(3pm)

=item * List::Util(3pm)

=item * Text::LevenshteinXS(3pm)

=back

=head1 BUGS AND LIMITATIONS

There is no support for intelligent whitespaces (to also match "-" and similar)
yet.

=head1 SEE ALSO

Travel::Status::DE::IRIS(3pm).

=head1 AUTHOR

Station data: Copyright (C) 2016 by DB Station&Service AG, Europaplatz 1, 10557 Berlin, Germany

Lookup code: Copyright (C) 2014-2016 by Daniel Friesel E<lt>derf@finalrewind.orgE<gt>

=head1 LICENSE

This module is licensed under the same terms as Perl itself. Note that the
station data used by this module is also available under a CC-BY 4.0 license on
L<http://data.deutschebahn.com/dataset/data-haltestellen>.

EOF
