######################################################################
#
# JIPSJ_by_Unicode_CultiCoLtd.pl
#
# Copyright (c) 2018 INABA Hitoshi <ina@cpan.org> in a CPAN
######################################################################

# JIPS code book, Culti Co.,Ltd.
# http://www.culti.co.jp/2016/02/01/jips%e3%82%b3%e3%83%bc%e3%83%89%e3%83%96%e3%83%83%e3%82%af/

use strict;

my %JIPSJ_by_Unicode_CultiCoLtd = ();
my %Unicode_by_JIPSJ_CultiCoLtd = ();

while (<DATA>) {
    chomp;
    my($unicode, $jis, $sjis, $kuten, $jipsj) = split(/\t/, $_);

# print STDERR "($unicode)($jis)($sjis)($kuten)($jipsj)\n";

    if ($unicode !~ /^[0123456789ABCDEF]+$/) {
        next;
    }
    if ($jipsj   !~ /^[0123456789ABCDEF]+$/) {
        next;
    }
    if (('E000' le $unicode) and ($unicode le 'F8FF')) {
        next;
    }

    # ignore Private Use Area
    if (defined $JIPSJ_by_Unicode_CultiCoLtd{$unicode}) {
        die;
    }
    if (defined $Unicode_by_JIPSJ_CultiCoLtd{$jipsj}) {
        die;
    }

    $JIPSJ_by_Unicode_CultiCoLtd{$unicode} = $jipsj;
    $Unicode_by_JIPSJ_CultiCoLtd{$jipsj}   = $unicode;
}

sub JIPSJ_by_Unicode_CultiCoLtd {
    my($unicode) = @_;
    return $JIPSJ_by_Unicode_CultiCoLtd{$unicode};
}

sub keys_of_JIPSJ_by_Unicode_CultiCoLtd {
    return keys %JIPSJ_by_Unicode_CultiCoLtd;
}

sub values_of_JIPSJ_by_Unicode_CultiCoLtd {
    return values %JIPSJ_by_Unicode_CultiCoLtd;
}

1;

# Unicode	JIS	SJIS	_	JIPS
__DATA__
3000	2121	8140	00101	2121
3001	2122	8141	00102	2122
3002	2123	8142	00103	2123
FF0C	2124	8143	00104	2124
FF0E	2125	8144	00105	2125
30FB	2126	8145	00106	2126
FF1A	2127	8146	00107	2127
FF1B	2128	8147	00108	2128
FF1F	2129	8148	00109	2129
FF01	212A	8149	00110	212A
309B	212B	814A	00111	212B
309C	212C	814B	00112	212C
00B4	212D	814C	00113	212D
FF40	212E	814D	00114	212E
00A8	212F	814E	00115	212F
FF3E	2130	814F	00116	2130
FFE3	2131	8150	00117	2131
FF3F	2132	8151	00118	2132
30FD	2133	8152	00119	2133
30FE	2134	8153	00120	2134
309D	2135	8154	00121	2135
309E	2136	8155	00122	2136
3003	2137	8156	00123	2137
4EDD	2138	8157	00124	2138
3005	2139	8158	00125	2139
3006	213A	8159	00126	213A
E000				213B
30FC	213C	815B	00128	213C
2014				213D
2010	213E	815D	00130	213E
FF0F	213F	815E	00131	213F
FF3C	2140	815F	00132	2140
301C				2141
2016				2142
FF5C	2143	8162	00135	2143
2026	2144	8163	00136	2144
2025	2145	8164	00137	2145
2018	2146	8165	00138	2146
2019	2147	8166	00139	2147
201C	2148	8167	00140	2148
201D	2149	8168	00141	2149
FF08	214A	8169	00142	214A
FF09	214B	816A	00143	214B
3014	214C	816B	00144	214C
3015	214D	816C	00145	214D
FF3B	214E	816D	00146	214E
FF3D	214F	816E	00147	214F
FF5B	2150	816F	00148	2150
FF5D	2151	8170	00149	2151
3008	2152	8171	00150	2152
3009	2153	8172	00151	2153
300A	2154	8173	00152	2154
300B	2155	8174	00153	2155
300C	2156	8175	00154	2156
300D	2157	8176	00155	2157
300E	2158	8177	00156	2158
300F	2159	8178	00157	2159
3010	215A	8179	00158	215A
3011	215B	817A	00159	215B
FF0B	215C	817B	00160	215C
2212				215D
00B1	215E	817D	00162	215E
00D7	215F	817E	00163	215F
00F7	2160	8180	00164	2160
FF1D	2161	8181	00165	2161
2260	2162	8182	00166	2162
FF1C	2163	8183	00167	2163
FF1E	2164	8184	00168	2164
2266	2165	8185	00169	2165
2267	2166	8186	00170	2166
221E	2167	8187	00171	2167
2234	2168	8188	00172	2168
E001				2169
2640	216A	818A	00174	216A
00B0	216B	818B	00175	216B
2032	216C	818C	00176	216C
2033	216D	818D	00177	216D
2103	216E	818E	00178	216E
FFE5	216F	818F	00179	216F
FF04	2170	8190	00180	2170
FFE0	2171	8191	00181	2171
FFE1	2172	8192	00182	2172
FF05	2173	8193	00183	2173
FF03	2174	8194	00184	2174
FF06	2175	8195	00185	2175
FF0A	2176	8196	00186	2176
FF20	2177	8197	00187	2177
00A7	2178	8198	00188	2178
2606	2179	8199	00189	2179
2605	217A	819A	00190	217A
25CB	217B	819B	00191	217B
25CF	217C	819C	00192	217C
25CE	217D	819D	00193	217D
25C7	217E	819E	00194	217E
25C6	2221	819F	00201	2221
25A1	2222	81A0	00202	2222
25A0	2223	81A1	00203	2223
25B3	2224	81A2	00204	2224
25B2	2225	81A3	00205	2225
25BD	2226	81A4	00206	2226
25BC	2227	81A5	00207	2227
203B	2228	81A6	00208	2228
3012	2229	81A7	00209	2229
2192	222A	81A8	00210	222A
2190	222B	81A9	00211	222B
2191	222C	81AA	00212	222C
2193	222D	81AB	00213	222D
3013	222E	81AC	00214	222E
FF10	2330	824F	00316	2330
FF11	2331	8250	00317	2331
FF12	2332	8251	00318	2332
FF13	2333	8252	00319	2333
FF14	2334	8253	00320	2334
FF15	2335	8254	00321	2335
FF16	2336	8255	00322	2336
FF17	2337	8256	00323	2337
FF18	2338	8257	00324	2338
FF19	2339	8258	00325	2339
FF21	2341	8260	00333	2341
FF22	2342	8261	00334	2342
FF23	2343	8262	00335	2343
FF24	2344	8263	00336	2344
FF25	2345	8264	00337	2345
FF26	2346	8265	00338	2346
FF27	2347	8266	00339	2347
FF28	2348	8267	00340	2348
FF29	2349	8268	00341	2349
FF2A	234A	8269	00342	234A
FF2B	234B	826A	00343	234B
FF2C	234C	826B	00344	234C
FF2D	234D	826C	00345	234D
FF2E	234E	826D	00346	234E
FF2F	234F	826E	00347	234F
FF30	2350	826F	00348	2350
FF31	2351	8270	00349	2351
FF32	2352	8271	00350	2352
FF33	2353	8272	00351	2353
FF34	2354	8273	00352	2354
FF35	2355	8274	00353	2355
FF36	2356	8275	00354	2356
FF37	2357	8276	00355	2357
FF38	2358	8277	00356	2358
FF39	2359	8278	00357	2359
FF3A	235A	8279	00358	235A
FF41	2361	8281	00365	2361
FF42	2362	8282	00366	2362
FF43	2363	8283	00367	2363
FF44	2364	8284	00368	2364
FF45	2365	8285	00369	2365
FF46	2366	8286	00370	2366
FF47	2367	8287	00371	2367
FF48	2368	8288	00372	2368
FF49	2369	8289	00373	2369
FF4A	236A	828A	00374	236A
FF4B	236B	828B	00375	236B
FF4C	236C	828C	00376	236C
FF4D	236D	828D	00377	236D
FF4E	236E	828E	00378	236E
FF4F	236F	828F	00379	236F
FF50	2370	8290	00380	2370
FF51	2371	8291	00381	2371
FF52	2372	8292	00382	2372
FF53	2373	8293	00383	2373
FF54	2374	8294	00384	2374
FF55	2375	8295	00385	2375
FF56	2376	8296	00386	2376
FF57	2377	8297	00387	2377
FF58	2378	8298	00388	2378
FF59	2379	8299	00389	2379
FF5A	237A	829A	00390	237A
3041	2421	829F	00401	2421
3042	2422	82A0	00402	2422
3043	2423	82A1	00403	2423
3044	2424	82A2	00404	2424
3045	2425	82A3	00405	2425
3046	2426	82A4	00406	2426
3047	2427	82A5	00407	2427
3048	2428	82A6	00408	2428
3049	2429	82A7	00409	2429
304A	242A	82A8	00410	242A
304B	242B	82A9	00411	242B
304C	242C	82AA	00412	242C
304D	242D	82AB	00413	242D
304E	242E	82AC	00414	242E
304F	242F	82AD	00415	242F
3050	2430	82AE	00416	2430
3051	2431	82AF	00417	2431
3052	2432	82B0	00418	2432
3053	2433	82B1	00419	2433
3054	2434	82B2	00420	2434
3055	2435	82B3	00421	2435
3056	2436	82B4	00422	2436
3057	2437	82B5	00423	2437
3058	2438	82B6	00424	2438
3059	2439	82B7	00425	2439
305A	243A	82B8	00426	243A
305B	243B	82B9	00427	243B
305C	243C	82BA	00428	243C
305D	243D	82BB	00429	243D
305E	243E	82BC	00430	243E
305F	243F	82BD	00431	243F
3060	2440	82BE	00432	2440
3061	2441	82BF	00433	2441
3062	2442	82C0	00434	2442
3063	2443	82C1	00435	2443
3064	2444	82C2	00436	2444
3065	2445	82C3	00437	2445
3066	2446	82C4	00438	2446
3067	2447	82C5	00439	2447
3068	2448	82C6	00440	2448
3069	2449	82C7	00441	2449
306A	244A	82C8	00442	244A
306B	244B	82C9	00443	244B
306C	244C	82CA	00444	244C
306D	244D	82CB	00445	244D
306E	244E	82CC	00446	244E
306F	244F	82CD	00447	244F
3070	2450	82CE	00448	2450
3071	2451	82CF	00449	2451
3072	2452	82D0	00450	2452
3073	2453	82D1	00451	2453
3074	2454	82D2	00452	2454
3075	2455	82D3	00453	2455
3076	2456	82D4	00454	2456
3077	2457	82D5	00455	2457
3078	2458	82D6	00456	2458
3079	2459	82D7	00457	2459
307A	245A	82D8	00458	245A
307B	245B	82D9	00459	245B
307C	245C	82DA	00460	245C
307D	245D	82DB	00461	245D
307E	245E	82DC	00462	245E
307F	245F	82DD	00463	245F
3080	2460	82DE	00464	2460
3081	2461	82DF	00465	2461
3082	2462	82E0	00466	2462
3083	2463	82E1	00467	2463
3084	2464	82E2	00468	2464
3085	2465	82E3	00469	2465
3086	2466	82E4	00470	2466
3087	2467	82E5	00471	2467
3088	2468	82E6	00472	2468
3089	2469	82E7	00473	2469
308A	246A	82E8	00474	246A
308B	246B	82E9	00475	246B
308C	246C	82EA	00476	246C
308D	246D	82EB	00477	246D
308E	246E	82EC	00478	246E
308F	246F	82ED	00479	246F
3090	2470	82EE	00480	2470
3091	2471	82EF	00481	2471
3092	2472	82F0	00482	2472
3093	2473	82F1	00483	2473
30A1	2521	8340	00501	2521
30A2	2522	8341	00502	2522
30A3	2523	8342	00503	2523
30A4	2524	8343	00504	2524
30A5	2525	8344	00505	2525
30A6	2526	8345	00506	2526
30A7	2527	8346	00507	2527
30A8	2528	8347	00508	2528
30A9	2529	8348	00509	2529
30AA	252A	8349	00510	252A
30AB	252B	834A	00511	252B
30AC	252C	834B	00512	252C
30AD	252D	834C	00513	252D
30AE	252E	834D	00514	252E
30AF	252F	834E	00515	252F
30B0	2530	834F	00516	2530
30B1	2531	8350	00517	2531
30B2	2532	8351	00518	2532
30B3	2533	8352	00519	2533
30B4	2534	8353	00520	2534
30B5	2535	8354	00521	2535
30B6	2536	8355	00522	2536
30B7	2537	8356	00523	2537
30B8	2538	8357	00524	2538
30B9	2539	8358	00525	2539
30BA	253A	8359	00526	253A
30BB	253B	835A	00527	253B
30BC	253C	835B	00528	253C
30BD	253D	835C	00529	253D
30BE	253E	835D	00530	253E
30BF	253F	835E	00531	253F
30C0	2540	835F	00532	2540
30C1	2541	8360	00533	2541
30C2	2542	8361	00534	2542
30C3	2543	8362	00535	2543
30C4	2544	8363	00536	2544
30C5	2545	8364	00537	2545
30C6	2546	8365	00538	2546
30C7	2547	8366	00539	2547
30C8	2548	8367	00540	2548
30C9	2549	8368	00541	2549
30CA	254A	8369	00542	254A
30CB	254B	836A	00543	254B
30CC	254C	836B	00544	254C
30CD	254D	836C	00545	254D
30CE	254E	836D	00546	254E
30CF	254F	836E	00547	254F
30D0	2550	836F	00548	2550
30D1	2551	8370	00549	2551
30D2	2552	8371	00550	2552
30D3	2553	8372	00551	2553
30D4	2554	8373	00552	2554
30D5	2555	8374	00553	2555
30D6	2556	8375	00554	2556
30D7	2557	8376	00555	2557
30D8	2558	8377	00556	2558
30D9	2559	8378	00557	2559
30DA	255A	8379	00558	255A
30DB	255B	837A	00559	255B
30DC	255C	837B	00560	255C
30DD	255D	837C	00561	255D
30DE	255E	837D	00562	255E
30DF	255F	837E	00563	255F
30E0	2560	8380	00564	2560
30E1	2561	8381	00565	2561
30E2	2562	8382	00566	2562
30E3	2563	8383	00567	2563
30E4	2564	8384	00568	2564
30E5	2565	8385	00569	2565
30E6	2566	8386	00570	2566
30E7	2567	8387	00571	2567
30E8	2568	8388	00572	2568
30E9	2569	8389	00573	2569
30EA	256A	838A	00574	256A
30EB	256B	838B	00575	256B
30EC	256C	838C	00576	256C
30ED	256D	838D	00577	256D
30EE	256E	838E	00578	256E
30EF	256F	838F	00579	256F
30F0	2570	8390	00580	2570
30F1	2571	8391	00581	2571
30F2	2572	8392	00582	2572
30F3	2573	8393	00583	2573
30F4	2574	8394	00584	2574
30F5	2575	8395	00585	2575
30F6	2576	8396	00586	2576
0391	2621	839F	00601	2621
0392	2622	83A0	00602	2622
0393	2623	83A1	00603	2623
0394	2624	83A2	00604	2624
0395	2625	83A3	00605	2625
0396	2626	83A4	00606	2626
0397	2627	83A5	00607	2627
0398	2628	83A6	00608	2628
0399	2629	83A7	00609	2629
039A	262A	83A8	00610	262A
039B	262B	83A9	00611	262B
039C	262C	83AA	00612	262C
039D	262D	83AB	00613	262D
039E	262E	83AC	00614	262E
039F	262F	83AD	00615	262F
03A0	2630	83AE	00616	2630
03A1	2631	83AF	00617	2631
03A3	2632	83B0	00618	2632
03A4	2633	83B1	00619	2633
03A5	2634	83B2	00620	2634
03A6	2635	83B3	00621	2635
03A7	2636	83B4	00622	2636
03A8	2637	83B5	00623	2637
03A9	2638	83B6	00624	2638
03B1	2641	83BF	00633	2641
03B2	2642	83C0	00634	2642
03B3	2643	83C1	00635	2643
03B4	2644	83C2	00636	2644
03B5	2645	83C3	00637	2645
03B6	2646	83C4	00638	2646
03B7	2647	83C5	00639	2647
03B8	2648	83C6	00640	2648
03B9	2649	83C7	00641	2649
03BA	264A	83C8	00642	264A
03BB	264B	83C9	00643	264B
03BC	264C	83CA	00644	264C
03BD	264D	83CB	00645	264D
03BE	264E	83CC	00646	264E
03BF	264F	83CD	00647	264F
03C0	2650	83CE	00648	2650
03C1	2651	83CF	00649	2651
03C3	2652	83D0	00650	2652
03C4	2653	83D1	00651	2653
03C5	2654	83D2	00652	2654
03C6	2655	83D3	00653	2655
03C7	2656	83D4	00654	2656
03C8	2657	83D5	00655	2657
03C9	2658	83D6	00656	2658
0410	2721	8440	00701	2721
0411	2722	8441	00702	2722
0412	2723	8442	00703	2723
0413	2724	8443	00704	2724
0414	2725	8444	00705	2725
0415	2726	8445	00706	2726
0401	2727	8446	00707	2727
0416	2728	8447	00708	2728
0417	2729	8448	00709	2729
0418	272A	8449	00710	272A
0419	272B	844A	00711	272B
041A	272C	844B	00712	272C
041B	272D	844C	00713	272D
041C	272E	844D	00714	272E
041D	272F	844E	00715	272F
041E	2730	844F	00716	2730
041F	2731	8450	00717	2731
0420	2732	8451	00718	2732
0421	2733	8452	00719	2733
0422	2734	8453	00720	2734
0423	2735	8454	00721	2735
0424	2736	8455	00722	2736
0425	2737	8456	00723	2737
0426	2738	8457	00724	2738
0427	2739	8458	00725	2739
0428	273A	8459	00726	273A
0429	273B	845A	00727	273B
042A	273C	845B	00728	273C
042B	273D	845C	00729	273D
042C	273E	845D	00730	273E
042D	273F	845E	00731	273F
042E	2740	845F	00732	2740
042F	2741	8460	00733	2741
0430	2751	8470	00749	2751
0431	2752	8471	00750	2752
0432	2753	8472	00751	2753
0433	2754	8473	00752	2754
0434	2755	8474	00753	2755
0435	2756	8475	00754	2756
0451	2757	8476	00755	2757
0436	2758	8477	00756	2758
0437	2759	8478	00757	2759
0438	275A	8479	00758	275A
0439	275B	847A	00759	275B
043A	275C	847B	00760	275C
043B	275D	847C	00761	275D
043C	275E	847D	00762	275E
043D	275F	847E	00763	275F
043E	2760	8480	00764	2760
043F	2761	8481	00765	2761
0440	2762	8482	00766	2762
0441	2763	8483	00767	2763
0442	2764	8484	00768	2764
0443	2765	8485	00769	2765
0444	2766	8486	00770	2766
0445	2767	8487	00771	2767
0446	2768	8488	00772	2768
0447	2769	8489	00773	2769
0448	276A	848A	00774	276A
0449	276B	848B	00775	276B
044A	276C	848C	00776	276C
044B	276D	848D	00777	276D
044C	276E	848E	00778	276E
044D	276F	848F	00779	276F
044E	2770	8490	00780	2770
044F	2771	8491	00781	2771
E002				2821
E003				2822
E004				2823
E005				2824
E006				2825
E007				2826
E008				2827
E009				2828
E00A				2829
E00B				2830
E00C				2831
E00D				2832
E00E				2833
E00F				2834
E010				2835
E011				2836
E012				2840
E013				2841
E014				2842
E015				2843
E016				2844
E017				2845
E018				2846
E019				2850
E01A				2851
E01B				2852
E01C				2853
E01D				2854
E01E				2855
E01F				2856
E020				2857
E021				2921
E022				2922
E023				2923
E024				2924
E025				2925
E026				2926
E027				2927
E028				2928
E029				2929
E02A				292A
E02B				292B
E02C				292C
E02D				292D
E02E				292E
E02F				292F
E030				2930
E031				2931
E032				2932
E033				2933
E034				2934
E035				2935
E036				2936
E037				2937
E038				2938
E039				2939
E03A				293A
E03B				293B
E03C				293C
E03D				293D
E03E				293E
E03F				293F
E040				2940
E041				2941
E042				2942
E043				2943
E044				2944
E045				2945
E046				2946
E047				2947
E048				2948
E049				2949
E04A				294A
E04B				294B
E04C				294C
E04D				294D
E04E				294E
E04F				294F
E050				2950
E051				2951
E052				2952
E053				2953
E054				2954
E055				2955
E056				2956
E057				2957
E058				2958
E059				2959
E05A				295A
E05B				295B
E05C				295C
E05D				295D
E05E				295E
E05F				295F
E060				2960
E061				2961
E062				2962
E063				2963
E064				2964
E065				2965
E066				2966
E067				2967
E068				2968
E069				2969
E06A				296A
E06B				296B
E06C				296C
E06D				296D
E06E				296E
E06F				296F
E070				2970
E071				2971
E072				2972
E073				2973
E074				2974
E075				2975
E076				2976
E077				2977
E078				2978
E079				2979
E07A				297A
E07B				297B
E07C				297C
E07D				297D
E07E				297E
E07F				2A21
E080				2A22
E081				2A23
E082				2A24
E083				2A25
E084				2A26
E085				2A27
E086				2A28
E087				2A29
E088				2A2A
E089				2A2B
E08A				2A2C
E08B				2A2D
E08C				2A2E
E08D				2A2F
E08E				2A30
E08F				2A31
E090				2A32
E091				2A33
E092				2A34
E093				2A35
E094				2A36
E095				2A37
E096				2A38
E097				2A39
E098				2A3A
E099				2A3B
E09A				2A3C
E09B				2A3D
E09C				2A3E
E09D				2A3F
E09E				2A40
E09F				2A41
E0A0				2A42
E0A1				2A43
E0A2				2A44
E0A3				2A45
E0A4				2A46
E0A5				2A47
E0A6				2A48
E0A7				2A49
E0A8				2A4A
E0A9				2A4B
E0AA				2A4C
E0AB				2A4D
E0AC				2A4E
E0AD				2A4F
E0AE				2A50
E0AF				2A51
E0B0				2A52
E0B1				2A53
E0B2				2A54
E0B3				2A55
E0B4				2A56
E0B5				2A57
E0B6				2A58
E0B7				2A59
E0B8				2A5A
E0B9				2A5B
E0BA				2A5C
E0BB				2A5D
E0BC				2A5E
E0BD				2A5F
E0BE				2A60
E0BF				2A61
E0C0				2A62
E0C1				2A63
E0C2				2A64
E0C3				2A65
E0C4				2A66
E0C5				2A67
E0C6				2A68
E0C7				2A69
E0C8				2A6A
E0C9				2A6B
E0CA				2A6C
E0CB				2A6D
E0CC				2A6E
E0CD				2A6F
E0CE				2A70
E0CF				2A71
E0D0				2A72
E0D1				2A73
E0D2				2A74
E0D3				2A75
E0D4				2A76
E0D5				2A77
E0D6				2A78
E0D7				2A79
E0D8				2A7A
E0D9				2A7B
E0DA				2A7C
E0DB				2A7D
E0DC				2A7E
E0DD				2B21
E0DE				2B22
E0DF				2B23
E0E0				2B24
E0E1				2B25
E0E2				2B26
E0E3				2B27
E0E4				2B28
E0E5				2B29
E0E6				2B2A
E0E7				2B2B
E0E8				2B2C
E0E9				2B2D
E0EA				2B2E
E0EB				2B2F
E0EC				2B30
E0ED				2B31
E0EE				2B32
E0EF				2B33
E0F0				2B34
E0F1				2B35
E0F2				2B36
E0F3				2B37
E0F4				2B38
E0F5				2B39
E0F6				2B3A
E0F7				2B3B
E0F8				2B3C
E0F9				2B3D
E0FA				2B3E
E0FB				2B3F
E0FC				2B40
E0FD				2B41
E0FE				2B42
E0FF				2B43
E100				2B44
E101				2B45
E102				2B46
E103				2B47
E104				2B48
E105				2B49
E106				2B4A
E107				2B4B
E108				2B4C
E109				2B4D
E10A				2B4E
E10B				2B4F
E10C				2B50
E10D				2B51
E10E				2B52
E10F				2B53
E110				2B54
E111				2B55
E112				2B56
E113				2B57
E114				2B58
E115				2B59
E116				2B5A
E117				2B5B
E118				2B5C
E119				2B5D
E11A				2B5E
E11B				2B5F
E11C				2B60
E11D				2B61
E11E				2B62
E11F				2B63
E120				2B64
E121				2B65
E122				2B66
E123				2B67
E124				2B68
E125				2B69
E126				2B6A
E127				2B6B
E128				2B6C
E129				2B6D
E12A				2B6E
E12B				2B6F
E12C				2B70
E12D				2B71
E12E				2B72
E12F				2B73
E130				2B74
E131				2B75
E132				2B76
E133				2B77
E134				2B78
E135				2B79
E136				2B7A
E137				2B7B
E138				2B7C
E139				2B7D
E13A				2B7E
E13B				2C24
E13C				2C25
E13D				2C26
E13E				2C27
E13F				2C28
E140				2C29
E141				2C2A
E142				2C2B
E143				2C2C
E144				2C2D
E145				2C2E
E146				2C2F
E147				2C30
E148				2C31
E149				2C32
E14A				2C33
E14B				2C34
E14C				2C35
E14D				2C36
E14E				2C37
E14F				2C38
E150				2C39
E151				2C3A
E152				2C3B
E153				2C3C
E154				2C3D
E155				2C3E
E156				2C3F
E157				2C40
E158				2C41
E159				2C42
E15A				2C43
E15B				2C44
E15C				2C45
E15D				2C46
E15E				2C47
E15F				2C48
E160				2C49
E161				2C4A
E162				2C4B
E163				2C4C
E164				2C4D
E165				2C4E
E166				2C4F
E167				2C50
E168				2C51
E169				2C52
E16A				2C53
E16B				2C54
E16C				2C55
E16D				2C56
E16E				2C57
E16F				2C58
E170				2C59
E171				2C5A
E172				2C5B
E173				2C5C
E174				2C5D
E175				2C5E
E176				2C5F
E177				2C60
E178				2C61
E179				2C62
E17A				2C63
E17B				2C64
E17C				2C65
E17D				2C66
E17E				2C67
E17F				2C68
E180				2C69
E181				2C6A
E182				2C6B
E183				2C6C
E184				2C6D
E185				2C6E
E186				2C6F
2460				2D21
2461				2D22
2462				2D23
2463				2D24
2464				2D25
2465				2D26
2466				2D27
2467				2D28
2468				2D29
2469				2D2A
246A				2D2B
246B				2D2C
246C				2D2D
246D				2D2E
246E				2D2F
246F				2D30
2470				2D31
2471				2D32
2472				2D33
2473				2D34
2160		FA4A	11511	2D35
2161		FA4B	11512	2D36
2162		FA4C	11513	2D37
2163		FA4D	11514	2D38
2164		FA4E	11515	2D39
2165		FA4F	11516	2D3A
2166		FA50	11517	2D3B
2167		FA51	11518	2D3C
2168		FA52	11519	2D3D
2169		FA53	11520	2D3E
3349				2D40
3314				2D41
E187				2D42
334D				2D43
E188				2D44
3327				2D45
E189				2D46
3336				2D47
3351	2D48	8767	01340	2D48
E18A				2D49
330D				2D4A
3326				2D4B
E18B				2D4C
332B				2D4D
334A				2D4E
E18C				2D4F
339C				2D50
339D				2D51
339E				2D52
338E				2D53
338F				2D54
33C4				2D55
33A1				2D56
337B	2D5F	877E	01363	2D5F
301D	2D60	8780	01364	2D60
301F	2D61	8781	01365	2D61
2116	2D62	8782	01366	2D62
33CD	2D63	8783	01367	2D63
E18D				2D64
32A4	2D65	8785	01369	2D65
32A5	2D66	8786	01370	2D66
32A6	2D67	8787	01371	2D67
32A7	2D68	8788	01372	2D68
32A8	2D69	8789	01373	2D69
3231		FA58	11525	2D6A
3232				2D6B
3239				2D6C
337E	2D6D	878D	01377	2D6D
337D	2D6E	878E	01378	2D6E
337C	2D6F	878F	01379	2D6F
2252	2262	81E0	00266	2D70
2261	2261	81DF	00265	2D71
222B	2269	81E7	00273	2D72
222E	2D73	8793	01383	2D73
2211	2D74	8794	01384	2D74
221A	2265	81E3	00269	2D75
22A5	225D	81DB	00261	2D76
2220	225C	81DA	00260	2D77
221F	2D78	8798	01388	2D78
22BF	2D79	8799	01389	2D79
2235	2268	81E6	00272	2D7A
2229	2241	81BF	00233	2D7B
222A	2240	81BE	00232	2D7C
E18E				2E21
E18F				2E22
E190				2E23
E191				2E24
E192				2E25
E193				2E26
E194				2E27
E195				2E28
E196				2E29
E197				2E2A
E198				2E2B
E199				2E2C
E19A				2E2D
E19B				2E2E
E19C				2E2F
E19D				2E30
E19E				2E31
E19F				2E32
E1A0				2E33
E1A1				2E34
E1A2				2E35
E1A3				2E36
E1A4				2E37
E1A5				2E38
E1A6				2E39
E1A7				2E3A
E1A8				2E3B
E1A9				2E3C
E1AA				2E3D
E1AB				2E3E
E1AC				2E3F
E1AD				2E40
E1AE				2E41
E1AF				2E42
E1B0				2E43
E1B1				2E44
E1B2				2E45
E1B3				2E46
E1B4				2E47
E1B5				2E48
E1B6				2E49
E1B7				2E4A
E1B8				2E4B
E1B9				2E4C
E1BA				2E4D
E1BB				2E4E
E1BC				2E4F
E1BD				2E50
E1BE				2E51
E1BF				2E52
E1C0				2E53
E1C1				2E54
E1C2				2E55
E1C3				2E56
E1C4				2E57
E1C5				2E58
E1C6				2E59
E1C7				2E5A
E1C8				2E5B
E1C9				2E5C
E1CA				2E5D
E1CB				2E5E
E1CC				2E5F
E1CD				2E60
E1CE				2E61
E1CF				2E62
E1D0				2E63
E1D1				2E64
E1D2				2E65
E1D3				2E66
E1D4				2E67
E1D5				2E68
E1D6				2E69
E1D7				2E6A
E1D8				2E6B
E1D9				2E6C
E1DA				2E6D
E1DB				2E6E
E1DC				2E6F
E1DD				2E70
E1DE				2E71
E1DF				2E72
E1E0				2E73
E1E1				2E74
E1E2				2E75
E1E3				2E76
E1E4				2E77
E1E5				2E78
E1E6				2E79
E1E7				2E7A
E1E8				2E7B
E1E9				2E7C
E1EA				2E7D
E1EB				2E7E
E1EC				2F21
E1ED				2F22
E1EE				2F23
E1EF				2F24
E1F0				2F25
E1F1				2F26
E1F2				2F27
E1F3				2F28
E1F4				2F29
E1F5				2F2A
E1F6				2F2B
E1F7				2F2C
E1F8				2F2D
E1F9				2F2E
E1FA				2F2F
E1FB				2F30
E1FC				2F31
E1FD				2F32
E1FE				2F33
E1FF				2F34
E200				2F35
E201				2F36
E202				2F37
E203				2F38
E204				2F39
E205				2F3A
E206				2F3B
E207				2F3C
E208				2F3D
E209				2F3E
E20A				2F3F
E20B				2F40
E20C				2F41
E20D				2F42
E20E				2F43
E20F				2F44
E210				2F45
E211				2F46
E212				2F47
E213				2F48
E214				2F49
E215				2F4A
E216				2F4B
E217				2F4C
E218				2F4D
E219				2F4E
E21A				2F4F
E21B				2F50
E21C				2F51
E21D				2F52
E21E				2F53
E21F				2F54
E220				2F55
E221				2F56
E222				2F57
E223				2F58
E224				2F59
E225				2F5A
E226				2F5B
E227				2F5C
E228				2F5D
E229				2F5E
E22A				2F5F
E22B				2F60
E22C				2F61
E22D				2F62
E22E				2F63
E22F				2F64
E230				2F65
E231				2F66
E232				2F67
E233				2F68
E234				2F69
E235				2F6A
E236				2F6B
E237				2F6C
E238				2F6D
E239				2F6E
E23A				2F6F
E23B				2F70
E23C				2F71
E23D				2F72
E23E				2F73
E23F				2F74
E240				2F75
E241				2F76
E242				2F77
E243				2F78
E244				2F79
E245				2F7A
E246				2F7B
E247				2F7C
E248				2F7D
E249				2F7E
4E9C	3021	889F	01601	3021
555E				3022
5A03	3023	88A1	01603	3023
963F	3024	88A2	01604	3024
54C0	3025	88A3	01605	3025
611B	3026	88A4	01606	3026
6328	3027	88A5	01607	3027
59F6	3028	88A6	01608	3028
9022	3029	88A7	01609	3029
8475	302A	88A8	01610	302A
831C	302B	88A9	01611	302B
7A50	302C	88AA	01612	302C
60AA	302D	88AB	01613	302D
63E1	302E	88AC	01614	302E
6E25	302F	88AD	01615	302F
65ED	3030	88AE	01616	3030
8466	3031	88AF	01617	3031
82A6	3032	88B0	01618	3032
9C3A	724D	E9CB	08245	3033
6893	3034	88B2	01620	3034
5727	3035	88B3	01621	3035
65A1	3036	88B4	01622	3036
6271	3037	88B5	01623	3037
5B9B	3038	88B6	01624	3038
59D0	3039	88B7	01625	3039
E24A				303A
E24B				303B
7D62	303C	88BA	01628	303C
7DBE	303D	88BB	01629	303D
9B8E	303E	88BC	01630	303E
6216	303F	88BD	01631	303F
7C9F	3040	88BE	01632	3040
88B7	3041	88BF	01633	3041
5B89	3042	88C0	01634	3042
5EB5	3043	88C1	01635	3043
6309	3044	88C2	01636	3044
6697	3045	88C3	01637	3045
6848	3046	88C4	01638	3046
95C7	3047	88C5	01639	3047
978D	3048	88C6	01640	3048
674F	3049	88C7	01641	3049
4EE5	304A	88C8	01642	304A
4F0A	304B	88C9	01643	304B
4F4D	304C	88CA	01644	304C
4F9D	304D	88CB	01645	304D
5049	304E	88CC	01646	304E
56F2	304F	88CD	01647	304F
5937	3050	88CE	01648	3050
59D4	3051	88CF	01649	3051
5A01	3052	88D0	01650	3052
5C09	3053	88D1	01651	3053
60DF	3054	88D2	01652	3054
610F	3055	88D3	01653	3055
6170	3056	88D4	01654	3056
6613	3057	88D5	01655	3057
6905	3058	88D6	01656	3058
70BA	3059	88D7	01657	3059
754F	305A	88D8	01658	305A
7570	305B	88D9	01659	305B
79FB	305C	88DA	01660	305C
7DAD	305D	88DB	01661	305D
7DEF	305E	88DC	01662	305E
80C3	305F	88DD	01663	305F
840E	3060	88DE	01664	3060
8863	3061	88DF	01665	3061
8B02	3062	88E0	01666	3062
9055	3063	88E1	01667	3063
907A	3064	88E2	01668	3064
533B	3065	88E3	01669	3065
4E95	3066	88E4	01670	3066
4EA5	3067	88E5	01671	3067
57DF	3068	88E6	01672	3068
80B2	3069	88E7	01673	3069
90C1	306A	88E8	01674	306A
78EF	306B	88E9	01675	306B
4E00	306C	88EA	01676	306C
58F1	306D	88EB	01677	306D
E24C				306E
9038	306F	88ED	01679	306F
7A32	3070	88EE	01680	3070
8328	3071	88EF	01681	3071
828B	3072	88F0	01682	3072
E24D				3073
5141	3074	88F2	01684	3074
5370	3075	88F3	01685	3075
54BD	3076	88F4	01686	3076
54E1	3077	88F5	01687	3077
56E0	3078	88F6	01688	3078
59FB	3079	88F7	01689	3079
5F15	307A	88F8	01690	307A
98F2	307B	88F9	01691	307B
E24E				307C
80E4	307D	88FB	01693	307D
852D	307E	88FC	01694	307E
9662	3121	8940	01701	3121
9670	3122	8941	01702	3122
96A0	3123	8942	01703	3123
97FB	3124	8943	01704	3124
540B	3125	8944	01705	3125
53F3	3126	8945	01706	3126
5B87	3127	8946	01707	3127
70CF	3128	8947	01708	3128
7FBD	3129	8948	01709	3129
E24F				312A
96E8	312B	894A	01711	312B
536F	312C	894B	01712	312C
9D5C	312D	894C	01713	312D
7ABA	312E	894D	01714	312E
4E11	312F	894E	01715	312F
7893	3130	894F	01716	3130
81FC	3131	8950	01717	3131
6E26	3132	8951	01718	3132
5618	3133	8952	01719	3133
5504	3134	8953	01720	3134
E250				3135
851A	3136	8955	01722	3136
9C3B	3137	8956	01723	3137
59E5	3138	8957	01724	3138
E251				3139
6D66	313A	8959	01726	313A
74DC	313B	895A	01727	313B
E252				313C
E253				313D
4E91	313E	895D	01730	313E
904B	313F	895E	01731	313F
96F2	3140	895F	01732	3140
834F	3141	8960	01733	3141
E254				3142
53E1	3143	8962	01735	3143
55B6	3144	8963	01736	3144
5B30	3145	8964	01737	3145
5F71	3146	8965	01738	3146
6620	3147	8966	01739	3147
66F3	3148	8967	01740	3148
6804	3149	8968	01741	3149
6C38	314A	8969	01742	314A
6CF3	314B	896A	01743	314B
6D29	314C	896B	01744	314C
745B	314D	896C	01745	314D
76C8	314E	896D	01746	314E
7A4E	314F	896E	01747	314F
9834	3150	896F	01748	3150
82F1	3151	8970	01749	3151
885B	3152	8971	01750	3152
8A60	3153	8972	01751	3153
92ED	3154	8973	01752	3154
6DB2	3155	8974	01753	3155
75AB	3156	8975	01754	3156
76CA	3157	8976	01755	3157
99C5	3158	8977	01756	3158
60A6	3159	8978	01757	3159
8B01	315A	8979	01758	315A
8D8A	315B	897A	01759	315B
95B2	315C	897B	01760	315C
698E	315D	897C	01761	315D
53AD	315E	897D	01762	315E
5186	315F	897E	01763	315F
5712	3160	8980	01764	3160
5830	3161	8981	01765	3161
5944	3162	8982	01766	3162
5BB4	3163	8983	01767	3163
5EF6	3164	8984	01768	3164
6028	3165	8985	01769	3165
63A9	3166	8986	01770	3166
63F4	3167	8987	01771	3167
E255				3168
6F14	3169	8989	01773	3169
E256				316A
E257				316B
7159	316C	898C	01776	316C
71D5	316D	898D	01777	316D
733F	316E	898E	01778	316E
E258				316F
8276	3170	8990	01780	3170
82D1	3171	8991	01781	3171
8597	3172	8992	01782	3172
9060	3173	8993	01783	3173
E259				3174
9D1B	3175	8995	01785	3175
5869	3176	8996	01786	3176
65BC	3177	8997	01787	3177
6C5A	3178	8998	01788	3178
7525	3179	8999	01789	3179
51F9	317A	899A	01790	317A
592E	317B	899B	01791	317B
5965	317C	899C	01792	317C
5F80	317D	899D	01793	317D
5FDC	317E	899E	01794	317E
62BC	3221	899F	01801	3221
65FA	3222	89A0	01802	3222
6A2A	3223	89A1	01803	3223
6B27	3224	89A2	01804	3224
6BB4	3225	89A3	01805	3225
738B	3226	89A4	01806	3226
E25A				3227
E25B				3228
9DAF	7274	E9F2	08284	3229
9DD7				322A
9EC4	322B	89A9	01811	322B
5CA1	322C	89AA	01812	322C
6C96	322D	89AB	01813	322D
837B	322E	89AC	01814	322E
5104	322F	89AD	01815	322F
5C4B	3230	89AE	01816	3230
61B6	3231	89AF	01817	3231
81C6	3232	89B0	01818	3232
6876	3233	89B1	01819	3233
7261	3234	89B2	01820	3234
4E59	3235	89B3	01821	3235
4FFA	3236	89B4	01822	3236
5378	3237	89B5	01823	3237
6069	3238	89B6	01824	3238
6E29	3239	89B7	01825	3239
7A4F	323A	89B8	01826	323A
97F3	323B	89B9	01827	323B
4E0B	323C	89BA	01828	323C
5316	323D	89BB	01829	323D
4EEE	323E	89BC	01830	323E
4F55	323F	89BD	01831	323F
4F3D	3240	89BE	01832	3240
4FA1	3241	89BF	01833	3241
4F73	3242	89C0	01834	3242
52A0	3243	89C1	01835	3243
53EF	3244	89C2	01836	3244
5609	3245	89C3	01837	3245
590F	3246	89C4	01838	3246
E25C				3247
E25D				3248
5BE1	3249	89C7	01841	3249
79D1	324A	89C8	01842	324A
6687	324B	89C9	01843	324B
679C	324C	89CA	01844	324C
67B6	324D	89CB	01845	324D
6B4C	324E	89CC	01846	324E
6CB3	324F	89CD	01847	324F
706B	3250	89CE	01848	3250
73C2	3251	89CF	01849	3251
798D	3252	89D0	01850	3252
79BE	3253	89D1	01851	3253
E25E				3254
7B87	3255	89D3	01853	3255
82B1	3256	89D4	01854	3256
82DB	3257	89D5	01855	3257
8304	3258	89D6	01856	3258
8377	3259	89D7	01857	3259
83EF	325A	89D8	01858	325A
83D3	325B	89D9	01859	325B
8766	325C	89DA	01860	325C
8AB2	325D	89DB	01861	325D
5629	325E	89DC	01862	325E
8CA8	325F	89DD	01863	325F
E25F				3260
904E	3261	89DF	01865	3261
971E	3262	89E0	01866	3262
868A	3263	89E1	01867	3263
4FC4	3264	89E2	01868	3264
5CE8	3265	89E3	01869	3265
6211	3266	89E4	01870	3266
7259	3267	89E5	01871	3267
753B	3268	89E6	01872	3268
81E5	3269	89E7	01873	3269
82BD	326A	89E8	01874	326A
86FE	326B	89E9	01875	326B
8CC0	326C	89EA	01876	326C
96C5	326D	89EB	01877	326D
9913	326E	89EC	01878	326E
99D5	326F	89ED	01879	326F
4ECB	3270	89EE	01880	3270
4F1A	3271	89EF	01881	3271
89E3	3272	89F0	01882	3272
56DE	3273	89F1	01883	3273
584A	3274	89F2	01884	3274
58CA	3275	89F3	01885	3275
E260				3276
5FEB	3277	89F5	01887	3277
602A	3278	89F6	01888	3278
6094	3279	89F7	01889	3279
E261				327A
61D0	327B	89F9	01891	327B
6212	327C	89FA	01892	327C
E262				327D
6539	327E	89FC	01894	327E
9B41	3321	8A40	01901	3321
E263				3322
68B0	3323	8A42	01903	3323
6D77	3324	8A43	01904	3324
7070	3325	8A44	01905	3325
754C	3326	8A45	01906	3326
7686	3327	8A46	01907	3327
7D75	3328	8A47	01908	3328
82A5	3329	8A48	01909	3329
87F9	332A	8A49	01910	332A
958B	332B	8A4A	01911	332B
968E	332C	8A4B	01912	332C
8C9D	332D	8A4C	01913	332D
51F1	332E	8A4D	01914	332E
52BE	332F	8A4E	01915	332F
5916	3330	8A4F	01916	3330
54B3	3331	8A50	01917	3331
5BB3	3332	8A51	01918	3332
5D16	3333	8A52	01919	3333
6168	3334	8A53	01920	3334
6982	3335	8A54	01921	3335
6DAF	3336	8A55	01922	3336
788D	3337	8A56	01923	3337
84CB	3338	8A57	01924	3338
8857	3339	8A58	01925	3339
8A72	333A	8A59	01926	333A
93A7	333B	8A5A	01927	333B
9AB8	333C	8A5B	01928	333C
6D6C	333D	8A5C	01929	333D
99A8	333E	8A5D	01930	333E
86D9	333F	8A5E	01931	333F
57A3	3340	8A5F	01932	3340
67FF	3341	8A60	01933	3341
8823	695A	E579	07358	3342
920E	3343	8A62	01935	3343
5283	3344	8A63	01936	3344
5687	3345	8A64	01937	3345
5404	3346	8A65	01938	3346
5ED3	3347	8A66	01939	3347
62E1	3348	8A67	01940	3348
652A	5978	9D98	05788	3349
683C	334A	8A69	01942	334A
6838	334B	8A6A	01943	334B
E264				334C
7372	334D	8A6C	01945	334D
78BA	334E	8A6D	01946	334E
7A6B	334F	8A6E	01947	334F
899A	3350	8A6F	01948	3350
89D2	3351	8A70	01949	3351
8D6B	3352	8A71	01950	3352
8F03	3353	8A72	01951	3353
90ED	3354	8A73	01952	3354
95A3	3355	8A74	01953	3355
9694	3356	8A75	01954	3356
9769	3357	8A76	01955	3357
5B66	3358	8A77	01956	3358
5CB3	3359	8A78	01957	3359
697D	335A	8A79	01958	335A
984D	335B	8A7A	01959	335B
984E	335C	8A7B	01960	335C
639B	335D	8A7C	01961	335D
7B20	335E	8A7D	01962	335E
6A2B	335F	8A7E	01963	335F
6A7F	3360	8A80	01964	3360
68B6	3361	8A81	01965	3361
9C0D	3362	8A82	01966	3362
6F5F	3363	8A83	01967	3363
5272	3364	8A84	01968	3364
E265				3365
6070	3366	8A86	01970	3366
62EC	3367	8A87	01971	3367
6D3B	3368	8A88	01972	3368
6E07	3369	8A89	01973	3369
6ED1	336A	8A8A	01974	336A
E266				336B
8910	336C	8A8C	01976	336C
8F44	336D	8A8D	01977	336D
4E14	336E	8A8E	01978	336E
9C39	336F	8A8F	01979	336F
53F6	3370	8A90	01980	3370
E267				3371
6A3A	3372	8A92	01982	3372
E268				3373
682A	3374	8A94	01984	3374
515C	3375	8A95	01985	3375
7AC8	635E	E27D	06762	3376
84B2	3377	8A97	01987	3377
E269				3378
938C	3379	8A99	01989	3379
E26A				337A
9D28	337B	8A9B	01991	337B
6822	337C	8A9C	01992	337C
8305	337D	8A9D	01993	337D
8431	337E	8A9E	01994	337E
7CA5	3421	8A9F	02001	3421
5208	3422	8AA0	02002	3422
82C5	3423	8AA1	02003	3423
74E6	3424	8AA2	02004	3424
4E7E	3425	8AA3	02005	3425
4F83	3426	8AA4	02006	3426
51A0	3427	8AA5	02007	3427
5BD2	3428	8AA6	02008	3428
520A	3429	8AA7	02009	3429
52D8	342A	8AA8	02010	342A
52E7	342B	8AA9	02011	342B
5DFB	342C	8AAA	02012	342C
559A	342D	8AAB	02013	342D
582A	342E	8AAC	02014	342E
59E6	342F	8AAD	02015	342F
5B8C	3430	8AAE	02016	3430
5B98	3431	8AAF	02017	3431
5BDB	3432	8AB0	02018	3432
5E72	3433	8AB1	02019	3433
5E79	3434	8AB2	02020	3434
60A3	3435	8AB3	02021	3435
611F	3436	8AB4	02022	3436
6163	3437	8AB5	02023	3437
61BE	3438	8AB6	02024	3438
63DB	3439	8AB7	02025	3439
6562	343A	8AB8	02026	343A
67D1	343B	8AB9	02027	343B
6853	343C	8ABA	02028	343C
68FA	343D	8ABB	02029	343D
6B3E	343E	8ABC	02030	343E
6B53	343F	8ABD	02031	343F
6C57	3440	8ABE	02032	3440
6F22	3441	8ABF	02033	3441
E26B				3442
704C	5E75	9FF3	06285	3443
74B0	3444	8AC2	02036	3444
7518	3445	8AC3	02037	3445
76E3	3446	8AC4	02038	3446
770B	3447	8AC5	02039	3447
7AFF	3448	8AC6	02040	3448
7BA1	3449	8AC7	02041	3449
7C21	344A	8AC8	02042	344A
7DE9	344B	8AC9	02043	344B
7F36	344C	8ACA	02044	344C
E26C				344D
809D	344E	8ACC	02046	344E
8266	344F	8ACD	02047	344F
839E	3450	8ACE	02048	3450
89B3	3451	8ACF	02049	3451
8AEB	6B5D	E67C	07561	3452
8CAB	3453	8AD1	02051	3453
9084	3454	8AD2	02052	3454
9451	3455	8AD3	02053	3455
9593	3456	8AD4	02054	3456
9591	3457	8AD5	02055	3457
95A2	3458	8AD6	02056	3458
9665	3459	8AD7	02057	3459
97D3	345A	8AD8	02058	345A
9928	345B	8AD9	02059	345B
8218	345C	8ADA	02060	345C
4E38	345D	8ADB	02061	345D
542B	345E	8ADC	02062	345E
5CB8	345F	8ADD	02063	345F
5DCC	3460	8ADE	02064	3460
73A9	3461	8ADF	02065	3461
764C	3462	8AE0	02066	3462
773C	3463	8AE1	02067	3463
5CA9	3464	8AE2	02068	3464
E26D				3465
8D0B	3466	8AE4	02070	3466
96C1	3467	8AE5	02071	3467
E26E				3468
9854	3469	8AE7	02073	3469
9858	346A	8AE8	02074	346A
4F01	346B	8AE9	02075	346B
4F0E	346C	8AEA	02076	346C
5371	346D	8AEB	02077	346D
559C	346E	8AEC	02078	346E
5668	346F	8AED	02079	346F
57FA	3470	8AEE	02080	3470
5947	3471	8AEF	02081	3471
5B09	3472	8AF0	02082	3472
5BC4	3473	8AF1	02083	3473
5C90	3474	8AF2	02084	3474
5E0C	3475	8AF3	02085	3475
5E7E	3476	8AF4	02086	3476
5FCC	3477	8AF5	02087	3477
63EE	3478	8AF6	02088	3478
673A	3479	8AF7	02089	3479
65D7	347A	8AF8	02090	347A
65E2	347B	8AF9	02091	347B
671F	347C	8AFA	02092	347C
68CB	347D	8AFB	02093	347D
68C4	347E	8AFC	02094	347E
6A5F	3521	8B40	02101	3521
E26F				3522
6BC5	3523	8B42	02103	3523
6C17	3524	8B43	02104	3524
6C7D	3525	8B44	02105	3525
757F	3526	8B45	02106	3526
7948	3527	8B46	02107	3527
5B63	3528	8B47	02108	3528
7A00	3529	8B48	02109	3529
7D00	352A	8B49	02110	352A
E270				352B
898F	352C	8B4B	02112	352C
8A18	352D	8B4C	02113	352D
8CB4	352E	8B4D	02114	352E
8D77	352F	8B4E	02115	352F
8ECC	3530	8B4F	02116	3530
8F1D	3531	8B50	02117	3531
98E2	3532	8B51	02118	3532
9A0E	3533	8B52	02119	3533
9B3C	3534	8B53	02120	3534
4E80	3535	8B54	02121	3535
507D	3536	8B55	02122	3536
5100	3537	8B56	02123	3537
5993	3538	8B57	02124	3538
5B9C	3539	8B58	02125	3539
622F	353A	8B59	02126	353A
6280	353B	8B5A	02127	353B
64EC	353C	8B5B	02128	353C
6B3A	353D	8B5C	02129	353D
72A0	353E	8B5D	02130	353E
7591	353F	8B5E	02131	353F
E271				3540
7FA9	3541	8B60	02133	3541
87FB	3542	8B61	02134	3542
8ABC	3543	8B62	02135	3543
8B70	3544	8B63	02136	3544
63AC	3545	8B64	02137	3545
83CA	3546	8B65	02138	3546
97A0	3547	8B66	02139	3547
5409	3548	8B67	02140	3548
5403	3549	8B68	02141	3549
55AB	354A	8B69	02142	354A
6854	354B	8B6A	02143	354B
6A58	354C	8B6B	02144	354C
8A70	354D	8B6C	02145	354D
7827	354E	8B6D	02146	354E
6775	354F	8B6E	02147	354F
9ECD	3550	8B6F	02148	3550
5374	3551	8B70	02149	3551
5BA2	3552	8B71	02150	3552
811A	3553	8B72	02151	3553
8650	3554	8B73	02152	3554
9006	3555	8B74	02153	3555
4E18	3556	8B75	02154	3556
E272				3557
4EC7	3558	8B77	02156	3558
4F11	3559	8B78	02157	3559
53CA	355A	8B79	02158	355A
5438	355B	8B7A	02159	355B
5BAE	355C	8B7B	02160	355C
5F13	355D	8B7C	02161	355D
6025	355E	8B7D	02162	355E
6551	355F	8B7E	02163	355F
673D	3560	8B80	02164	3560
6C42	3561	8B81	02165	3561
6C72	3562	8B82	02166	3562
6CE3	3563	8B83	02167	3563
7078	3564	8B84	02168	3564
7403	3565	8B85	02169	3565
7A76	3566	8B86	02170	3566
7AAE	3567	8B87	02171	3567
7B08	3568	8B88	02172	3568
7D1A	3569	8B89	02173	3569
7CFE	356A	8B8A	02174	356A
7D66	356B	8B8B	02175	356B
65E7	356C	8B8C	02176	356C
725B	356D	8B8D	02177	356D
53BB	356E	8B8E	02178	356E
5C45	356F	8B8F	02179	356F
5DE8	3570	8B90	02180	3570
62D2	3571	8B91	02181	3571
62E0	3572	8B92	02182	3572
6319	3573	8B93	02183	3573
6E20	3574	8B94	02184	3574
865A	3575	8B95	02185	3575
8A31	3576	8B96	02186	3576
8DDD	3577	8B97	02187	3577
92F8	3578	8B98	02188	3578
6F01	3579	8B99	02189	3579
79A6	357A	8B9A	02190	357A
9B5A	357B	8B9B	02191	357B
4EA8	357C	8B9C	02192	357C
4EAB	357D	8B9D	02193	357D
4EAC	357E	8B9E	02194	357E
4F9B	3621	8B9F	02201	3621
E273				3622
50D1	3623	8BA1	02203	3623
5147	3624	8BA2	02204	3624
7AF6	3625	8BA3	02205	3625
5171	3626	8BA4	02206	3626
51F6	3627	8BA5	02207	3627
5354	3628	8BA6	02208	3628
5321	3629	8BA7	02209	3629
E274				362A
53EB	362B	8BA9	02211	362B
55AC	362C	8BAA	02212	362C
5883	362D	8BAB	02213	362D
5CE1	362E	8BAC	02214	362E
5F37	362F	8BAD	02215	362F
5F4A	3630	8BAE	02216	3630
602F	3631	8BAF	02217	3631
6050	3632	8BB0	02218	3632
606D	3633	8BB1	02219	3633
631F	3634	8BB2	02220	3634
6559	3635	8BB3	02221	3635
6A4B	3636	8BB4	02222	3636
6CC1	3637	8BB5	02223	3637
72C2	3638	8BB6	02224	3638
72ED	3639	8BB7	02225	3639
77EF	363A	8BB8	02226	363A
80F8	363B	8BB9	02227	363B
8105	363C	8BBA	02228	363C
8208	363D	8BBB	02229	363D
854E	363E	8BBC	02230	363E
90F7	363F	8BBD	02231	363F
93E1	3640	8BBE	02232	3640
97FF	3641	8BBF	02233	3641
E275				3642
9A5A	3643	8BC1	02235	3643
4EF0	3644	8BC2	02236	3644
51DD	3645	8BC3	02237	3645
582F	7421	EA9F	08401	3646
6681	3647	8BC5	02239	3647
696D	3648	8BC6	02240	3648
5C40	3649	8BC7	02241	3649
66F2	364A	8BC8	02242	364A
E276				364B
7389	364C	8BCA	02244	364C
6850	364D	8BCB	02245	364D
7C81	364E	8BCC	02246	364E
E277				364F
52E4	3650	8BCE	02248	3650
E278				3651
5DFE	3652	8BD0	02250	3652
9326	3653	8BD1	02251	3653
65A4	3654	8BD2	02252	3654
6B23	3655	8BD3	02253	3655
6B3D	3656	8BD4	02254	3656
7434	3657	8BD5	02255	3657
7981	3658	8BD6	02256	3658
79BD	3659	8BD7	02257	3659
7B4B	365A	8BD8	02258	365A
7DCA	365B	8BD9	02259	365B
82B9	365C	8BDA	02260	365C
E279				365D
887F	365E	8BDC	02262	365E
895F	365F	8BDD	02263	365F
8B39	3660	8BDE	02264	3660
8FD1	3661	8BDF	02265	3661
91D1	3662	8BE0	02266	3662
541F	3663	8BE1	02267	3663
9280	3664	8BE2	02268	3664
4E5D	3665	8BE3	02269	3665
5036	3666	8BE4	02270	3666
53E5	3667	8BE5	02271	3667
533A	3668	8BE6	02272	3668
72D7	3669	8BE7	02273	3669
7396	366A	8BE8	02274	366A
77E9	366B	8BE9	02275	366B
82E6	366C	8BEA	02276	366C
E27A				366D
99C6	366E	8BEC	02278	366E
99C8	366F	8BED	02279	366F
99D2	3670	8BEE	02280	3670
5177	3671	8BEF	02281	3671
611A	3672	8BF0	02282	3672
865E	3673	8BF1	02283	3673
E27B				3674
7A7A	3675	8BF3	02285	3675
5076	3676	8BF4	02286	3676
5BD3	3677	8BF5	02287	3677
9047	3678	8BF6	02288	3678
9685	3679	8BF7	02289	3679
4E32	367A	8BF8	02290	367A
E27C				367B
91E7	367C	8BFA	02292	367C
E27D				367D
5C48	367E	8BFC	02294	367E
6398	3721	8C40	02301	3721
7A9F	3722	8C41	02302	3722
6C93	3723	8C42	02303	3723
E27E				3724
8F61	3725	8C44	02305	3725
7AAA	3726	8C45	02306	3726
718A	3727	8C46	02307	3727
9688	3728	8C47	02308	3728
7C82	3729	8C48	02309	3729
6817	372A	8C49	02310	372A
7E70	372B	8C4A	02311	372B
6851	372C	8C4B	02312	372C
936C	372D	8C4C	02313	372D
52F2	372E	8C4D	02314	372E
541B	372F	8C4E	02315	372F
85AB	3730	8C4F	02316	3730
8A13	3731	8C50	02317	3731
7FA4	3732	8C51	02318	3732
8ECD	3733	8C52	02319	3733
90E1	3734	8C53	02320	3734
5366	3735	8C54	02321	3735
8888	3736	8C55	02322	3736
E27F				3737
4FC2	3738	8C57	02324	3738
50BE	3739	8C58	02325	3739
5211	373A	8C59	02326	373A
5144	373B	8C5A	02327	373B
5553	373C	8C5B	02328	373C
572D	373D	8C5C	02329	373D
73EA	373E	8C5D	02330	373E
578B	373F	8C5E	02331	373F
5951	3740	8C5F	02332	3740
5F62	3741	8C60	02333	3741
5F84	3742	8C61	02334	3742
6075	3743	8C62	02335	3743
6176	3744	8C63	02336	3744
E280				3745
61A9	3746	8C65	02338	3746
63B2	3747	8C66	02339	3747
643A	3748	8C67	02340	3748
656C	3749	8C68	02341	3749
666F	374A	8C69	02342	374A
6842	374B	8C6A	02343	374B
6E13	374C	8C6B	02344	374C
7566	374D	8C6C	02345	374D
E281				374E
7CFB	374F	8C6E	02347	374F
7D4C	3750	8C6F	02348	3750
7D99	3751	8C70	02349	3751
E282				3752
7F6B	3753	8C72	02351	3753
830E	3754	8C73	02352	3754
E283				3755
86CD	3756	8C75	02354	3756
8A08	3757	8C76	02355	3757
8A63	3758	8C77	02356	3758
8B66	3759	8C78	02357	3759
8EFD	375A	8C79	02358	375A
9838	7074	E8F2	08084	375B
9D8F	375C	8C7B	02360	375C
82B8	375D	8C7C	02361	375D
8FCE	375E	8C7D	02362	375E
9BE8	375F	8C7E	02363	375F
5287	3760	8C80	02364	3760
621F	3761	8C81	02365	3761
6483	3762	8C82	02366	3762
6FC0	3763	8C83	02367	3763
E284				3764
6841	3765	8C85	02369	3765
5091	3766	8C86	02370	3766
6B20	3767	8C87	02371	3767
6C7A	3768	8C88	02372	3768
6F54	3769	8C89	02373	3769
E285				376A
7D50	376B	8C8B	02375	376B
8840	376C	8C8C	02376	376C
8A23	376D	8C8D	02377	376D
6708	376E	8C8E	02378	376E
4EF6	376F	8C8F	02379	376F
5039	3770	8C90	02380	3770
E286				3771
5065	3772	8C92	02382	3772
517C	3773	8C93	02383	3773
5238	3774	8C94	02384	3774
5263	3775	8C95	02385	3775
55A7	3776	8C96	02386	3776
570F	3777	8C97	02387	3777
5805	3778	8C98	02388	3778
E287				3779
5EFA	377A	8C9A	02390	377A
61B2	377B	8C9B	02391	377B
61F8	377C	8C9C	02392	377C
62F3	377D	8C9D	02393	377D
E288				377E
691C	3821	8C9F	02401	3821
6A29	3822	8CA0	02402	3822
727D	3823	8CA1	02403	3823
72AC	3824	8CA2	02404	3824
732E	3825	8CA3	02405	3825
7814	3826	8CA4	02406	3826
786F	3827	8CA5	02407	3827
7D79	3828	8CA6	02408	3828
770C	3829	8CA7	02409	3829
80A9	382A	8CA8	02410	382A
898B	382B	8CA9	02411	382B
8B19	382C	8CAA	02412	382C
8CE2	382D	8CAB	02413	382D
8ED2	382E	8CAC	02414	382E
9063	382F	8CAD	02415	382F
E289				3830
967A	3831	8CAF	02417	3831
9855	3832	8CB0	02418	3832
9A13	3833	8CB1	02419	3833
9E7C				3834
5143	3835	8CB3	02421	3835
539F	3836	8CB4	02422	3836
53B3	3837	8CB5	02423	3837
5E7B	3838	8CB6	02424	3838
5F26	3839	8CB7	02425	3839
6E1B	383A	8CB8	02426	383A
6E90	383B	8CB9	02427	383B
7384	383C	8CBA	02428	383C
73FE	383D	8CBB	02429	383D
7D43	383E	8CBC	02430	383E
8237	383F	8CBD	02431	383F
8A00	3840	8CBE	02432	3840
E28A				3841
9650	3842	8CC0	02434	3842
4E4E	3843	8CC1	02435	3843
500B	3844	8CC2	02436	3844
53E4	3845	8CC3	02437	3845
547C	3846	8CC4	02438	3846
56FA	3847	8CC5	02439	3847
59D1	3848	8CC6	02440	3848
5B64	3849	8CC7	02441	3849
5DF1	384A	8CC8	02442	384A
5EAB	384B	8CC9	02443	384B
5F27	384C	8CCA	02444	384C
6238	384D	8CCB	02445	384D
6545	384E	8CCC	02446	384E
67AF	384F	8CCD	02447	384F
6E56	3850	8CCE	02448	3850
72D0	3851	8CCF	02449	3851
7CCA	3852	8CD0	02450	3852
88B4	3853	8CD1	02451	3853
80A1	3854	8CD2	02452	3854
80E1	3855	8CD3	02453	3855
83F0	3856	8CD4	02454	3856
864E	3857	8CD5	02455	3857
8A87	3858	8CD6	02456	3858
8DE8	3859	8CD7	02457	3859
9237	385A	8CD8	02458	385A
96C7	385B	8CD9	02459	385B
9867	385C	8CDA	02460	385C
9F13	385D	8CDB	02461	385D
4E94	385E	8CDC	02462	385E
4E92	385F	8CDD	02463	385F
4F0D	3860	8CDE	02464	3860
5348	3861	8CDF	02465	3861
5449	3862	8CE0	02466	3862
543E	3863	8CE1	02467	3863
5A2F	3864	8CE2	02468	3864
5F8C	3865	8CE3	02469	3865
5FA1	3866	8CE4	02470	3866
609F	3867	8CE5	02471	3867
68A7	3868	8CE6	02472	3868
6A8E	3869	8CE7	02473	3869
745A	386A	8CE8	02474	386A
7881	386B	8CE9	02475	386B
8A9E	386C	8CEA	02476	386C
8AA4	386D	8CEB	02477	386D
8B77	386E	8CEC	02478	386E
9190	386F	8CED	02479	386F
4E5E	3870	8CEE	02480	3870
9BC9	3871	8CEF	02481	3871
4EA4	3872	8CF0	02482	3872
E28B				3873
4FAF	3874	8CF2	02484	3874
5019	3875	8CF3	02485	3875
5016	3876	8CF4	02486	3876
5149	3877	8CF5	02487	3877
E28C				3878
529F	3879	8CF7	02489	3879
52B9	387A	8CF8	02490	387A
52FE	387B	8CF9	02491	387B
539A	387C	8CFA	02492	387C
53E3	387D	8CFB	02493	387D
5411	387E	8CFC	02494	387E
540E	3921	8D40	02501	3921
5589	3922	8D41	02502	3922
5751	3923	8D42	02503	3923
57A2	3924	8D43	02504	3924
597D	3925	8D44	02505	3925
5B54	3926	8D45	02506	3926
5B5D	3927	8D46	02507	3927
5B8F	3928	8D47	02508	3928
5DE5	3929	8D48	02509	3929
5DE7	392A	8D49	02510	392A
E28D				392B
5E78	392C	8D4B	02512	392C
5E83	392D	8D4C	02513	392D
5E9A	392E	8D4D	02514	392E
5EB7	392F	8D4E	02515	392F
5F18	3930	8D4F	02516	3930
6052	3931	8D50	02517	3931
614C	3932	8D51	02518	3932
6297	3933	8D52	02519	3933
62D8	3934	8D53	02520	3934
63A7	3935	8D54	02521	3935
653B	3936	8D55	02522	3936
E28E				3937
6643	3938	8D57	02524	3938
66F4	3939	8D58	02525	3939
676D	393A	8D59	02526	393A
6821	393B	8D5A	02527	393B
E28F				393C
69CB	393D	8D5C	02529	393D
6C5F	393E	8D5D	02530	393E
6D2A	393F	8D5E	02531	393F
6D69	3940	8D5F	02532	3940
6E2F	3941	8D60	02533	3941
E290				3942
7532	3943	8D62	02535	3943
7687	3944	8D63	02536	3944
786C	3945	8D64	02537	3945
7A3F	3946	8D65	02538	3946
7CE0	3947	8D66	02539	3947
7D05	3948	8D67	02540	3948
7D18	3949	8D68	02541	3949
7D5E	394A	8D69	02542	394A
7DB1	394B	8D6A	02543	394B
8015	394C	8D6B	02544	394C
E291				394D
80AF	394E	8D6D	02546	394E
80B1	394F	8D6E	02547	394F
8154	3950	8D6F	02548	3950
818F	3951	8D70	02549	3951
822A	3952	8D71	02550	3952
8352	3953	8D72	02551	3953
884C	3954	8D73	02552	3954
8861	3955	8D74	02553	3955
8B1B	3956	8D75	02554	3956
8CA2	3957	8D76	02555	3957
8CFC	3958	8D77	02556	3958
90CA	3959	8D78	02557	3959
9175	395A	8D79	02558	395A
9271	395B	8D7A	02559	395B
7926	6268	E1E6	06672	395C
92FC	395D	8D7C	02561	395D
95A4	395E	8D7D	02562	395E
964D	395F	8D7E	02563	395F
9805	3960	8D80	02564	3960
9999	3961	8D81	02565	3961
9AD8	3962	8D82	02566	3962
9D3B	3963	8D83	02567	3963
525B	3964	8D84	02568	3964
52AB	3965	8D85	02569	3965
53F7	3966	8D86	02570	3966
5408	3967	8D87	02571	3967
E292				3968
E293				3969
E294				396A
E295				396B
8F5F	396C	8D8C	02576	396C
9EB4				396D
514B	396E	8D8E	02578	396E
523B	396F	8D8F	02579	396F
544A	3970	8D90	02580	3970
56FD	3971	8D91	02581	3971
7A40	3972	8D92	02582	3972
9177	3973	8D93	02583	3973
E296				3974
9ED2	3975	8D95	02585	3975
7344	3976	8D96	02586	3976
6F09	3977	8D97	02587	3977
8170	3978	8D98	02588	3978
E297				3979
5FFD	397A	8D9A	02590	397A
60DA	397B	8D9B	02591	397B
9AA8	397C	8D9C	02592	397C
72DB	397D	8D9D	02593	397D
8FBC	397E	8D9E	02594	397E
6B64	3A21	8D9F	02601	3A21
9803	3A22	8DA0	02602	3A22
4ECA	3A23	8DA1	02603	3A23
56F0	3A24	8DA2	02604	3A24
5764	3A25	8DA3	02605	3A25
58BE	3A26	8DA4	02606	3A26
5A5A	3A27	8DA5	02607	3A27
6068	3A28	8DA6	02608	3A28
61C7	3A29	8DA7	02609	3A29
660F	3A2A	8DA8	02610	3A2A
6606	3A2B	8DA9	02611	3A2B
6839	3A2C	8DAA	02612	3A2C
68B1	3A2D	8DAB	02613	3A2D
6DF7	3A2E	8DAC	02614	3A2E
75D5	3A2F	8DAD	02615	3A2F
7D3A	3A30	8DAE	02616	3A30
826E	3A31	8DAF	02617	3A31
9B42	3A32	8DB0	02618	3A32
4E9B	3A33	8DB1	02619	3A33
4F50	3A34	8DB2	02620	3A34
E298				3A35
5506	3A36	8DB4	02622	3A36
5D6F	3A37	8DB5	02623	3A37
5DE6	3A38	8DB6	02624	3A38
5DEE	3A39	8DB7	02625	3A39
67FB	3A3A	8DB8	02626	3A3A
6C99	3A3B	8DB9	02627	3A3B
7473	3A3C	8DBA	02628	3A3C
7802	3A3D	8DBB	02629	3A3D
8A50	3A3E	8DBC	02630	3A3E
9396	3A3F	8DBD	02631	3A3F
88DF	3A40	8DBE	02632	3A40
5750	3A41	8DBF	02633	3A41
5EA7	3A42	8DC0	02634	3A42
632B	3A43	8DC1	02635	3A43
50B5	3A44	8DC2	02636	3A44
50AC	3A45	8DC3	02637	3A45
518D	3A46	8DC4	02638	3A46
6700	3A47	8DC5	02639	3A47
54C9	3A48	8DC6	02640	3A48
585E	3A49	8DC7	02641	3A49
59BB	3A4A	8DC8	02642	3A4A
5BB0	3A4B	8DC9	02643	3A4B
5F69	3A4C	8DCA	02644	3A4C
624D	3A4D	8DCB	02645	3A4D
63A1	3A4E	8DCC	02646	3A4E
683D	3A4F	8DCD	02647	3A4F
6B73	3A50	8DCE	02648	3A50
6E08	3A51	8DCF	02649	3A51
707D	3A52	8DD0	02650	3A52
E299				3A53
7280	3A54	8DD2	02652	3A54
7815	3A55	8DD3	02653	3A55
7826	3A56	8DD4	02654	3A56
796D	3A57	8DD5	02655	3A57
658E	3A58	8DD6	02656	3A58
7D30	3A59	8DD7	02657	3A59
83DC	3A5A	8DD8	02658	3A5A
88C1	3A5B	8DD9	02659	3A5B
8F09	3A5C	8DDA	02660	3A5C
969B	3A5D	8DDB	02661	3A5D
5264	3A5E	8DDC	02662	3A5E
5728	3A5F	8DDD	02663	3A5F
6750	3A60	8DDE	02664	3A60
7F6A	3A61	8DDF	02665	3A61
8CA1	3A62	8DE0	02666	3A62
51B4	3A63	8DE1	02667	3A63
5742	3A64	8DE2	02668	3A64
962A	3A65	8DE3	02669	3A65
583A	3A66	8DE4	02670	3A66
E29A				3A67
80B4	3A68	8DE6	02672	3A68
54B2	3A69	8DE7	02673	3A69
5D0E	3A6A	8DE8	02674	3A6A
57FC	3A6B	8DE9	02675	3A6B
7895	3A6C	8DEA	02676	3A6C
9DFA	3A6D	8DEB	02677	3A6D
4F5C	3A6E	8DEC	02678	3A6E
524A	3A6F	8DED	02679	3A6F
548B	3A70	8DEE	02680	3A70
643E	3A71	8DEF	02681	3A71
6628	3A72	8DF0	02682	3A72
6714	3A73	8DF1	02683	3A73
E29B				3A74
7A84	3A75	8DF3	02685	3A75
7B56	3A76	8DF4	02686	3A76
7D22	3A77	8DF5	02687	3A77
932F	3A78	8DF6	02688	3A78
685C	3A79	8DF7	02689	3A79
9BAD	3A7A	8DF8	02690	3A7A
7B39	3A7B	8DF9	02691	3A7B
5319	3A7C	8DFA	02692	3A7C
518A	3A7D	8DFB	02693	3A7D
5237	3A7E	8DFC	02694	3A7E
5BDF	3B21	8E40	02701	3B21
62F6	3B22	8E41	02702	3B22
64AE	3B23	8E42	02703	3B23
64E6	3B24	8E43	02704	3B24
672D	3B25	8E44	02705	3B25
6BBA	3B26	8E45	02706	3B26
E29C				3B27
96D1	3B28	8E47	02708	3B28
7690	3B29	8E48	02709	3B29
E29D				3B2A
634C	3B2B	8E4A	02711	3B2B
E29E				3B2C
E29F				3B2D
76BF	3B2E	8E4D	02714	3B2E
6652	3B2F	8E4E	02715	3B2F
4E09	3B30	8E4F	02716	3B30
5098	3B31	8E50	02717	3B31
53C2	3B32	8E51	02718	3B32
5C71	3B33	8E52	02719	3B33
60E8	3B34	8E53	02720	3B34
6492	3B35	8E54	02721	3B35
6563	3B36	8E55	02722	3B36
685F	3B37	8E56	02723	3B37
71E6	3B38	8E57	02724	3B38
E2A0				3B39
7523	3B3A	8E59	02726	3B3A
7B97	3B3B	8E5A	02727	3B3B
7E82	3B3C	8E5B	02728	3B3C
8695	3B3D	8E5C	02729	3B3D
8B83	3B3E	8E5D	02730	3B3E
8CDB	3B3F	8E5E	02731	3B3F
9178	3B40	8E5F	02732	3B40
E2A1				3B41
65AC	3B42	8E61	02734	3B42
66AB	3B43	8E62	02735	3B43
6B8B	3B44	8E63	02736	3B44
4ED5	3B45	8E64	02737	3B45
4ED4	3B46	8E65	02738	3B46
4F3A	3B47	8E66	02739	3B47
4F7F	3B48	8E67	02740	3B48
523A	3B49	8E68	02741	3B49
53F8	3B4A	8E69	02742	3B4A
53F2	3B4B	8E6A	02743	3B4B
55E3	3B4C	8E6B	02744	3B4C
56DB	3B4D	8E6C	02745	3B4D
58EB	3B4E	8E6D	02746	3B4E
59CB	3B4F	8E6E	02747	3B4F
59C9	3B50	8E6F	02748	3B50
59FF	3B51	8E70	02749	3B51
5B50	3B52	8E71	02750	3B52
5C4D	3B53	8E72	02751	3B53
5E02	3B54	8E73	02752	3B54
5E2B	3B55	8E74	02753	3B55
5FD7	3B56	8E75	02754	3B56
601D	3B57	8E76	02755	3B57
6307	3B58	8E77	02756	3B58
652F	3B59	8E78	02757	3B59
5B5C	3B5A	8E79	02758	3B5A
65AF	3B5B	8E7A	02759	3B5B
65BD	3B5C	8E7B	02760	3B5C
65E8	3B5D	8E7C	02761	3B5D
679D	3B5E	8E7D	02762	3B5E
6B62	3B5F	8E7E	02763	3B5F
6B7B	3B60	8E80	02764	3B60
6C0F	3B61	8E81	02765	3B61
7345	3B62	8E82	02766	3B62
7949	3B63	8E83	02767	3B63
79C1	3B64	8E84	02768	3B64
7CF8	3B65	8E85	02769	3B65
7D19	3B66	8E86	02770	3B66
7D2B	3B67	8E87	02771	3B67
80A2	3B68	8E88	02772	3B68
8102	3B69	8E89	02773	3B69
81F3	3B6A	8E8A	02774	3B6A
8996	3B6B	8E8B	02775	3B6B
8A5E	3B6C	8E8C	02776	3B6C
8A69	3B6D	8E8D	02777	3B6D
8A66	3B6E	8E8E	02778	3B6E
8A8C	3B6F	8E8F	02779	3B6F
8AEE	3B70	8E90	02780	3B70
8CC7	3B71	8E91	02781	3B71
8CDC	3B72	8E92	02782	3B72
96CC	3B73	8E93	02783	3B73
98FC	3B74	8E94	02784	3B74
6B6F	3B75	8E95	02785	3B75
4E8B	3B76	8E96	02786	3B76
4F3C	3B77	8E97	02787	3B77
4F8D	3B78	8E98	02788	3B78
5150	3B79	8E99	02789	3B79
5B57	3B7A	8E9A	02790	3B7A
5BFA	3B7B	8E9B	02791	3B7B
6148	3B7C	8E9C	02792	3B7C
6301	3B7D	8E9D	02793	3B7D
6642	3B7E	8E9E	02794	3B7E
6B21	3C21	8E9F	02801	3C21
6ECB	3C22	8EA0	02802	3C22
6CBB	3C23	8EA1	02803	3C23
723E	3C24	8EA2	02804	3C24
74BD	3C25	8EA3	02805	3C25
75D4	3C26	8EA4	02806	3C26
78C1	3C27	8EA5	02807	3C27
793A	3C28	8EA6	02808	3C28
800C	3C29	8EA7	02809	3C29
8033	3C2A	8EA8	02810	3C2A
81EA	3C2B	8EA9	02811	3C2B
8494	3C2C	8EAA	02812	3C2C
8F9E	3C2D	8EAB	02813	3C2D
6C50	3C2E	8EAC	02814	3C2E
9E7F	3C2F	8EAD	02815	3C2F
5F0F	3C30	8EAE	02816	3C30
8B58	3C31	8EAF	02817	3C31
9D2B	3C32	8EB0	02818	3C32
7AFA	3C33	8EB1	02819	3C33
8EF8	3C34	8EB2	02820	3C34
5B8D	3C35	8EB3	02821	3C35
96EB	3C36	8EB4	02822	3C36
4E03	3C37	8EB5	02823	3C37
E2A2				3C38
57F7	3C39	8EB7	02825	3C39
5931	3C3A	8EB8	02826	3C3A
5AC9	3C3B	8EB9	02827	3C3B
5BA4	3C3C	8EBA	02828	3C3C
6089	3C3D	8EBB	02829	3C3D
6E7F	3C3E	8EBC	02830	3C3E
E2A3				3C3F
75BE	3C40	8EBE	02832	3C40
8CEA	3C41	8EBF	02833	3C41
5B9F	3C42	8EC0	02834	3C42
8500	3C43	8EC1	02835	3C43
7BE0	3C44	8EC2	02836	3C44
5072	3C45	8EC3	02837	3C45
67F4	3C46	8EC4	02838	3C46
829D	3C47	8EC5	02839	3C47
5C62				3C48
8602	6922	E541	07302	3C49
7E1E	3C4A	8EC8	02842	3C4A
820E	3C4B	8EC9	02843	3C4B
5199	3C4C	8ECA	02844	3C4C
5C04	3C4D	8ECB	02845	3C4D
6368	3C4E	8ECC	02846	3C4E
8D66	3C4F	8ECD	02847	3C4F
659C	3C50	8ECE	02848	3C50
716E	3C51	8ECF	02849	3C51
793E	3C52	8ED0	02850	3C52
7D17	3C53	8ED1	02851	3C53
8005	3C54	8ED2	02852	3C54
8B1D	3C55	8ED3	02853	3C55
8ECA	3C56	8ED4	02854	3C56
E2A4				3C57
86C7	3C58	8ED6	02856	3C58
90AA	3C59	8ED7	02857	3C59
501F	3C5A	8ED8	02858	3C5A
52FA	3C5B	8ED9	02859	3C5B
5C3A	3C5C	8EDA	02860	3C5C
E2A5				3C5D
E2A6				3C5E
7235	3C5F	8EDD	02863	3C5F
914C	3C60	8EDE	02864	3C60
91C8	3C61	8EDF	02865	3C61
932B	3C62	8EE0	02866	3C62
82E5	3C63	8EE1	02867	3C63
5BC2	3C64	8EE2	02868	3C64
5F31	3C65	8EE3	02869	3C65
60F9	3C66	8EE4	02870	3C66
4E3B	3C67	8EE5	02871	3C67
53D6	3C68	8EE6	02872	3C68
5B88	3C69	8EE7	02873	3C69
624B	3C6A	8EE8	02874	3C6A
6731	3C6B	8EE9	02875	3C6B
6B8A	3C6C	8EEA	02876	3C6C
72E9	3C6D	8EEB	02877	3C6D
73E0	3C6E	8EEC	02878	3C6E
7A2E	3C6F	8EED	02879	3C6F
816B	3C70	8EEE	02880	3C70
8DA3	3C71	8EEF	02881	3C71
9152	3C72	8EF0	02882	3C72
9996	3C73	8EF1	02883	3C73
5112	3C74	8EF2	02884	3C74
53D7	3C75	8EF3	02885	3C75
546A	3C76	8EF4	02886	3C76
5BFF	3C77	8EF5	02887	3C77
6388	3C78	8EF6	02888	3C78
6A39	3C79	8EF7	02889	3C79
7DAC	3C7A	8EF8	02890	3C7A
9700	3C7B	8EF9	02891	3C7B
56DA	3C7C	8EFA	02892	3C7C
53CE	3C7D	8EFB	02893	3C7D
5468	3C7E	8EFC	02894	3C7E
5B97	3D21	8F40	02901	3D21
5C31	3D22	8F41	02902	3D22
5DDE	3D23	8F42	02903	3D23
4FEE	3D24	8F43	02904	3D24
6101	3D25	8F44	02905	3D25
62FE	3D26	8F45	02906	3D26
6D32	3D27	8F46	02907	3D27
79C0	3D28	8F47	02908	3D28
79CB	3D29	8F48	02909	3D29
7D42	3D2A	8F49	02910	3D2A
7E61				3D2B
7FD2	3D2C	8F4B	02912	3D2C
81ED	3D2D	8F4C	02913	3D2D
821F	3D2E	8F4D	02914	3D2E
8490	3D2F	8F4E	02915	3D2F
8846	3D30	8F4F	02916	3D30
8972	3D31	8F50	02917	3D31
8B90	3D32	8F51	02918	3D32
8E74	3D33	8F52	02919	3D33
8F2F	3D34	8F53	02920	3D34
9031	3D35	8F54	02921	3D35
E2A7				3D36
916C	3D37	8F56	02923	3D37
96C6	3D38	8F57	02924	3D38
919C	3D39	8F58	02925	3D39
4EC0	3D3A	8F59	02926	3D3A
4F4F	3D3B	8F5A	02927	3D3B
5145	3D3C	8F5B	02928	3D3C
5341	3D3D	8F5C	02929	3D3D
5F93	3D3E	8F5D	02930	3D3E
620E	3D3F	8F5E	02931	3D3F
67D4	3D40	8F5F	02932	3D40
6C41	3D41	8F60	02933	3D41
6E0B	3D42	8F61	02934	3D42
7363	3D43	8F62	02935	3D43
7E26	3D44	8F63	02936	3D44
91CD	3D45	8F64	02937	3D45
9283	3D46	8F65	02938	3D46
53D4	3D47	8F66	02939	3D47
5919	3D48	8F67	02940	3D48
5BBF	3D49	8F68	02941	3D49
6DD1	3D4A	8F69	02942	3D4A
795D	3D4B	8F6A	02943	3D4B
7E2E	3D4C	8F6B	02944	3D4C
7C9B	3D4D	8F6C	02945	3D4D
587E	3D4E	8F6D	02946	3D4E
719F	3D4F	8F6E	02947	3D4F
51FA	3D50	8F6F	02948	3D50
8853	3D51	8F70	02949	3D51
8FF0	3D52	8F71	02950	3D52
4FCA	3D53	8F72	02951	3D53
5CFB	3D54	8F73	02952	3D54
6625	3D55	8F74	02953	3D55
77AC	3D56	8F75	02954	3D56
7AE3	3D57	8F76	02955	3D57
821C	3D58	8F77	02956	3D58
99FF	3D59	8F78	02957	3D59
51C6	3D5A	8F79	02958	3D5A
E2A8				3D5B
65EC	3D5C	8F7B	02960	3D5C
E2A9				3D5D
6B89	3D5E	8F7D	02962	3D5E
6DF3	3D5F	8F7E	02963	3D5F
6E96	3D60	8F80	02964	3D60
6F64	3D61	8F81	02965	3D61
E2AA				3D62
7D14	3D63	8F83	02967	3D63
5DE1	3D64	8F84	02968	3D64
9075	3D65	8F85	02969	3D65
9187	3D66	8F86	02970	3D66
9806	3D67	8F87	02971	3D67
51E6	3D68	8F88	02972	3D68
521D	3D69	8F89	02973	3D69
6240	3D6A	8F8A	02974	3D6A
6691	3D6B	8F8B	02975	3D6B
E2AB				3D6C
E2AC				3D6D
5EB6	3D6E	8F8E	02978	3D6E
7DD2	3D6F	8F8F	02979	3D6F
7F72	3D70	8F90	02980	3D70
66F8	3D71	8F91	02981	3D71
E2AD				3D72
E2AE				3D73
8AF8	3D74	8F94	02984	3D74
52A9	3D75	8F95	02985	3D75
53D9	3D76	8F96	02986	3D76
5973	3D77	8F97	02987	3D77
5E8F	3D78	8F98	02988	3D78
5F90	3D79	8F99	02989	3D79
6055	3D7A	8F9A	02990	3D7A
92E4	3D7B	8F9B	02991	3D7B
9664	3D7C	8F9C	02992	3D7C
50B7	3D7D	8F9D	02993	3D7D
511F	3D7E	8F9E	02994	3D7E
52DD	3E21	8F9F	03001	3E21
5320	3E22	8FA0	03002	3E22
5347	3E23	8FA1	03003	3E23
53EC	3E24	8FA2	03004	3E24
E2AF				3E25
5546	3E26	8FA4	03006	3E26
5531	3E27	8FA5	03007	3E27
5617	3E28	8FA6	03008	3E28
5968	3E29	8FA7	03009	3E29
59BE	3E2A	8FA8	03010	3E2A
5A3C	3E2B	8FA9	03011	3E2B
5BB5	3E2C	8FAA	03012	3E2C
5C06	3E2D	8FAB	03013	3E2D
5C0F	3E2E	8FAC	03014	3E2E
5C11	3E2F	8FAD	03015	3E2F
5C1A	3E30	8FAE	03016	3E30
5E84	3E31	8FAF	03017	3E31
5E8A	3E32	8FB0	03018	3E32
E2B0				3E33
5F70	3E34	8FB2	03020	3E34
627F	3E35	8FB3	03021	3E35
6284	3E36	8FB4	03022	3E36
62DB	3E37	8FB5	03023	3E37
638C	3E38	8FB6	03024	3E38
6377	3E39	8FB7	03025	3E39
6607	3E3A	8FB8	03026	3E3A
660C	3E3B	8FB9	03027	3E3B
662D	3E3C	8FBA	03028	3E3C
6676	3E3D	8FBB	03029	3E3D
E2B1				3E3E
E2B2				3E3F
6A1F	3E40	8FBE	03032	3E40
6A35	3E41	8FBF	03033	3E41
6CBC	3E42	8FC0	03034	3E42
6D88	3E43	8FC1	03035	3E43
6E09	3E44	8FC2	03036	3E44
6E58	3E45	8FC3	03037	3E45
713C	3E46	8FC4	03038	3E46
7126	3E47	8FC5	03039	3E47
7167	3E48	8FC6	03040	3E48
75C7	3E49	8FC7	03041	3E49
7701	3E4A	8FC8	03042	3E4A
785D	3E4B	8FC9	03043	3E4B
7901	3E4C	8FCA	03044	3E4C
7965	3E4D	8FCB	03045	3E4D
79F0	3E4E	8FCC	03046	3E4E
7AE0	3E4F	8FCD	03047	3E4F
7B11	3E50	8FCE	03048	3E50
7CA7	3E51	8FCF	03049	3E51
7D39	3E52	8FD0	03050	3E52
8096	3E53	8FD1	03051	3E53
83D6	3E54	8FD2	03052	3E54
8523				3E55
8549	3E56	8FD4	03054	3E56
885D	3E57	8FD5	03055	3E57
88F3	3E58	8FD6	03056	3E58
E2B3				3E59
8A3C	3E5A	8FD8	03058	3E5A
8A54	3E5B	8FD9	03059	3E5B
8A73	3E5C	8FDA	03060	3E5C
E2B4				3E5D
8CDE	3E5E	8FDC	03062	3E5E
91AC				3E5F
9266	3E60	8FDE	03064	3E60
937E	3E61	8FDF	03065	3E61
9418	3E62	8FE0	03066	3E62
969C	3E63	8FE1	03067	3E63
E2B5				3E64
4E0A	3E65	8FE3	03069	3E65
4E08	3E66	8FE4	03070	3E66
4E1E	3E67	8FE5	03071	3E67
4E57	3E68	8FE6	03072	3E68
5197	3E69	8FE7	03073	3E69
5270	3E6A	8FE8	03074	3E6A
57CE	3E6B	8FE9	03075	3E6B
5834	3E6C	8FEA	03076	3E6C
58CC	3E6D	8FEB	03077	3E6D
5B22	3E6E	8FEC	03078	3E6E
5E38	3E6F	8FED	03079	3E6F
60C5	3E70	8FEE	03080	3E70
64FE	3E71	8FEF	03081	3E71
6761	3E72	8FF0	03082	3E72
E2B6				3E73
6D44	3E74	8FF2	03084	3E74
72B6	3E75	8FF3	03085	3E75
7573	3E76	8FF4	03086	3E76
7A63	3E77	8FF5	03087	3E77
84B8	3E78	8FF6	03088	3E78
8B72	3E79	8FF7	03089	3E79
91B8	3E7A	8FF8	03090	3E7A
9320	3E7B	8FF9	03091	3E7B
5631	3E7C	8FFA	03092	3E7C
57F4	3E7D	8FFB	03093	3E7D
98FE	3E7E	8FFC	03094	3E7E
62ED	3F21	9040	03101	3F21
E2B7				3F22
E2B8				3F23
71ED	3F24	9043	03104	3F24
7E54	3F25	9044	03105	3F25
8077	3F26	9045	03106	3F26
8272	3F27	9046	03107	3F27
89E6	3F28	9047	03108	3F28
98DF	3F29	9048	03109	3F29
E2B9				3F2A
8FB1	3F2B	904A	03111	3F2B
5C3B	3F2C	904B	03112	3F2C
4F38	3F2D	904C	03113	3F2D
4FE1	3F2E	904D	03114	3F2E
4FB5	3F2F	904E	03115	3F2F
5507	3F30	904F	03116	3F30
5A20	3F31	9050	03117	3F31
5BDD	3F32	9051	03118	3F32
5BE9	3F33	9052	03119	3F33
5FC3	3F34	9053	03120	3F34
614E	3F35	9054	03121	3F35
632F	3F36	9055	03122	3F36
65B0	3F37	9056	03123	3F37
664B	3F38	9057	03124	3F38
68EE	3F39	9058	03125	3F39
699B	3F3A	9059	03126	3F3A
6D78	3F3B	905A	03127	3F3B
6DF1	3F3C	905B	03128	3F3C
7533	3F3D	905C	03129	3F3D
75B9	3F3E	905D	03130	3F3E
771F	3F3F	905E	03131	3F3F
795E	3F40	905F	03132	3F40
E2BA				3F41
7D33	3F42	9061	03134	3F42
81E3	3F43	9062	03135	3F43
82AF	3F44	9063	03136	3F44
85AA	3F45	9064	03137	3F45
89AA	3F46	9065	03138	3F46
8A3A	3F47	9066	03139	3F47
8EAB	3F48	9067	03140	3F48
8F9B	3F49	9068	03141	3F49
9032	3F4A	9069	03142	3F4A
91DD	3F4B	906A	03143	3F4B
9707	3F4C	906B	03144	3F4C
4EBA	3F4D	906C	03145	3F4D
4EC1	3F4E	906D	03146	3F4E
5203	3F4F	906E	03147	3F4F
5875	3F50	906F	03148	3F50
58EC	3F51	9070	03149	3F51
5C0B	3F52	9071	03150	3F52
751A	3F53	9072	03151	3F53
5C3D	3F54	9073	03152	3F54
814E	3F55	9074	03153	3F55
E2BB				3F56
8FC5	3F57	9076	03155	3F57
9663	3F58	9077	03156	3F58
E2BC				3F59
7B25	3F5A	9079	03158	3F5A
E2BD				3F5B
9808	3F5C	907B	03160	3F5C
9162	3F5D	907C	03161	3F5D
56F3	3F5E	907D	03162	3F5E
53A8	3F5F	907E	03163	3F5F
E2BE				3F60
5439	3F61	9081	03165	3F61
5782	3F62	9082	03166	3F62
5E25	3F63	9083	03167	3F63
63A8	3F64	9084	03168	3F64
6C34	3F65	9085	03169	3F65
708A	3F66	9086	03170	3F66
7761	3F67	9087	03171	3F67
7C8B	3F68	9088	03172	3F68
E2BF				3F69
8870	3F6A	908A	03174	3F6A
9042	3F6B	908B	03175	3F6B
9154	3F6C	908C	03176	3F6C
9310	3F6D	908D	03177	3F6D
9318	3F6E	908E	03178	3F6E
968F	3F6F	908F	03179	3F6F
745E	3F70	9090	03180	3F70
9AC4	3F71	9091	03181	3F71
5D07	3F72	9092	03182	3F72
5D69	3F73	9093	03183	3F73
6570	3F74	9094	03184	3F74
67A2	3F75	9095	03185	3F75
8DA8	3F76	9096	03186	3F76
96DB	3F77	9097	03187	3F77
636E	3F78	9098	03188	3F78
6749	3F79	9099	03189	3F79
6919	3F7A	909A	03190	3F7A
83C5	3F7B	909B	03191	3F7B
9817	3F7C	909C	03192	3F7C
96C0	3F7D	909D	03193	3F7D
88FE	3F7E	909E	03194	3F7E
6F84	4021	909F	03201	4021
E2C0				4022
5BF8	4023	90A1	03203	4023
4E16	4024	90A2	03204	4024
702C	4025	90A3	03205	4025
755D	4026	90A4	03206	4026
662F	4027	90A5	03207	4027
51C4	4028	90A6	03208	4028
5236	4029	90A7	03209	4029
52E2	402A	90A8	03210	402A
59D3	402B	90A9	03211	402B
5F81	402C	90AA	03212	402C
6027	402D	90AB	03213	402D
6210	402E	90AC	03214	402E
653F	402F	90AD	03215	402F
6574	4030	90AE	03216	4030
661F	4031	90AF	03217	4031
6674	4032	90B0	03218	4032
68F2	4033	90B1	03219	4033
6816	4034	90B2	03220	4034
6B63	4035	90B3	03221	4035
6E05	4036	90B4	03222	4036
7272	4037	90B5	03223	4037
751F	4038	90B6	03224	4038
76DB	4039	90B7	03225	4039
7CBE	403A	90B8	03226	403A
8056	403B	90B9	03227	403B
58F0	403C	90BA	03228	403C
88FD	403D	90BB	03229	403D
897F	403E	90BC	03230	403E
8AA0	403F	90BD	03231	403F
8A93	4040	90BE	03232	4040
8ACB	4041	90BF	03233	4041
E2C1				4042
9192	4043	90C1	03235	4043
9752	4044	90C2	03236	4044
9759	4045	90C3	03237	4045
6589	4046	90C4	03238	4046
7A0E	4047	90C5	03239	4047
8106	4048	90C6	03240	4048
96BB	4049	90C7	03241	4049
5E2D	404A	90C8	03242	404A
60DC	404B	90C9	03243	404B
621A	404C	90CA	03244	404C
E2C2				404D
6614	404E	90CC	03246	404E
6790	404F	90CD	03247	404F
77F3	4050	90CE	03248	4050
7A4D	4051	90CF	03249	4051
7C4D	4052	90D0	03250	4052
7E3E	4053	90D1	03251	4053
810A	4054	90D2	03252	4054
8CAC	4055	90D3	03253	4055
8D64	4056	90D4	03254	4056
8DE1	4057	90D5	03255	4057
8E5F	4058	90D6	03256	4058
78A9	4059	90D7	03257	4059
5207	405A	90D8	03258	405A
62D9	405B	90D9	03259	405B
63A5	405C	90DA	03260	405C
6442	405D	90DB	03261	405D
6298	405E	90DC	03262	405E
8A2D	405F	90DD	03263	405F
7A83	4060	90DE	03264	4060
7BC0	4061	90DF	03265	4061
8AAC	4062	90E0	03266	4062
96EA	4063	90E1	03267	4063
7D76	4064	90E2	03268	4064
820C	4065	90E3	03269	4065
E2C3				4066
4ED9	4067	90E5	03271	4067
5148	4068	90E6	03272	4068
5343	4069	90E7	03273	4069
5360	406A	90E8	03274	406A
5BA3	406B	90E9	03275	406B
5C02	406C	90EA	03276	406C
5C16	406D	90EB	03277	406D
5DDD	406E	90EC	03278	406E
6226	406F	90ED	03279	406F
6247	4070	90EE	03280	4070
E2C4				4071
6813	4072	90F0	03282	4072
E2C5				4073
6CC9	4074	90F2	03284	4074
6D45	4075	90F3	03285	4075
6D17	4076	90F4	03286	4076
67D3	4077	90F5	03287	4077
6F5C	4078	90F6	03288	4078
E2C6				4079
E2C7				407A
65CB	407B	90F9	03291	407B
7A7F	407C	90FA	03292	407C
7BAD	407D	90FB	03293	407D
7DDA	407E	90FC	03294	407E
7E4A	4121	9140	03301	4121
7FA8	4122	9141	03302	4122
817A	4123	9142	03303	4123
821B	4124	9143	03304	4124
E2C8				4125
85A6	4126	9145	03306	4126
8A6E	4127	9146	03307	4127
8CE4	6C4D	E6CB	07645	4128
8DF5	4129	9148	03309	4129
9078	412A	9149	03310	412A
9077	412B	914A	03311	412B
92AD	412C	914B	03312	412C
9291	412D	914C	03313	412D
9583	412E	914D	03314	412E
9BAE	412F	914E	03315	412F
524D	4130	914F	03316	4130
5584	4131	9150	03317	4131
6F38	4132	9151	03318	4132
7136	4133	9152	03319	4133
5168	4134	9153	03320	4134
7985	4135	9154	03321	4135
7E55	4136	9155	03322	4136
81B3	4137	9156	03323	4137
7CCE	4138	9157	03324	4138
E2C9				4139
5851	413A	9159	03326	413A
5CA8	413B	915A	03327	413B
63AA	413C	915B	03328	413C
66FE	413D	915C	03329	413D
66FD	413E	915D	03330	413E
695A	413F	915E	03331	413F
72D9	4140	915F	03332	4140
758F	4141	9160	03333	4141
758E	4142	9161	03334	4142
790E	4143	9162	03335	4143
7956	4144	9163	03336	4144
79DF	4145	9164	03337	4145
7C97	4146	9165	03338	4146
7D20	4147	9166	03339	4147
7D44	4148	9167	03340	4148
8607	4149	9168	03341	4149
8A34	414A	9169	03342	414A
963B	414B	916A	03343	414B
E2CA				414C
9F20	414D	916C	03345	414D
50E7	414E	916D	03346	414E
E2CB				414F
53CC	4150	916F	03348	4150
53E2	4151	9170	03349	4151
5009	4152	9171	03350	4152
55AA	4153	9172	03351	4153
58EE	4154	9173	03352	4154
594F	4155	9174	03353	4155
723D	4156	9175	03354	4156
5B8B	4157	9176	03355	4157
5C64	4158	9177	03356	4158
531D	4159	9178	03357	4159
60E3	415A	9179	03358	415A
60F3	415B	917A	03359	415B
635C	415C	917B	03360	415C
6383	415D	917C	03361	415D
633F	415E	917D	03362	415E
E2CC				415F
64CD	4160	9180	03364	4160
65E9	4161	9181	03365	4161
66F9	4162	9182	03366	4162
5DE3	4163	9183	03367	4163
69CD	4164	9184	03368	4164
69FD	4165	9185	03369	4165
6F15	4166	9186	03370	4166
71E5	4167	9187	03371	4167
4E89	4168	9188	03372	4168
E2CD				4169
76F8	416A	918A	03374	416A
7A93	416B	918B	03375	416B
7CDF	416C	918C	03376	416C
E2CE				416D
7D9C	416E	918E	03378	416E
E2CF				416F
8349	4170	9190	03380	4170
8358	4171	9191	03381	4171
846C	4172	9192	03382	4172
84BC	4173	9193	03383	4173
85FB	4174	9194	03384	4174
88C5	4175	9195	03385	4175
8D70	4176	9196	03386	4176
9001	4177	9197	03387	4177
906D	4178	9198	03388	4178
9397	4179	9199	03389	4179
971C	417A	919A	03390	417A
9A12	417B	919B	03391	417B
E2D0				417C
5897	417D	919D	03393	417D
618E	417E	919E	03394	417E
81D3	4221	919F	03401	4221
8535	4222	91A0	03402	4222
8D08	4223	91A1	03403	4223
9020	4224	91A2	03404	4224
4FC3	4225	91A3	03405	4225
5074	4226	91A4	03406	4226
5247	4227	91A5	03407	4227
5373	4228	91A6	03408	4228
606F	4229	91A7	03409	4229
6349	422A	91A8	03410	422A
675F	422B	91A9	03411	422B
6E2C	422C	91AA	03412	422C
8DB3	422D	91AB	03413	422D
901F	422E	91AC	03414	422E
4FD7	422F	91AD	03415	422F
5C5E	4230	91AE	03416	4230
8CCA	4231	91AF	03417	4231
65CF	4232	91B0	03418	4232
7D9A	4233	91B1	03419	4233
5352	4234	91B2	03420	4234
8896	4235	91B3	03421	4235
5176	4236	91B4	03422	4236
63C3	4237	91B5	03423	4237
5B58	4238	91B6	03424	4238
5B6B	4239	91B7	03425	4239
5C0A	423A	91B8	03426	423A
640D	423B	91B9	03427	423B
6751	423C	91BA	03428	423C
E2D1				423D
4ED6	423E	91BC	03430	423E
591A	423F	91BD	03431	423F
592A	4240	91BE	03432	4240
6C70	4241	91BF	03433	4241
8A51	4242	91C0	03434	4242
553E	4243	91C1	03435	4243
5815	4244	91C2	03436	4244
59A5	4245	91C3	03437	4245
60F0	4246	91C4	03438	4246
6253	4247	91C5	03439	4247
67C1	4248	91C6	03440	4248
8235	4249	91C7	03441	4249
6955	424A	91C8	03442	424A
9640	424B	91C9	03443	424B
99C4	424C	91CA	03444	424C
E2D2				424D
4F53	424E	91CC	03446	424E
5806	424F	91CD	03447	424F
5BFE	4250	91CE	03448	4250
8010	4251	91CF	03449	4251
5CB1	4252	91D0	03450	4252
5E2F	4253	91D1	03451	4253
5F85	4254	91D2	03452	4254
6020	4255	91D3	03453	4255
614B	4256	91D4	03454	4256
6234	4257	91D5	03455	4257
66FF	4258	91D6	03456	4258
6CF0	4259	91D7	03457	4259
6EDE	425A	91D8	03458	425A
80CE	425B	91D9	03459	425B
E2D3				425C
82D4	425D	91DB	03461	425D
888B	425E	91DC	03462	425E
8CB8	425F	91DD	03463	425F
9000	4260	91DE	03464	4260
902E	4261	91DF	03465	4261
E2D4				4262
E2D5				4263
9BDB	4264	91E2	03468	4264
4EE3	4265	91E3	03469	4265
53F0	4266	91E4	03470	4266
5927	4267	91E5	03471	4267
7B2C	4268	91E6	03472	4268
918D	4269	91E7	03473	4269
984C	426A	91E8	03474	426A
9DF9	426B	91E9	03475	426B
6EDD	426C	91EA	03476	426C
7027	426D	91EB	03477	426D
5353	426E	91EC	03478	426E
E2D6				426F
5B85	4270	91EE	03480	4270
6258	4271	91EF	03481	4271
629E	4272	91F0	03482	4272
62D3	4273	91F1	03483	4273
6CA2	4274	91F2	03484	4274
E2D7				4275
E2D8				4276
8A17	4277	91F5	03487	4277
9438	4278	91F6	03488	4278
6FC1	4279	91F7	03489	4279
8AFE	427A	91F8	03490	427A
8338	427B	91F9	03491	427B
51E7	427C	91FA	03492	427C
E2D9				427D
53EA	427E	91FC	03494	427E
53E9	4321	9240	03501	4321
4F46	4322	9241	03502	4322
9054	4323	9242	03503	4323
8FB0	4324	9243	03504	4324
596A	4325	9244	03505	4325
8131	4326	9245	03506	4326
E2DA				4327
7AEA	4328	9247	03508	4328
E2DB				4329
E2DC				432A
8C37	432B	924A	03511	432B
72F8	432C	924B	03512	432C
E2DD				432D
E2DE				432E
8AB0	432F	924E	03515	432F
4E39	4330	924F	03516	4330
5358	4331	9250	03517	4331
5606	4332	9251	03518	4332
5766	4333	9252	03519	4333
62C5	4334	9253	03520	4334
63A2	4335	9254	03521	4335
65E6	4336	9255	03522	4336
6B4E	4337	9256	03523	4337
E2DF				4338
6E5B	4339	9258	03525	4339
70AD	433A	9259	03526	433A
77ED	433B	925A	03527	433B
7AEF	433C	925B	03528	433C
E2E0				433D
7DBB	433E	925D	03530	433E
803D	433F	925E	03531	433F
80C6	4340	925F	03532	4340
86CB	4341	9260	03533	4341
8A95	4342	9261	03534	4342
935B	4343	9262	03535	4343
56E3	4344	9263	03536	4344
58C7	4345	9264	03537	4345
5F3E	4346	9265	03538	4346
65AD	4347	9266	03539	4347
6696	4348	9267	03540	4348
6A80	4349	9268	03541	4349
6BB5	434A	9269	03542	434A
7537	434B	926A	03543	434B
E2E1				434C
5024	434D	926C	03545	434D
77E5	434E	926D	03546	434E
5730	434F	926E	03547	434F
5F1B	4350	926F	03548	4350
6065	4351	9270	03549	4351
667A	4352	9271	03550	4352
6C60	4353	9272	03551	4353
75F4	4354	9273	03552	4354
7A1A	4355	9274	03553	4355
E2E2				4356
81F4	4357	9276	03555	4357
8718	4358	9277	03556	4358
9045	4359	9278	03557	4359
99B3	435A	9279	03558	435A
7BC9	435B	927A	03559	435B
755C	435C	927B	03560	435C
7AF9	435D	927C	03561	435D
7B51	435E	927D	03562	435E
84C4	435F	927E	03563	435F
9010	4360	9280	03564	4360
79E9	4361	9281	03565	4361
7A92	4362	9282	03566	4362
8336	4363	9283	03567	4363
5AE1	4364	9284	03568	4364
7740	4365	9285	03569	4365
4E2D	4366	9286	03570	4366
4EF2	4367	9287	03571	4367
5B99	4368	9288	03572	4368
5FE0	4369	9289	03573	4369
62BD	436A	928A	03574	436A
663C	436B	928B	03575	436B
67F1	436C	928C	03576	436C
6CE8	436D	928D	03577	436D
866B	436E	928E	03578	436E
8877	436F	928F	03579	436F
E2E3				4370
914E	4371	9291	03581	4371
92F3	4372	9292	03582	4372
99D0	4373	9293	03583	4373
6A17	4374	9294	03584	4374
E2E4				4375
732A	4376	9296	03586	4376
82E7	4377	9297	03587	4377
8457	4378	9298	03588	4378
8CAF	4379	9299	03589	4379
4E01	437A	929A	03590	437A
5146	437B	929B	03591	437B
E2E5				437C
558B	437D	929D	03593	437D
5BF5	437E	929E	03594	437E
5E16	4421	929F	03601	4421
5E33	4422	92A0	03602	4422
5E81	4423	92A1	03603	4423
5F14	4424	92A2	03604	4424
5F35	4425	92A3	03605	4425
5F6B	4426	92A4	03606	4426
5FB4	4427	92A5	03607	4427
61F2	4428	92A6	03608	4428
6311	4429	92A7	03609	4429
66A2	442A	92A8	03610	442A
671D	442B	92A9	03611	442B
6F6E	442C	92AA	03612	442C
7252	442D	92AB	03613	442D
753A	442E	92AC	03614	442E
773A	442F	92AD	03615	442F
8074	4430	92AE	03616	4430
8139	4431	92AF	03617	4431
8178	4432	92B0	03618	4432
8776	4433	92B1	03619	4433
8ABF	4434	92B2	03620	4434
8ADC	4435	92B3	03621	4435
8D85	4436	92B4	03622	4436
8DF3	4437	92B5	03623	4437
929A	4438	92B6	03624	4438
9577	4439	92B7	03625	4439
9802	443A	92B8	03626	443A
9CE5	443B	92B9	03627	443B
52C5	443C	92BA	03628	443C
E2E6				443D
E2E7				443E
6715	443F	92BD	03631	443F
6C88	4440	92BE	03632	4440
73CD	4441	92BF	03633	4441
8CC3	4442	92C0	03634	4442
93AE	4443	92C1	03635	4443
9673	4444	92C2	03636	4444
6D25	4445	92C3	03637	4445
E2E8				4446
690E	4447	92C5	03639	4447
E2E9				4448
8FFD	4449	92C7	03641	4449
E2EA				444A
75DB	444B	92C9	03643	444B
901A	444C	92CA	03644	444C
E2EB				444D
6802	444E	92CC	03646	444E
6451				444F
69FB	4450	92CE	03648	4450
4F43	4451	92CF	03649	4451
6F2C	4452	92D0	03650	4452
67D8	4453	92D1	03651	4453
8FBB	4454	92D2	03652	4454
8526	4455	92D3	03653	4455
7DB4	4456	92D4	03654	4456
9354	4457	92D5	03655	4457
693F	4458	92D6	03656	4458
6F70	4459	92D7	03657	4459
576A	445A	92D8	03658	445A
58FA	5464	9AE2	05268	445B
5B2C	445C	92DA	03660	445C
7D2C	445D	92DB	03661	445D
722A	445E	92DC	03662	445E
540A	445F	92DD	03663	445F
91E3	4460	92DE	03664	4460
9DB4	4461	92DF	03665	4461
4EAD	4462	92E0	03666	4462
4F4E	4463	92E1	03667	4463
505C	4464	92E2	03668	4464
5075	4465	92E3	03669	4465
5243	4466	92E4	03670	4466
8C9E	4467	92E5	03671	4467
5448	4468	92E6	03672	4468
5824	4469	92E7	03673	4469
5B9A	446A	92E8	03674	446A
5E1D	446B	92E9	03675	446B
5E95	446C	92EA	03676	446C
5EAD	446D	92EB	03677	446D
5EF7	446E	92EC	03678	446E
5F1F	446F	92ED	03679	446F
608C	4470	92EE	03680	4470
62B5	4471	92EF	03681	4471
E2EC				4472
63D0	4473	92F1	03683	4473
68AF	4474	92F2	03684	4474
6C40	4475	92F3	03685	4475
7887	4476	92F4	03686	4476
798E	4477	92F5	03687	4477
7A0B	4478	92F6	03688	4478
7DE0	4479	92F7	03689	4479
8247	447A	92F8	03690	447A
8A02	447B	92F9	03691	447B
8AE6	447C	92FA	03692	447C
8E44	447D	92FB	03693	447D
9013	447E	92FC	03694	447E
90B8	4521	9340	03701	4521
E2ED				4522
91D8	4523	9342	03703	4523
9F0E	4524	9343	03704	4524
6CE5	4525	9344	03705	4525
6458	4526	9345	03706	4526
E2EE				4527
6575	4528	9347	03708	4528
6EF4	4529	9348	03709	4529
7684	452A	9349	03710	452A
7B1B	452B	934A	03711	452B
9069	452C	934B	03712	452C
93D1	452D	934C	03713	452D
E2EF				452E
54F2	452F	934E	03715	452F
5FB9	4530	934F	03716	4530
64A4	4531	9350	03717	4531
8F4D	4532	9351	03718	4532
8FED	4533	9352	03719	4533
9244	4534	9353	03720	4534
5178	4535	9354	03721	4535
5861				4536
5929	4537	9356	03723	4537
5C55	4538	9357	03724	4538
5E97	4539	9358	03725	4539
6DFB	453A	9359	03726	453A
7E8F	453B	935A	03727	453B
751C	453C	935B	03728	453C
8CBC	453D	935C	03729	453D
8EE2	453E	935D	03730	453E
985A				453F
70B9	4540	935F	03732	4540
4F1D	4541	9360	03733	4541
6BBF	4542	9361	03734	4542
6FB1	4543	9362	03735	4543
7530	4544	9363	03736	4544
96FB	4545	9364	03737	4545
514E	4546	9365	03738	4546
5410	4547	9366	03739	4547
E2F0				4548
5857	4549	9368	03741	4549
59AC	454A	9369	03742	454A
E2F1				454B
5F92	454C	936B	03744	454C
6597	454D	936C	03745	454D
675C	454E	936D	03746	454E
6E21	454F	936E	03747	454F
767B	4550	936F	03748	4550
E2F2				4551
E2F3				4552
9014	4553	9372	03751	4553
90FD	4554	9373	03752	4554
934D	4555	9374	03753	4555
7825	4556	9375	03754	4556
792A	626A	E1E8	06674	4557
52AA	4558	9377	03756	4558
5EA6	4559	9378	03757	4559
571F	455A	9379	03758	455A
5974	455B	937A	03759	455B
6012	455C	937B	03760	455C
5012	455D	937C	03761	455D
515A	455E	937D	03762	455E
51AC	455F	937E	03763	455F
51CD	4560	9380	03764	4560
5200	4561	9381	03765	4561
5510	4562	9382	03766	4562
5854	4563	9383	03767	4563
E2F4				4564
5957	4565	9385	03769	4565
5B95	4566	9386	03770	4566
5CF6	4567	9387	03771	4567
5D8B	4568	9388	03772	4568
60BC	4569	9389	03773	4569
6295	456A	938A	03774	456A
642D	456B	938B	03775	456B
6771	456C	938C	03776	456C
6843	456D	938D	03777	456D
6AAE	5B6D	9E8D	05977	456E
68DF	456F	938F	03779	456F
76D7	4570	9390	03780	4570
6DD8	4571	9391	03781	4571
6E6F	4572	9392	03782	4572
6FE4	5E39	9FB7	06225	4573
706F	4574	9394	03784	4574
71C8	4575	9395	03785	4575
5F53	4576	9396	03786	4576
75D8	4577	9397	03787	4577
79B1				4578
7B49	4579	9399	03789	4579
7B54	457A	939A	03790	457A
7B52	457B	939B	03791	457B
7CD6	457C	939C	03792	457C
7D71	457D	939D	03793	457D
5230	457E	939E	03794	457E
8463	4621	939F	03801	4621
8569	4622	93A0	03802	4622
85E4	4623	93A1	03803	4623
8A0E	4624	93A2	03804	4624
8B04	4625	93A3	03805	4625
8C46	4626	93A4	03806	4626
8E0F	4627	93A5	03807	4627
9003	4628	93A6	03808	4628
900F	4629	93A7	03809	4629
9419	462A	93A8	03810	462A
9676	462B	93A9	03811	462B
982D	462C	93AA	03812	462C
9A30	462D	93AB	03813	462D
95D8	462E	93AC	03814	462E
50CD	462F	93AD	03815	462F
52D5	4630	93AE	03816	4630
540C	4631	93AF	03817	4631
5802	4632	93B0	03818	4632
5C0E	4633	93B1	03819	4633
61A7	4634	93B2	03820	4634
649E	4635	93B3	03821	4635
6D1E	4636	93B4	03822	4636
77B3	4637	93B5	03823	4637
7AE5	4638	93B6	03824	4638
80F4	4639	93B7	03825	4639
8404	463A	93B8	03826	463A
9053	463B	93B9	03827	463B
9285	463C	93BA	03828	463C
5CE0	463D	93BB	03829	463D
E2F5				463E
533F	463F	93BD	03831	463F
5F97	4640	93BE	03832	4640
5FB3	4641	93BF	03833	4641
7006				4642
7279	4643	93C1	03835	4643
7763	4644	93C2	03836	4644
79BF	4645	93C3	03837	4645
7BE4	4646	93C4	03838	4646
6BD2	4647	93C5	03839	4647
72EC	4648	93C6	03840	4648
8AAD	4649	93C7	03841	4649
6803	464A	93C8	03842	464A
E2F6				464B
51F8	464C	93CA	03844	464C
7A81	464D	93CB	03845	464D
6934	464E	93CC	03846	464E
5C4A	464F	93CD	03847	464F
9CF6	4650	93CE	03848	4650
82EB	4651	93CF	03849	4651
5BC5	4652	93D0	03850	4652
9149	4653	93D1	03851	4653
E2F7				4654
E2F8				4655
5C6F	4656	93D4	03854	4656
60C7	4657	93D5	03855	4657
6566	4658	93D6	03856	4658
6C8C	4659	93D7	03857	4659
E2F9				465A
E2FA				465B
E2FB				465C
5451	465D	93DB	03861	465D
66C7	465E	93DC	03862	465E
920D	465F	93DD	03863	465F
5948	4660	93DE	03864	4660
E2FC				4661
5185	4662	93E0	03866	4662
4E4D	4663	93E1	03867	4663
51EA	4664	93E2	03868	4664
8599	4665	93E3	03869	4665
E2FD				4666
E2FE				4667
637A	4668	93E6	03872	4668
934B	4669	93E7	03873	4669
E2FF				466A
99B4	466B	93E9	03875	466B
7E04	466C	93EA	03876	466C
7577	466D	93EB	03877	466D
5357	466E	93EC	03878	466E
6960	466F	93ED	03879	466F
8EDF	4670	93EE	03880	4670
96E3	4671	93EF	03881	4671
6C5D	4672	93F0	03882	4672
4E8C	4673	93F1	03883	4673
5C3C	4674	93F2	03884	4674
5F10	4675	93F3	03885	4675
9087	6D6E	E78E	07778	4676
5302	4677	93F5	03887	4677
8CD1	4678	93F6	03888	4678
8089	4679	93F7	03889	4679
8679	467A	93F8	03890	467A
5EFF	467B	93F9	03891	467B
65E5	467C	93FA	03892	467C
4E73	467D	93FB	03893	467D
5165	467E	93FC	03894	467E
5982	4721	9440	03901	4721
5C3F	4722	9441	03902	4722
97EE	4723	9442	03903	4723
4EFB	4724	9443	03904	4724
598A	4725	9444	03905	4725
5FCD	4726	9445	03906	4726
8A8D	4727	9446	03907	4727
6FE1	4728	9447	03908	4728
E300				4729
7962	472A	9449	03910	472A
5BE7	472B	944A	03911	472B
8471	472C	944B	03912	472C
732B	472D	944C	03913	472D
E301				472E
5E74	472F	944E	03915	472F
5FF5	4730	944F	03916	4730
637B	4731	9450	03917	4731
649A	4732	9451	03918	4732
71C3	4733	9452	03919	4733
7C98	4734	9453	03920	4734
4E43	4735	9454	03921	4735
E302				4736
4E4B	4737	9456	03923	4737
57DC	4738	9457	03924	4738
56CA				4739
60A9	473A	9459	03926	473A
6FC3	473B	945A	03927	473B
7D0D	473C	945B	03928	473C
80FD	473D	945C	03929	473D
8133	473E	945D	03930	473E
81BF	473F	945E	03931	473F
8FB2	4740	945F	03932	4740
8997	4741	9460	03933	4741
86A4	4742	9461	03934	4742
5DF4	4743	9462	03935	4743
628A	4744	9463	03936	4744
64AD	4745	9464	03937	4745
8987	4746	9465	03938	4746
6777	4747	9466	03939	4747
6CE2	4748	9467	03940	4748
6D3E	4749	9468	03941	4749
7436	474A	9469	03942	474A
7834	474B	946A	03943	474B
5A46	474C	946B	03944	474C
7F75	474D	946C	03945	474D
82AD	474E	946D	03946	474E
99AC	474F	946E	03947	474F
4FF3	4750	946F	03948	4750
5EC3	4751	9470	03949	4751
62DD	4752	9471	03950	4752
6392	4753	9472	03951	4753
6557	4754	9473	03952	4754
676F	4755	9474	03953	4755
76C3	4756	9475	03954	4756
E303				4757
80CC	4758	9477	03956	4758
80BA	4759	9478	03957	4759
8F29	475A	9479	03958	475A
914D	475B	947A	03959	475B
500D	475C	947B	03960	475C
57F9	475D	947C	03961	475D
5A92	475E	947D	03962	475E
6885	475F	947E	03963	475F
6973	4760	9480	03964	4760
7164	4761	9481	03965	4761
72FD	4762	9482	03966	4762
8CB7	4763	9483	03967	4763
58F2	4764	9484	03968	4764
8CE0	4765	9485	03969	4765
966A	4766	9486	03970	4766
E304				4767
8805	6A24	E5A2	07404	4768
E305				4769
77E7	476A	948A	03974	476A
8429	476B	948B	03975	476B
4F2F	476C	948C	03976	476C
E306				476D
535A	476E	948E	03978	476E
62CD	476F	948F	03979	476F
67CF	4770	9490	03980	4770
6CCA	4771	9491	03981	4771
767D	4772	9492	03982	4772
7B94	4773	9493	03983	4773
7C95	4774	9494	03984	4774
8236	4775	9495	03985	4775
8584	4776	9496	03986	4776
8FEB	4777	9497	03987	4777
66DD	4778	9498	03988	4778
6F20	4779	9499	03989	4779
7206	477A	949A	03990	477A
7E1B	477B	949B	03991	477B
83AB	477C	949C	03992	477C
E307				477D
9EA6	477E	949E	03994	477E
51FD	4821	949F	04001	4821
7BB1	4822	94A0	04002	4822
7872	4823	94A1	04003	4823
E308				4824
8087	4825	94A3	04005	4825
7B48	4826	94A4	04006	4826
6AE8	4827	94A5	04007	4827
5E61	4828	94A6	04008	4828
808C	4829	94A7	04009	4829
7551	482A	94A8	04010	482A
7560	482B	94A9	04011	482B
516B	482C	94AA	04012	482C
9262	482D	94AB	04013	482D
E309				482E
767A	482F	94AD	04015	482F
E30A				4830
9AEA	4831	94AF	04017	4831
4F10	4832	94B0	04018	4832
7F70	4833	94B1	04019	4833
629C	4834	94B2	04020	4834
7B4F	4835	94B3	04021	4835
95A5	4836	94B4	04022	4836
9CE9	4837	94B5	04023	4837
567A	4838	94B6	04024	4838
5859	4839	94B7	04025	4839
86E4	483A	94B8	04026	483A
96BC	483B	94B9	04027	483B
4F34	483C	94BA	04028	483C
5224	483D	94BB	04029	483D
534A	483E	94BC	04030	483E
53CD	483F	94BD	04031	483F
53DB	4840	94BE	04032	4840
5E06	4841	94BF	04033	4841
642C	4842	94C0	04034	4842
6591	4843	94C1	04035	4843
677F	4844	94C2	04036	4844
6C3E	4845	94C3	04037	4845
6C4E	4846	94C4	04038	4846
7248	4847	94C5	04039	4847
72AF	4848	94C6	04040	4848
E30B				4849
7554	484A	94C8	04042	484A
7E41	484B	94C9	04043	484B
822C	484C	94CA	04044	484C
85E9	484D	94CB	04045	484D
8CA9	484E	94CC	04046	484E
7BC4	484F	94CD	04047	484F
91C6	4850	94CE	04048	4850
7169	4851	94CF	04049	4851
E30C				4852
98EF	4853	94D1	04051	4853
E30D				4854
6669	4855	94D3	04053	4855
756A	4856	94D4	04054	4856
76E4	4857	94D5	04055	4857
78D0	4858	94D6	04056	4858
8543	4859	94D7	04057	4859
86EE	485A	94D8	04058	485A
532A	485B	94D9	04059	485B
E30E				485C
5426	485D	94DB	04061	485D
5983	485E	94DC	04062	485E
5E87	485F	94DD	04063	485F
5F7C	4860	94DE	04064	4860
60B2	4861	94DF	04065	4861
E30F				4862
6279	4863	94E1	04067	4863
62AB	4864	94E2	04068	4864
6590	4865	94E3	04069	4865
6BD4	4866	94E4	04070	4866
6CCC	4867	94E5	04071	4867
75B2	4868	94E6	04072	4868
76AE	4869	94E7	04073	4869
E310				486A
79D8	486B	94E9	04075	486B
7DCB	486C	94EA	04076	486C
7F77	486D	94EB	04077	486D
80A5	486E	94EC	04078	486E
88AB	486F	94ED	04079	486F
8AB9	4870	94EE	04080	4870
8CBB	4871	94EF	04081	4871
907F	4872	94F0	04082	4872
975E	4873	94F1	04083	4873
98DB	4874	94F2	04084	4874
E311				4875
7C38	4876	94F4	04086	4876
5099	4877	94F5	04087	4877
5C3E	4878	94F6	04088	4878
5FAE	4879	94F7	04089	4879
6787	487A	94F8	04090	487A
6BD8	487B	94F9	04091	487B
7435	487C	94FA	04092	487C
7709	487D	94FB	04093	487D
7F8E	487E	94FC	04094	487E
9F3B	4921	9540	04101	4921
E312				4922
E313				4923
5339	4924	9543	04104	4924
758B	4925	9544	04105	4925
9AED	4926	9545	04106	4926
5F66	4927	9546	04107	4927
E314				4928
83F1	4929	9548	04109	4929
8098	492A	9549	04110	492A
5F3C	492B	954A	04111	492B
5FC5	492C	954B	04112	492C
7562	492D	954C	04113	492D
7B46	492E	954D	04114	492E
E315				492F
6A9C	5B58	9E77	05956	4930
59EB	4931	9550	04117	4931
E316				4932
7D10	4933	9552	04119	4933
767E	4934	9553	04120	4934
E317				4935
4FF5	4936	9555	04122	4936
5F6A	4937	9556	04123	4937
6A19	4938	9557	04124	4938
6C37	4939	9558	04125	4939
6F02	493A	9559	04126	493A
74E2	493B	955A	04127	493B
7968	493C	955B	04128	493C
8868	493D	955C	04129	493D
8A55	493E	955D	04130	493E
8C79	493F	955E	04131	493F
E318				4940
63CF	4941	9560	04133	4941
75C5	4942	9561	04134	4942
79D2	4943	9562	04135	4943
82D7	4944	9563	04136	4944
9328	4945	9564	04137	4945
92F2	4946	9565	04138	4946
849C	4947	9566	04139	4947
86ED	4948	9567	04140	4948
9C2D	4949	9568	04141	4949
54C1	494A	9569	04142	494A
5F6C	494B	956A	04143	494B
658C	494C	956B	04144	494C
6D5C	494D	956C	04145	494D
E319				494E
E31A				494F
8CD3	4950	956F	04148	4950
E31B				4951
654F	4952	9571	04150	4952
E31C				4953
4E0D	4954	9573	04152	4954
4ED8	4955	9574	04153	4955
57E0	4956	9575	04154	4956
592B	4957	9576	04155	4957
5A66	4958	9577	04156	4958
5BCC	4959	9578	04157	4959
51A8	495A	9579	04158	495A
5E03	495B	957A	04159	495B
5E9C	495C	957B	04160	495C
6016	495D	957C	04161	495D
6276	495E	957D	04162	495E
6577	495F	957E	04163	495F
E31D				4960
666E	4961	9581	04165	4961
6D6E	4962	9582	04166	4962
7236	4963	9583	04167	4963
7B26	4964	9584	04168	4964
8150	4965	9585	04169	4965
819A	4966	9586	04170	4966
8299	4967	9587	04171	4967
8B5C	4968	9588	04172	4968
8CA0	4969	9589	04173	4969
8CE6	496A	958A	04174	496A
8D74	496B	958B	04175	496B
961C	496C	958C	04176	496C
9644	496D	958D	04177	496D
4FAE	496E	958E	04178	496E
64AB	496F	958F	04179	496F
6B66	4970	9590	04180	4970
821E	4971	9591	04181	4971
8461	4972	9592	04182	4972
856A	4973	9593	04183	4973
90E8	4974	9594	04184	4974
5C01	4975	9595	04185	4975
6953	4976	9596	04186	4976
98A8	4977	9597	04187	4977
847A	4978	9598	04188	4978
8557	4979	9599	04189	4979
4F0F	497A	959A	04190	497A
526F	497B	959B	04191	497B
5FA9	497C	959C	04192	497C
5E45	497D	959D	04193	497D
670D	497E	959E	04194	497E
798F	4A21	959F	04201	4A21
8179	4A22	95A0	04202	4A22
8907	4A23	95A1	04203	4A23
8986	4A24	95A2	04204	4A24
6DF5	4A25	95A3	04205	4A25
5F17	4A26	95A4	04206	4A26
6255	4A27	95A5	04207	4A27
6CB8	4A28	95A6	04208	4A28
4ECF	4A29	95A7	04209	4A29
7269	4A2A	95A8	04210	4A2A
9B92	4A2B	95A9	04211	4A2B
E31E				4A2C
543B	4A2D	95AB	04213	4A2D
5674	4A2E	95AC	04214	4A2E
58B3	4A2F	95AD	04215	4A2F
61A4	4A30	95AE	04216	4A30
626E	4A31	95AF	04217	4A31
711A	4A32	95B0	04218	4A32
596E	4A33	95B1	04219	4A33
E31F				4A34
7CDE	4A35	95B3	04221	4A35
E320				4A36
E321				4A37
6587	4A38	95B6	04224	4A38
805E	4A39	95B7	04225	4A39
4E19	4A3A	95B8	04226	4A3A
4F75	4A3B	95B9	04227	4A3B
5175	4A3C	95BA	04228	4A3C
5840	4A3D	95BB	04229	4A3D
5E63	4A3E	95BC	04230	4A3E
5E73	4A3F	95BD	04231	4A3F
5F0A	4A40	95BE	04232	4A40
67C4	4A41	95BF	04233	4A41
4E26	4A42	95C0	04234	4A42
E322				4A43
9589	4A44	95C2	04236	4A44
965B	4A45	95C3	04237	4A45
7C73	4A46	95C4	04238	4A46
9801	4A47	95C5	04239	4A47
50FB	4A48	95C6	04240	4A48
58C1	4A49	95C7	04241	4A49
7656	4A4A	95C8	04242	4A4A
78A7	4A4B	95C9	04243	4A4B
5225	4A4C	95CA	04244	4A4C
E323				4A4D
8511	4A4E	95CC	04246	4A4E
7B86	4A4F	95CD	04247	4A4F
504F	4A50	95CE	04248	4A50
5909	4A51	95CF	04249	4A51
7247	4A52	95D0	04250	4A52
7BC7	4A53	95D1	04251	4A53
7DE8	4A54	95D2	04252	4A54
8FBA	4A55	95D3	04253	4A55
E324				4A56
904D	4A57	95D5	04255	4A57
4FBF	4A58	95D6	04256	4A58
52C9	4A59	95D7	04257	4A59
E325				4A5A
5F01	4A5B	95D9	04259	4A5B
E326				4A5C
4FDD	4A5D	95DB	04261	4A5D
8217	4A5E	95DC	04262	4A5E
92EA	4A5F	95DD	04263	4A5F
5703	4A60	95DE	04264	4A60
6355	4A61	95DF	04265	4A61
6B69	4A62	95E0	04266	4A62
752B	4A63	95E1	04267	4A63
88DC	4A64	95E2	04268	4A64
8F14	4A65	95E3	04269	4A65
7A42	4A66	95E4	04270	4A66
52DF	4A67	95E5	04271	4A67
5893	4A68	95E6	04272	4A68
6155	4A69	95E7	04273	4A69
620A	4A6A	95E8	04274	4A6A
66AE	4A6B	95E9	04275	4A6B
6BCD	4A6C	95EA	04276	4A6C
7C3F	4A6D	95EB	04277	4A6D
83E9	4A6E	95EC	04278	4A6E
5023	4A6F	95ED	04279	4A6F
4FF8	4A70	95EE	04280	4A70
5305	4A71	95EF	04281	4A71
5446	4A72	95F0	04282	4A72
5831	4A73	95F1	04283	4A73
5949	4A74	95F2	04284	4A74
5B9D	4A75	95F3	04285	4A75
5CF0	4A76	95F4	04286	4A76
5CEF	4A77	95F5	04287	4A77
5D29	4A78	95F6	04288	4A78
E327				4A79
62B1	4A7A	95F8	04290	4A7A
6367	4A7B	95F9	04291	4A7B
653E	4A7C	95FA	04292	4A7C
65B9	4A7D	95FB	04293	4A7D
670B	4A7E	95FC	04294	4A7E
6CD5	4B21	9640	04301	4B21
E328				4B22
70F9	4B23	9642	04303	4B23
7832	4B24	9643	04304	4B24
E329				4B25
80DE	4B26	9645	04306	4B26
82B3	4B27	9646	04307	4B27
840C	4B28	9647	04308	4B28
E32A				4B29
8702	4B2A	9649	04310	4B2A
8912	4B2B	964A	04311	4B2B
8A2A	4B2C	964B	04312	4B2C
8C4A	4B2D	964C	04313	4B2D
90A6	4B2E	964D	04314	4B2E
92D2	4B2F	964E	04315	4B2F
98FD	4B30	964F	04316	4B30
9CF3	4B31	9650	04317	4B31
9D6C	4B32	9651	04318	4B32
4E4F	4B33	9652	04319	4B33
4EA1	4B34	9653	04320	4B34
508D	4B35	9654	04321	4B35
5256	4B36	9655	04322	4B36
574A	4B37	9656	04323	4B37
59A8	4B38	9657	04324	4B38
5E3D	4B39	9658	04325	4B39
5FD8	4B3A	9659	04326	4B3A
5FD9	4B3B	965A	04327	4B3B
623F	4B3C	965B	04328	4B3C
66B4	4B3D	965C	04329	4B3D
671B	4B3E	965D	04330	4B3E
67D0	4B3F	965E	04331	4B3F
68D2	4B40	965F	04332	4B40
5192	4B41	9660	04333	4B41
7D21	4B42	9661	04334	4B42
80AA	4B43	9662	04335	4B43
81A8	4B44	9663	04336	4B44
8B00	4B45	9664	04337	4B45
8C8C	4B46	9665	04338	4B46
8CBF	4B47	9666	04339	4B47
927E	4B48	9667	04340	4B48
9632	4B49	9668	04341	4B49
5420	4B4A	9669	04342	4B4A
E32B				4B4B
5317	4B4C	966B	04344	4B4C
50D5	4B4D	966C	04345	4B4D
535C	4B4E	966D	04346	4B4E
58A8	4B4F	966E	04347	4B4F
64B2	4B50	966F	04348	4B50
6734	4B51	9670	04349	4B51
7267	4B52	9671	04350	4B52
7766	4B53	9672	04351	4B53
7A46	4B54	9673	04352	4B54
91E6	4B55	9674	04353	4B55
52C3	4B56	9675	04354	4B56
6CA1	4B57	9676	04355	4B57
6B86	4B58	9677	04356	4B58
5800	4B59	9678	04357	4B59
5E4C	4B5A	9679	04358	4B5A
5954	4B5B	967A	04359	4B5B
672C	4B5C	967B	04360	4B5C
7FFB	4B5D	967C	04361	4B5D
51E1	4B5E	967D	04362	4B5E
E32C				4B5F
6469	4B60	9680	04364	4B60
78E8	4B61	9681	04365	4B61
9B54	4B62	9682	04366	4B62
9EBB	4B63	9683	04367	4B63
57CB	4B64	9684	04368	4B64
59B9	4B65	9685	04369	4B65
6627	4B66	9686	04370	4B66
679A	4B67	9687	04371	4B67
6BCE	4B68	9688	04372	4B68
54E9	4B69	9689	04373	4B69
69C7	7422	EAA0	08402	4B6A
5E55	4B6B	968B	04375	4B6B
819C	4B6C	968C	04376	4B6C
6795	4B6D	968D	04377	4B6D
9BAA	4B6E	968E	04378	4B6E
67FE	4B6F	968F	04379	4B6F
E32D				4B70
685D	4B71	9691	04381	4B71
4EA6	4B72	9692	04382	4B72
4FE3	4B73	9693	04383	4B73
53C8	4B74	9694	04384	4B74
62B9	4B75	9695	04385	4B75
672B	4B76	9696	04386	4B76
6CAB	4B77	9697	04387	4B77
8FC4	4B78	9698	04388	4B78
5118	5056	98D4	04854	4B79
7E6D	4B7A	969A	04390	4B7A
9EBF	4B7B	969B	04391	4B7B
4E07	4B7C	969C	04392	4B7C
6162	4B7D	969D	04393	4B7D
6E80	4B7E	969E	04394	4B7E
6F2B	4C21	969F	04401	4C21
8513	4C22	96A0	04402	4C22
5473	4C23	96A1	04403	4C23
672A	4C24	96A2	04404	4C24
9B45	4C25	96A3	04405	4C25
5DF3	4C26	96A4	04406	4C26
7B95	4C27	96A5	04407	4C27
5CAC	4C28	96A6	04408	4C28
5BC6	4C29	96A7	04409	4C29
871C	4C2A	96A8	04410	4C2A
E32E				4C2B
84D1	4C2C	96AA	04412	4C2C
7A14	4C2D	96AB	04413	4C2D
8108	4C2E	96AC	04414	4C2E
5999	4C2F	96AD	04415	4C2F
E32F				4C30
6C11	4C31	96AF	04417	4C31
7720	4C32	96B0	04418	4C32
52D9	4C33	96B1	04419	4C33
5922	4C34	96B2	04420	4C34
7121	4C35	96B3	04421	4C35
725F	4C36	96B4	04422	4C36
77DB	4C37	96B5	04423	4C37
9727	4C38	96B6	04424	4C38
9D61	4C39	96B7	04425	4C39
690B	4C3A	96B8	04426	4C3A
5A7F	4C3B	96B9	04427	4C3B
5A18	4C3C	96BA	04428	4C3C
51A5	4C3D	96BB	04429	4C3D
540D	4C3E	96BC	04430	4C3E
547D	4C3F	96BD	04431	4C3F
660E	4C40	96BE	04432	4C40
76DF	4C41	96BF	04433	4C41
8FF7	4C42	96C0	04434	4C42
9298	4C43	96C1	04435	4C43
9CF4	4C44	96C2	04436	4C44
59EA	4C45	96C3	04437	4C45
725D	4C46	96C4	04438	4C46
6EC5	4C47	96C5	04439	4C47
514D	4C48	96C6	04440	4C48
68C9	4C49	96C7	04441	4C49
7DBF	4C4A	96C8	04442	4C4A
7DEC	4C4B	96C9	04443	4C4B
9762	4C4C	96CA	04444	4C4C
E330				4C4D
6478	4C4E	96CC	04446	4C4E
6A21	4C4F	96CD	04447	4C4F
8302	4C50	96CE	04448	4C50
5984	4C51	96CF	04449	4C51
5B5F	4C52	96D0	04450	4C52
6BDB	4C53	96D1	04451	4C53
731B	4C54	96D2	04452	4C54
76F2	4C55	96D3	04453	4C55
7DB2	4C56	96D4	04454	4C56
8017	4C57	96D5	04455	4C57
E331				4C58
E332				4C59
6728	4C5A	96D8	04458	4C5A
9ED9	4C5B	96D9	04459	4C5B
76EE	4C5C	96DA	04460	4C5C
6762	4C5D	96DB	04461	4C5D
52FF	4C5E	96DC	04462	4C5E
E333				4C5F
5C24	4C60	96DE	04464	4C60
623B	4C61	96DF	04465	4C61
E334				4C62
8CB0	4C63	96E1	04467	4C63
554F	4C64	96E2	04468	4C64
60B6	4C65	96E3	04469	4C65
7D0B	4C66	96E4	04470	4C66
9580	4C67	96E5	04471	4C67
5301	4C68	96E6	04472	4C68
4E5F	4C69	96E7	04473	4C69
51B6	4C6A	96E8	04474	4C6A
591C	4C6B	96E9	04475	4C6B
E335				4C6C
8036	4C6D	96EB	04477	4C6D
91CE	4C6E	96EC	04478	4C6E
5F25	4C6F	96ED	04479	4C6F
77E2	4C70	96EE	04480	4C70
5384	4C71	96EF	04481	4C71
5F79	4C72	96F0	04482	4C72
7D04	4C73	96F1	04483	4C73
85AC	4C74	96F2	04484	4C74
8A33	4C75	96F3	04485	4C75
8E8D	4C76	96F4	04486	4C76
9756	4C77	96F5	04487	4C77
67F3	4C78	96F6	04488	4C78
85EA	692E	E54D	07314	4C79
E336				4C7A
6109	4C7B	96F9	04491	4C7B
E337				4C7C
6CB9	4C7D	96FB	04493	4C7D
E338				4C7E
8AED	4D21	9740	04501	4D21
8F38	4D22	9741	04502	4D22
552F	4D23	9742	04503	4D23
4F51	4D24	9743	04504	4D24
512A	4D25	9744	04505	4D25
52C7	4D26	9745	04506	4D26
53CB	4D27	9746	04507	4D27
5BA5	4D28	9747	04508	4D28
5E7D	4D29	9748	04509	4D29
60A0	4D2A	9749	04510	4D2A
6182	4D2B	974A	04511	4D2B
63D6	4D2C	974B	04512	4D2C
6709	4D2D	974C	04513	4D2D
67DA	4D2E	974D	04514	4D2E
6E67	4D2F	974E	04515	4D2F
6D8C	4D30	974F	04516	4D30
7336	4D31	9750	04517	4D31
E339				4D32
7531	4D33	9752	04519	4D33
7950	4D34	9753	04520	4D34
88D5	4D35	9754	04521	4D35
8A98	4D36	9755	04522	4D36
904A	4D37	9756	04523	4D37
9091	4D38	9757	04524	4D38
90F5	4D39	9758	04525	4D39
96C4	4D3A	9759	04526	4D3A
878D	4D3B	975A	04527	4D3B
5915	4D3C	975B	04528	4D3C
4E88	4D3D	975C	04529	4D3D
4F59	4D3E	975D	04530	4D3E
4E0E	4D3F	975E	04531	4D3F
8A89	4D40	975F	04532	4D40
8F3F	4D41	9760	04533	4D41
9810	4D42	9761	04534	4D42
50AD	4D43	9762	04535	4D43
5E7C	4D44	9763	04536	4D44
5996	4D45	9764	04537	4D45
5BB9	4D46	9765	04538	4D46
5EB8	4D47	9766	04539	4D47
63DA	4D48	9767	04540	4D48
63FA	4D49	9768	04541	4D49
64C1	4D4A	9769	04542	4D4A
66DC	4D4B	976A	04543	4D4B
694A	4D4C	976B	04544	4D4C
69D8	4D4D	976C	04545	4D4D
6D0B	4D4E	976D	04546	4D4E
6EB6	4D4F	976E	04547	4D4F
E33A				4D50
7528	4D51	9770	04549	4D51
7AAF	4D52	9771	04550	4D52
7F8A	4D53	9772	04551	4D53
E33B				4D54
8449	4D55	9774	04553	4D55
84C9	4D56	9775	04554	4D56
8981	4D57	9776	04555	4D57
8B21	4D58	9777	04556	4D58
8E0A	4D59	9778	04557	4D59
9059	7423	EAA1	08403	4D5A
967D	4D5B	977A	04559	4D5B
990A	4D5C	977B	04560	4D5C
617E	4D5D	977C	04561	4D5D
6291	4D5E	977D	04562	4D5E
6B32	4D5F	977E	04563	4D5F
6C83	4D60	9780	04564	4D60
6D74	4D61	9781	04565	4D61
7FCC	4D62	9782	04566	4D62
7FFC	4D63	9783	04567	4D63
6DC0	4D64	9784	04568	4D64
7F85	4D65	9785	04569	4D65
87BA	4D66	9786	04570	4D66
88F8	4D67	9787	04571	4D67
6765	4D68	9788	04572	4D68
E33C				4D69
983C	4D6A	978A	04574	4D6A
96F7	4D6B	978B	04575	4D6B
6D1B	4D6C	978C	04576	4D6C
7D61	4D6D	978D	04577	4D6D
843D	4D6E	978E	04578	4D6E
916A	4D6F	978F	04579	4D6F
4E71	4D70	9790	04580	4D70
5375	4D71	9791	04581	4D71
5D50	4D72	9792	04582	4D72
6B04	4D73	9793	04583	4D73
6FEB	4D74	9794	04584	4D74
85CD	4D75	9795	04585	4D75
862D	4D76	9796	04586	4D76
89A7	4D77	9797	04587	4D77
5229	4D78	9798	04588	4D78
540F	4D79	9799	04589	4D79
5C65	4D7A	979A	04590	4D7A
674E	4D7B	979B	04591	4D7B
68A8	4D7C	979C	04592	4D7C
7406	4D7D	979D	04593	4D7D
7483	4D7E	979E	04594	4D7E
75E2	4E21	979F	04601	4E21
88CF	4E22	97A0	04602	4E22
88E1	4E23	97A1	04603	4E23
91CC	4E24	97A2	04604	4E24
96E2	4E25	97A3	04605	4E25
9678	4E26	97A4	04606	4E26
5F8B	4E27	97A5	04607	4E27
7387	4E28	97A6	04608	4E28
7ACB	4E29	97A7	04609	4E29
844E	4E2A	97A8	04610	4E2A
63A0	4E2B	97A9	04611	4E2B
7565	4E2C	97AA	04612	4E2C
5289	4E2D	97AB	04613	4E2D
6D41	4E2E	97AC	04614	4E2E
6E9C	4E2F	97AD	04615	4E2F
7409	4E30	97AE	04616	4E30
7559	4E31	97AF	04617	4E31
786B	4E32	97B0	04618	4E32
7C92	4E33	97B1	04619	4E33
9686	4E34	97B2	04620	4E34
7ADC	4E35	97B3	04621	4E35
9F8D	4E36	97B4	04622	4E36
4FB6	4E37	97B5	04623	4E37
616E	4E38	97B6	04624	4E38
65C5	4E39	97B7	04625	4E39
865C	4E3A	97B8	04626	4E3A
4E86	4E3B	97B9	04627	4E3B
4EAE	4E3C	97BA	04628	4E3C
50DA	4E3D	97BB	04629	4E3D
4E21	4E3E	97BC	04630	4E3E
51CC	4E3F	97BD	04631	4E3F
5BEE	4E40	97BE	04632	4E40
6599	4E41	97BF	04633	4E41
6881	4E42	97C0	04634	4E42
6DBC	4E43	97C1	04635	4E43
731F	4E44	97C2	04636	4E44
7642	4E45	97C3	04637	4E45
77AD	4E46	97C4	04638	4E46
7A1C	4E47	97C5	04639	4E47
7CE7	4E48	97C6	04640	4E48
826F	4E49	97C7	04641	4E49
8AD2	4E4A	97C8	04642	4E4A
E33D				4E4B
91CF	4E4C	97CA	04644	4E4C
9675	4E4D	97CB	04645	4E4D
9818	4E4E	97CC	04646	4E4E
529B	4E4F	97CD	04647	4E4F
7DD1	4E50	97CE	04648	4E50
502B	4E51	97CF	04649	4E51
5398	4E52	97D0	04650	4E52
6797	4E53	97D1	04651	4E53
6DCB	4E54	97D2	04652	4E54
71D0	4E55	97D3	04653	4E55
7433	4E56	97D4	04654	4E56
81E8	4E57	97D5	04655	4E57
8F2A	4E58	97D6	04656	4E58
96A3	4E59	97D7	04657	4E59
9C57	4E5A	97D8	04658	4E5A
9E9F	4E5B	97D9	04659	4E5B
7460	4E5C	97DA	04660	4E5C
5841	4E5D	97DB	04661	4E5D
6D99	4E5E	97DC	04662	4E5E
7D2F	4E5F	97DD	04663	4E5F
985E	4E60	97DE	04664	4E60
4EE4	4E61	97DF	04665	4E61
4F36	4E62	97E0	04666	4E62
4F8B	4E63	97E1	04667	4E63
51B7	4E64	97E2	04668	4E64
52B1	4E65	97E3	04669	4E65
5DBA	4E66	97E4	04670	4E66
601C	4E67	97E5	04671	4E67
73B2	4E68	97E6	04672	4E68
793C	4E69	97E7	04673	4E69
82D3	4E6A	97E8	04674	4E6A
9234	4E6B	97E9	04675	4E6B
96B7	4E6C	97EA	04676	4E6C
96F6	4E6D	97EB	04677	4E6D
970A	4E6E	97EC	04678	4E6E
E33E				4E6F
9F62	4E70	97EE	04680	4E70
66A6	4E71	97EF	04681	4E71
6B74	4E72	97F0	04682	4E72
5217	4E73	97F1	04683	4E73
52A3	4E74	97F2	04684	4E74
70C8	4E75	97F3	04685	4E75
88C2	4E76	97F4	04686	4E76
5EC9	4E77	97F5	04687	4E77
604B	4E78	97F6	04688	4E78
6190	4E79	97F7	04689	4E79
E33F				4E7A
E340				4E7B
7C3E	4E7C	97FA	04692	4E7C
7DF4	4E7D	97FB	04693	4E7D
806F	4E7E	97FC	04694	4E7E
E341				4F21
9023	4F22	9841	04702	4F22
932C	4F23	9842	04703	4F23
5442	4F24	9843	04704	4F24
9B6F	4F25	9844	04705	4F25
6AD3	4F26	9845	04706	4F26
7089	4F27	9846	04707	4F27
8CC2	4F28	9847	04708	4F28
8DEF	4F29	9848	04709	4F29
9732	4F2A	9849	04710	4F2A
52B4	4F2B	984A	04711	4F2B
5A41	4F2C	984B	04712	4F2C
EB67				4F2D
5F04	4F2E	984D	04714	4F2E
6717	4F2F	984E	04715	4F2F
697C	4F30	984F	04716	4F30
E342				4F31
6D6A	4F32	9851	04718	4F32
6F0F	4F33	9852	04719	4F33
7262	4F34	9853	04720	4F34
72FC	4F35	9854	04721	4F35
7C60	6446	E2C4	06838	4F36
8001	4F37	9856	04723	4F37
807E	4F38	9857	04724	4F38
881F				4F39
90CE	4F3A	9859	04726	4F3A
516D	4F3B	985A	04727	4F3B
9E93	4F3C	985B	04728	4F3C
7984	4F3D	985C	04729	4F3D
808B	4F3E	985D	04730	4F3E
9332	4F3F	985E	04731	4F3F
8AD6	4F40	985F	04732	4F40
502D	4F41	9860	04733	4F41
548C	4F42	9861	04734	4F42
8A71	4F43	9862	04735	4F43
6B6A	4F44	9863	04736	4F44
8CC4	4F45	9864	04737	4F45
8107	4F46	9865	04738	4F46
60D1	4F47	9866	04739	4F47
67A0	4F48	9867	04740	4F48
9DF2	4F49	9868	04741	4F49
4E99	4F4A	9869	04742	4F4A
4E98	4F4B	986A	04743	4F4B
9C10	4F4C	986B	04744	4F4C
8A6B	4F4D	986C	04745	4F4D
85C1	4F4E	986D	04746	4F4E
8568	4F4F	986E	04747	4F4F
6900	4F50	986F	04748	4F50
6E7E	4F51	9870	04749	4F51
7897	4F52	9871	04750	4F52
8155	4F53	9872	04751	4F53
5F0C	5021	989F	04801	5021
4E10	5022	98A0	04802	5022
4E15	5023	98A1	04803	5023
4E2A	5024	98A2	04804	5024
4E31	5025	98A3	04805	5025
4E36	5026	98A4	04806	5026
4E3C	5027	98A5	04807	5027
4E3F	5028	98A6	04808	5028
E343				5029
4E56	502A	98A8	04810	502A
4E58	502B	98A9	04811	502B
4E82	502C	98AA	04812	502C
4E85	502D	98AB	04813	502D
8C6B	502E	98AC	04814	502E
4E8A	502F	98AD	04815	502F
8212	5030	98AE	04816	5030
5F0D	5031	98AF	04817	5031
4E8E	5032	98B0	04818	5032
4E9E	5033	98B1	04819	5033
E344				5034
4EA0	5035	98B3	04821	5035
4EA2	5036	98B4	04822	5036
4EB0	5037	98B5	04823	5037
4EB3	5038	98B6	04824	5038
4EB6	5039	98B7	04825	5039
4ECE	503A	98B8	04826	503A
4ECD	503B	98B9	04827	503B
4EC4	503C	98BA	04828	503C
4EC6	503D	98BB	04829	503D
4EC2	503E	98BC	04830	503E
E345				503F
4EDE	5040	98BE	04832	5040
E346				5041
4EDF	5042	98C0	04834	5042
4EF7	5043	98C1	04835	5043
4F09	5044	98C2	04836	5044
4F5A	5045	98C3	04837	5045
4F30	5046	98C4	04838	5046
4F5B	5047	98C5	04839	5047
4F5D	5048	98C6	04840	5048
4F57	5049	98C7	04841	5049
4F47	504A	98C8	04842	504A
4F76	504B	98C9	04843	504B
4F88	504C	98CA	04844	504C
4F8F	504D	98CB	04845	504D
4F98	504E	98CC	04846	504E
4F7B	504F	98CD	04847	504F
4F69	5050	98CE	04848	5050
4F70	5051	98CF	04849	5051
4F91	5052	98D0	04850	5052
4F6F	5053	98D1	04851	5053
4F86	5054	98D2	04852	5054
4F96	5055	98D3	04853	5055
4FAD	4B79	9699	04389	5056
4FD4	5057	98D5	04855	5057
4FDF	5058	98D6	04856	5058
4FCE	5059	98D7	04857	5059
4FD8	505A	98D8	04858	505A
4FDB	505B	98D9	04859	505B
4FD1	505C	98DA	04860	505C
4FDA	505D	98DB	04861	505D
4FD0	505E	98DC	04862	505E
4FE4	505F	98DD	04863	505F
4FE5	5060	98DE	04864	5060
501A	5061	98DF	04865	5061
5028	5062	98E0	04866	5062
5014	5063	98E1	04867	5063
502A	5064	98E2	04868	5064
5025	5065	98E3	04869	5065
5005	5066	98E4	04870	5066
4F1C	5067	98E5	04871	5067
4FF6	5068	98E6	04872	5068
5021	5069	98E7	04873	5069
5029	506A	98E8	04874	506A
502C	506B	98E9	04875	506B
4FFE	506C	98EA	04876	506C
4FEF	506D	98EB	04877	506D
5011	506E	98EC	04878	506E
5006	506F	98ED	04879	506F
5043	5070	98EE	04880	5070
5047	5071	98EF	04881	5071
6703	5072	98F0	04882	5072
5055	5073	98F1	04883	5073
5050	5074	98F2	04884	5074
5048	5075	98F3	04885	5075
505A	5076	98F4	04886	5076
5056	5077	98F5	04887	5077
506C	5078	98F6	04888	5078
E347				5079
5080	507A	98F8	04890	507A
509A	507B	98F9	04891	507B
5085	507C	98FA	04892	507C
50B4	507D	98FB	04893	507D
50B2	507E	98FC	04894	507E
50C9	5121	9940	04901	5121
50CA	5122	9941	04902	5122
50B3	5123	9942	04903	5123
50C2	5124	9943	04904	5124
50D6	5125	9944	04905	5125
50DE	5126	9945	04906	5126
50E5	5127	9946	04907	5127
50ED	5128	9947	04908	5128
50E3	5129	9948	04909	5129
50EE	512A	9949	04910	512A
50F9	512B	994A	04911	512B
50F5	512C	994B	04912	512C
5109	512D	994C	04913	512D
5101	512E	994D	04914	512E
5102	512F	994E	04915	512F
5116	5130	994F	04916	5130
5115	5131	9950	04917	5131
5114	5132	9951	04918	5132
511A	5133	9952	04919	5133
5121	5134	9953	04920	5134
513A	5135	9954	04921	5135
5137	5136	9955	04922	5136
513C	5137	9956	04923	5137
513B	5138	9957	04924	5138
513F	5139	9958	04925	5139
5140	513A	9959	04926	513A
5152	513B	995A	04927	513B
514C	513C	995B	04928	513C
E348				513D
5162	513E	995D	04930	513E
7AF8	513F	995E	04931	513F
5169	5140	995F	04932	5140
516A	5141	9960	04933	5141
516E	5142	9961	04934	5142
5180	5143	9962	04935	5143
5182	5144	9963	04936	5144
56D8	5145	9964	04937	5145
518C	5146	9965	04938	5146
E349				5147
518F	5148	9967	04940	5148
5191	5149	9968	04941	5149
5193	514A	9969	04942	514A
E34A				514B
5196	514C	996B	04944	514C
E34B				514D
51A6	514E	996D	04946	514E
51A2	514F	996E	04947	514F
51A9	5150	996F	04948	5150
51AA	5151	9970	04949	5151
51AB	5152	9971	04950	5152
51B3	5153	9972	04951	5153
51B1	5154	9973	04952	5154
51B2	5155	9974	04953	5155
51B0	5156	9975	04954	5156
51B5	5157	9976	04955	5157
51BD	5158	9977	04956	5158
51C5	5159	9978	04957	5159
51C9	515A	9979	04958	515A
51DB	515B	997A	04959	515B
51E0	515C	997B	04960	515C
8655	515D	997C	04961	515D
51E9	515E	997D	04962	515E
51ED	515F	997E	04963	515F
51F0	5160	9980	04964	5160
51F5	5161	9981	04965	5161
51FE	5162	9982	04966	5162
E34C				5163
520B	5164	9984	04968	5164
5214	5165	9985	04969	5165
520E	5166	9986	04970	5166
5227	5167	9987	04971	5167
522A	5168	9988	04972	5168
522E	5169	9989	04973	5169
5233	516A	998A	04974	516A
5239	516B	998B	04975	516B
E34D				516C
5244	516D	998D	04977	516D
524B	516E	998E	04978	516E
524C	516F	998F	04979	516F
525E	5170	9990	04980	5170
5254	5171	9991	04981	5171
526A	5172	9992	04982	5172
5274	5173	9993	04983	5173
5269	5174	9994	04984	5174
5273	5175	9995	04985	5175
527F	5176	9996	04986	5176
527D	5177	9997	04987	5177
528D	5178	9998	04988	5178
E34E				5179
5292	517A	999A	04990	517A
E34F				517B
5288	517C	999C	04992	517C
5291	517D	999D	04993	517D
8FA8	517E	999E	04994	517E
8FA7	5221	999F	05001	5221
52AC	5222	99A0	05002	5222
52AD	5223	99A1	05003	5223
52BC	5224	99A2	05004	5224
52B5	5225	99A3	05005	5225
52C1	5226	99A4	05006	5226
52CD	5227	99A5	05007	5227
52D7	5228	99A6	05008	5228
52DE	5229	99A7	05009	5229
52E3	522A	99A8	05010	522A
52E6	522B	99A9	05011	522B
98ED	522C	99AA	05012	522C
52E0	522D	99AB	05013	522D
52F3	522E	99AC	05014	522E
52F5	522F	99AD	05015	522F
52F8	5230	99AE	05016	5230
52F9	5231	99AF	05017	5231
5306	5232	99B0	05018	5232
5308	5233	99B1	05019	5233
7538	5234	99B2	05020	5234
530D	5235	99B3	05021	5235
5310	5236	99B4	05022	5236
530F	5237	99B5	05023	5237
E350				5238
531A	5239	99B7	05025	5239
5323	523A	99B8	05026	523A
532F	523B	99B9	05027	523B
5331	523C	99BA	05028	523C
5333	523D	99BB	05029	523D
5338	523E	99BC	05030	523E
E351				523F
5346	5240	99BE	05032	5240
5345	5241	99BF	05033	5241
4E17	5242	99C0	05034	5242
5349	5243	99C1	05035	5243
534D	5244	99C2	05036	5244
51D6	5245	99C3	05037	5245
535E	5246	99C4	05038	5246
5369	5247	99C5	05039	5247
536E	5248	99C6	05040	5248
5918	5249	99C7	05041	5249
537B	524A	99C8	05042	524A
5377	524B	99C9	05043	524B
5382	524C	99CA	05044	524C
5396	524D	99CB	05045	524D
53A0	524E	99CC	05046	524E
53A6	524F	99CD	05047	524F
53A5	5250	99CE	05048	5250
53AE	5251	99CF	05049	5251
53B0	5252	99D0	05050	5252
53B6	5253	99D1	05051	5253
53C3	5254	99D2	05052	5254
7C12	5255	99D3	05053	5255
96D9	5256	99D4	05054	5256
53DF	5257	99D5	05055	5257
66FC	5258	99D6	05056	5258
71EE	5259	99D7	05057	5259
53EE	525A	99D8	05058	525A
53E8	525B	99D9	05059	525B
53ED	525C	99DA	05060	525C
53FA	525D	99DB	05061	525D
5401	525E	99DC	05062	525E
543D	525F	99DD	05063	525F
5440	5260	99DE	05064	5260
542C	5261	99DF	05065	5261
542D	5262	99E0	05066	5262
543C	5263	99E1	05067	5263
542E	5264	99E2	05068	5264
5436	5265	99E3	05069	5265
5429	5266	99E4	05070	5266
E352				5267
544E	5268	99E6	05072	5268
548F	5269	99E7	05073	5269
5475	526A	99E8	05074	526A
548E	526B	99E9	05075	526B
545F	526C	99EA	05076	526C
5471	526D	99EB	05077	526D
5477	526E	99EC	05078	526E
5470	526F	99ED	05079	526F
5492	5270	99EE	05080	5270
547B	5271	99EF	05081	5271
5480	5272	99F0	05082	5272
5476	5273	99F1	05083	5273
5484	5274	99F2	05084	5274
5490	5275	99F3	05085	5275
5486	5276	99F4	05086	5276
54C7	5277	99F5	05087	5277
54A2	5278	99F6	05088	5278
54B8	5279	99F7	05089	5279
54A5	527A	99F8	05090	527A
E353				527B
54C4	527C	99FA	05092	527C
54C8	527D	99FB	05093	527D
54A8	527E	99FC	05094	527E
54AB	5321	9A40	05101	5321
54C2	5322	9A41	05102	5322
54A4	5323	9A42	05103	5323
54BE	5324	9A43	05104	5324
54BC	5325	9A44	05105	5325
54D8	5326	9A45	05106	5326
54E5	5327	9A46	05107	5327
54E6	5328	9A47	05108	5328
550F	5329	9A48	05109	5329
5514	532A	9A49	05110	532A
E354				532B
54EE	532C	9A4B	05112	532C
54ED	532D	9A4C	05113	532D
54FA	532E	9A4D	05114	532E
54E2	532F	9A4E	05115	532F
E355				5330
5540	5331	9A50	05117	5331
5563	5332	9A51	05118	5332
554C	5333	9A52	05119	5333
552E	5334	9A53	05120	5334
555C	5335	9A54	05121	5335
5545	5336	9A55	05122	5336
E356				5337
5557	5338	9A57	05124	5338
5538	5339	9A58	05125	5339
E357				533A
555D	533B	9A5A	05127	533B
5599	533C	9A5B	05128	533C
5580	533D	9A5C	05129	533D
54AF	533E	9A5D	05130	533E
558A	533F	9A5E	05131	533F
559F	5340	9A5F	05132	5340
557B	5341	9A60	05133	5341
557E	5342	9A61	05134	5342
5598	5343	9A62	05135	5343
559E	5344	9A63	05136	5344
55AE	5345	9A64	05137	5345
557C	5346	9A65	05138	5346
5583	5347	9A66	05139	5347
E358				5348
5587	5349	9A68	05141	5349
55A8	534A	9A69	05142	534A
55DA	534B	9A6A	05143	534B
55C5	534C	9A6B	05144	534C
55DF	534D	9A6C	05145	534D
55C4	534E	9A6D	05146	534E
55DC	534F	9A6E	05147	534F
E359				5350
55D4	5351	9A70	05149	5351
5614	5352	9A71	05150	5352
55F7	5353	9A72	05151	5353
5616	5354	9A73	05152	5354
55FE	5355	9A74	05153	5355
55FD	5356	9A75	05154	5356
561B	5357	9A76	05155	5357
55F9	5358	9A77	05156	5358
564E	5359	9A78	05157	5359
5650	535A	9A79	05158	535A
71DF	535B	9A7A	05159	535B
5634	535C	9A7B	05160	535C
5636	535D	9A7C	05161	535D
E35A				535E
5638	535F	9A7E	05163	535F
566B	5360	9A80	05164	5360
5664	5361	9A81	05165	5361
562F	5362	9A82	05166	5362
566C	5363	9A83	05167	5363
566A	5364	9A84	05168	5364
5686	5365	9A85	05169	5365
5680	5366	9A86	05170	5366
568A	5367	9A87	05171	5367
56A0	5368	9A88	05172	5368
5694	5369	9A89	05173	5369
568F	536A	9A8A	05174	536A
E35B				536B
56AE	536C	9A8C	05176	536C
56B6	536D	9A8D	05177	536D
56B4	536E	9A8E	05178	536E
56C2	536F	9A8F	05179	536F
56BC	5370	9A90	05180	5370
56C1	5371	9A91	05181	5371
56C3	5372	9A92	05182	5372
56C0	5373	9A93	05183	5373
56C8	5374	9A94	05184	5374
56CE	5375	9A95	05185	5375
56D1	5376	9A96	05186	5376
E35C				5377
56D7	5378	9A98	05188	5378
56EE	5379	9A99	05189	5379
56F9	537A	9A9A	05190	537A
5700	537B	9A9B	05191	537B
56FF	537C	9A9C	05192	537C
5704	537D	9A9D	05193	537D
5709	537E	9A9E	05194	537E
5708	5421	9A9F	05201	5421
570B	5422	9AA0	05202	5422
E35D				5423
5713	5424	9AA2	05204	5424
5718	5425	9AA3	05205	5425
5716	5426	9AA4	05206	5426
55C7	5427	9AA5	05207	5427
571C	5428	9AA6	05208	5428
5726	5429	9AA7	05209	5429
5737	542A	9AA8	05210	542A
5738	542B	9AA9	05211	542B
574E	542C	9AAA	05212	542C
573B	542D	9AAB	05213	542D
5740	542E	9AAC	05214	542E
574F	542F	9AAD	05215	542F
5769	5430	9AAE	05216	5430
57C0	5431	9AAF	05217	5431
5788	5432	9AB0	05218	5432
5761	5433	9AB1	05219	5433
577F	5434	9AB2	05220	5434
5789	5435	9AB3	05221	5435
5793	5436	9AB4	05222	5436
57A0	5437	9AB5	05223	5437
57B3	5438	9AB6	05224	5438
57A4	5439	9AB7	05225	5439
57AA	543A	9AB8	05226	543A
57B0	543B	9AB9	05227	543B
57C3	543C	9ABA	05228	543C
57C6	543D	9ABB	05229	543D
57D4	543E	9ABC	05230	543E
57D2	543F	9ABD	05231	543F
57D3	5440	9ABE	05232	5440
580A	5441	9ABF	05233	5441
57D6	5442	9AC0	05234	5442
57E3	5443	9AC1	05235	5443
E35E				5444
5819	5445	9AC3	05237	5445
581D	5446	9AC4	05238	5446
5872	5447	9AC5	05239	5447
5821	5448	9AC6	05240	5448
5862	5449	9AC7	05241	5449
584B	544A	9AC8	05242	544A
5870	544B	9AC9	05243	544B
6BC0	544C	9ACA	05244	544C
5852	544D	9ACB	05245	544D
583D	544E	9ACC	05246	544E
5879	544F	9ACD	05247	544F
5885	5450	9ACE	05248	5450
58B9	5451	9ACF	05249	5451
589F	5452	9AD0	05250	5452
58AB	5453	9AD1	05251	5453
58BA	5454	9AD2	05252	5454
58DE	5455	9AD3	05253	5455
58BB	5456	9AD4	05254	5456
58B8	5457	9AD5	05255	5457
58AE	5458	9AD6	05256	5458
58C5	5459	9AD7	05257	5459
58D3	545A	9AD8	05258	545A
58D1	545B	9AD9	05259	545B
58D7	545C	9ADA	05260	545C
58D9	545D	9ADB	05261	545D
58D8	545E	9ADC	05262	545E
58E5	545F	9ADD	05263	545F
58DC	5460	9ADE	05264	5460
58E4	5461	9ADF	05265	5461
58DF	5462	9AE0	05266	5462
58EF	5463	9AE1	05267	5463
58F7	445B	92D9	03659	5464
58F9	5465	9AE3	05269	5465
58FB	5466	9AE4	05270	5466
58FC	5467	9AE5	05271	5467
58FD	5468	9AE6	05272	5468
5902	5469	9AE7	05273	5469
E35F				546A
5910	546B	9AE9	05275	546B
591B	546C	9AEA	05276	546C
68A6	546D	9AEB	05277	546D
5925	546E	9AEC	05278	546E
592C	546F	9AED	05279	546F
592D	5470	9AEE	05280	5470
5932	5471	9AEF	05281	5471
5938	5472	9AF0	05282	5472
593E	5473	9AF1	05283	5473
7AD2	5474	9AF2	05284	5474
5955	5475	9AF3	05285	5475
5950	5476	9AF4	05286	5476
594E	5477	9AF5	05287	5477
595A	5478	9AF6	05288	5478
5958	5479	9AF7	05289	5479
5962	547A	9AF8	05290	547A
5960	547B	9AF9	05291	547B
5967	547C	9AFA	05292	547C
596C	547D	9AFB	05293	547D
5969	547E	9AFC	05294	547E
5978	5521	9B40	05301	5521
5981	5522	9B41	05302	5522
599D	5523	9B42	05303	5523
4F5E	5524	9B43	05304	5524
4FAB	5525	9B44	05305	5525
59A3	5526	9B45	05306	5526
59B2	5527	9B46	05307	5527
59C6	5528	9B47	05308	5528
59E8	5529	9B48	05309	5529
59DC	552A	9B49	05310	552A
598D	552B	9B4A	05311	552B
59D9	552C	9B4B	05312	552C
59DA	552D	9B4C	05313	552D
5A25	552E	9B4D	05314	552E
5A1F	552F	9B4E	05315	552F
5A11	5530	9B4F	05316	5530
5A1C	5531	9B50	05317	5531
5A09	5532	9B51	05318	5532
5A1A	5533	9B52	05319	5533
5A40	5534	9B53	05320	5534
5A6C	5535	9B54	05321	5535
5A49	5536	9B55	05322	5536
5A35	5537	9B56	05323	5537
5A36	5538	9B57	05324	5538
5A62	5539	9B58	05325	5539
5A6A	553A	9B59	05326	553A
5A9A	553B	9B5A	05327	553B
5ABC	553C	9B5B	05328	553C
E360				553D
5ACB	553E	9B5D	05330	553E
5AC2	553F	9B5E	05331	553F
5ABD	5540	9B5F	05332	5540
5AE3	5541	9B60	05333	5541
5AD7	5542	9B61	05334	5542
5AE6	5543	9B62	05335	5543
5AE9	5544	9B63	05336	5544
5AD6	5545	9B64	05337	5545
5AFA	5546	9B65	05338	5546
5AFB	5547	9B66	05339	5547
5B0C	5548	9B67	05340	5548
5B0B	5549	9B68	05341	5549
5B16	554A	9B69	05342	554A
5B32	554B	9B6A	05343	554B
5AD0	554C	9B6B	05344	554C
5B2A	554D	9B6C	05345	554D
5B36	554E	9B6D	05346	554E
5B3E	554F	9B6E	05347	554F
5B43	5550	9B6F	05348	5550
5B45	5551	9B70	05349	5551
5B40	5552	9B71	05350	5552
5B51	5553	9B72	05351	5553
5B55	5554	9B73	05352	5554
5B5A	5555	9B74	05353	5555
5B5B	5556	9B75	05354	5556
5B65	5557	9B76	05355	5557
5B69	5558	9B77	05356	5558
E361				5559
5B73	555A	9B79	05358	555A
5B75	555B	9B7A	05359	555B
5B78	555C	9B7B	05360	555C
E362				555D
5B7A	555E	9B7D	05362	555E
5B80	555F	9B7E	05363	555F
5B83	5560	9B80	05364	5560
5BA6	5561	9B81	05365	5561
5BB8	5562	9B82	05366	5562
E363				5563
5BC7	5564	9B84	05368	5564
5BC9	5565	9B85	05369	5565
5BD4	5566	9B86	05370	5566
5BD0	5567	9B87	05371	5567
5BE4	5568	9B88	05372	5568
5BE6	5569	9B89	05373	5569
5BE2	556A	9B8A	05374	556A
5BDE	556B	9B8B	05375	556B
5BE5	556C	9B8C	05376	556C
5BEB	556D	9B8D	05377	556D
5BF0	556E	9B8E	05378	556E
5BF6	556F	9B8F	05379	556F
5BF3	5570	9B90	05380	5570
5C05	5571	9B91	05381	5571
5C07	5572	9B92	05382	5572
5C08	5573	9B93	05383	5573
5C0D	5574	9B94	05384	5574
5C13	5575	9B95	05385	5575
5C20	5576	9B96	05386	5576
5C22	5577	9B97	05387	5577
5C28	5578	9B98	05388	5578
5C38	5579	9B99	05389	5579
5C39	557A	9B9A	05390	557A
5C41	557B	9B9B	05391	557B
5C46	557C	9B9C	05392	557C
5C4E	557D	9B9D	05393	557D
5C53	557E	9B9E	05394	557E
5C50	5621	9B9F	05401	5621
E364				5622
5B71	5623	9BA1	05403	5623
5C6C	5624	9BA2	05404	5624
5C6E	5625	9BA3	05405	5625
4E62	5626	9BA4	05406	5626
5C76	5627	9BA5	05407	5627
5C79	5628	9BA6	05408	5628
5C8C	5629	9BA7	05409	5629
5C91	562A	9BA8	05410	562A
5C94	562B	9BA9	05411	562B
599B	562C	9BAA	05412	562C
5CAB	562D	9BAB	05413	562D
5CBB	562E	9BAC	05414	562E
5CB6	562F	9BAD	05415	562F
5CBC	5630	9BAE	05416	5630
5CB7	5631	9BAF	05417	5631
5CC5	5632	9BB0	05418	5632
5CBE	5633	9BB1	05419	5633
5CC7	5634	9BB2	05420	5634
5CD9	5635	9BB3	05421	5635
5CE9	5636	9BB4	05422	5636
5CFD	5637	9BB5	05423	5637
E365				5638
5CED	5639	9BB7	05425	5639
5D8C	563A	9BB8	05426	563A
5CEA	563B	9BB9	05427	563B
5D0B	563C	9BBA	05428	563C
5D15	563D	9BBB	05429	563D
5D17	563E	9BBC	05430	563E
5D5C	563F	9BBD	05431	563F
5D1F	5640	9BBE	05432	5640
5D1B	5641	9BBF	05433	5641
5D11	5642	9BC0	05434	5642
5D14	5643	9BC1	05435	5643
5D22	5644	9BC2	05436	5644
5D1A	5645	9BC3	05437	5645
5D19	5646	9BC4	05438	5646
5D18	5647	9BC5	05439	5647
5D4C	5648	9BC6	05440	5648
5D52	5649	9BC7	05441	5649
5D4E	564A	9BC8	05442	564A
5D4B	564B	9BC9	05443	564B
5D6C	564C	9BCA	05444	564C
5D73	564D	9BCB	05445	564D
5D76	564E	9BCC	05446	564E
5D87	564F	9BCD	05447	564F
5D84	5650	9BCE	05448	5650
5D82	5651	9BCF	05449	5651
5DA2	5652	9BD0	05450	5652
5D9D	5653	9BD1	05451	5653
5DAC	5654	9BD2	05452	5654
5DAE	5655	9BD3	05453	5655
5DBD	5656	9BD4	05454	5656
5D90	5657	9BD5	05455	5657
5DB7	5658	9BD6	05456	5658
5DBC	5659	9BD7	05457	5659
E366				565A
5DCD	565B	9BD9	05459	565B
5DD3	565C	9BDA	05460	565C
5DD2	565D	9BDB	05461	565D
5DD6	565E	9BDC	05462	565E
5DDB	565F	9BDD	05463	565F
5DEB	5660	9BDE	05464	5660
5DF2	5661	9BDF	05465	5661
5DF5	5662	9BE0	05466	5662
5E0B	5663	9BE1	05467	5663
5E1A	5664	9BE2	05468	5664
5E19	5665	9BE3	05469	5665
5E11	5666	9BE4	05470	5666
5E1B	5667	9BE5	05471	5667
5E36	5668	9BE6	05472	5668
5E37	5669	9BE7	05473	5669
5E44	566A	9BE8	05474	566A
5E43	566B	9BE9	05475	566B
5E40	566C	9BEA	05476	566C
5E4E	566D	9BEB	05477	566D
5E57	566E	9BEC	05478	566E
5E54	566F	9BED	05479	566F
5E5F	5670	9BEE	05480	5670
5E62	5671	9BEF	05481	5671
E367				5672
5E47	5673	9BF1	05483	5673
5E75	5674	9BF2	05484	5674
5E76	5675	9BF3	05485	5675
5E7A	5676	9BF4	05486	5676
9EBC	5677	9BF5	05487	5677
5E7F	5678	9BF6	05488	5678
5EA0	5679	9BF7	05489	5679
5EC1	567A	9BF8	05490	567A
5EC2	567B	9BF9	05491	567B
5EC8	567C	9BFA	05492	567C
5ED0	567D	9BFB	05493	567D
5ECF	567E	9BFC	05494	567E
5ED6	5721	9C40	05501	5721
5EE3	5722	9C41	05502	5722
5EDD	5723	9C42	05503	5723
5EDA	5724	9C43	05504	5724
5EDB	5725	9C44	05505	5725
E368				5726
5EE1	5727	9C46	05507	5727
5EE8	5728	9C47	05508	5728
5EE9	5729	9C48	05509	5729
5EEC	572A	9C49	05510	572A
5EF1	572B	9C4A	05511	572B
5EF3	572C	9C4B	05512	572C
5EF0	572D	9C4C	05513	572D
E369				572E
E36A				572F
5EFE	5730	9C4F	05516	5730
5F03	5731	9C50	05517	5731
5F09	5732	9C51	05518	5732
5F5D	5733	9C52	05519	5733
5F5C	5734	9C53	05520	5734
5F0B	5735	9C54	05521	5735
5F11	5736	9C55	05522	5736
5F16	5737	9C56	05523	5737
5F29	5738	9C57	05524	5738
E36B				5739
5F38	573A	9C59	05526	573A
5F41	573B	9C5A	05527	573B
5F48	573C	9C5B	05528	573C
5F4C	573D	9C5C	05529	573D
5F4E	573E	9C5D	05530	573E
5F2F	573F	9C5E	05531	573F
5F51	5740	9C5F	05532	5740
E36C				5741
5F57	5742	9C61	05534	5742
5F59	5743	9C62	05535	5743
5F61	5744	9C63	05536	5744
5F6D	5745	9C64	05537	5745
5F73	5746	9C65	05538	5746
5F77	5747	9C66	05539	5747
5F83	5748	9C67	05540	5748
5F82	5749	9C68	05541	5749
5F7F	574A	9C69	05542	574A
5F8A	574B	9C6A	05543	574B
5F88	574C	9C6B	05544	574C
5F91	574D	9C6C	05545	574D
5F87	574E	9C6D	05546	574E
5F9E	574F	9C6E	05547	574F
5F99	5750	9C6F	05548	5750
5F98	5751	9C70	05549	5751
5FA0	5752	9C71	05550	5752
5FA8	5753	9C72	05551	5753
5FAD	5754	9C73	05552	5754
5FBC	5755	9C74	05553	5755
5FD6	5756	9C75	05554	5756
5FFB	5757	9C76	05555	5757
5FE4	5758	9C77	05556	5758
5FF8	5759	9C78	05557	5759
5FF1	575A	9C79	05558	575A
5FDD	575B	9C7A	05559	575B
E36D				575C
5FFF	575D	9C7C	05561	575D
6021	575E	9C7D	05562	575E
6060	575F	9C7E	05563	575F
6019	5760	9C80	05564	5760
6010	5761	9C81	05565	5761
6029	5762	9C82	05566	5762
600E	5763	9C83	05567	5763
6031	5764	9C84	05568	5764
601B	5765	9C85	05569	5765
6015	5766	9C86	05570	5766
602B	5767	9C87	05571	5767
6026	5768	9C88	05572	5768
600F	5769	9C89	05573	5769
603A	576A	9C8A	05574	576A
605A	576B	9C8B	05575	576B
6041	576C	9C8C	05576	576C
606A	576D	9C8D	05577	576D
6077	576E	9C8E	05578	576E
605F	576F	9C8F	05579	576F
604A	5770	9C90	05580	5770
6046	5771	9C91	05581	5771
604D	5772	9C92	05582	5772
6063	5773	9C93	05583	5773
6043	5774	9C94	05584	5774
6064	5775	9C95	05585	5775
6042	5776	9C96	05586	5776
606C	5777	9C97	05587	5777
606B	5778	9C98	05588	5778
6059	5779	9C99	05589	5779
6081	577A	9C9A	05590	577A
608D	577B	9C9B	05591	577B
60E7	577C	9C9C	05592	577C
6083	577D	9C9D	05593	577D
609A	577E	9C9E	05594	577E
6084	5821	9C9F	05601	5821
609B	5822	9CA0	05602	5822
6096	5823	9CA1	05603	5823
E36E				5824
6092	5825	9CA3	05605	5825
60A7	5826	9CA4	05606	5826
E36F				5827
60E1	5828	9CA6	05608	5828
60B8	5829	9CA7	05609	5829
60E0	582A	9CA8	05610	582A
60D3	582B	9CA9	05611	582B
60B4	582C	9CAA	05612	582C
5FF0	582D	9CAB	05613	582D
60BD	582E	9CAC	05614	582E
60C6	582F	9CAD	05615	582F
60B5	5830	9CAE	05616	5830
60D8	5831	9CAF	05617	5831
614D	5832	9CB0	05618	5832
6115	5833	9CB1	05619	5833
6106	5834	9CB2	05620	5834
60F6	5835	9CB3	05621	5835
60F7	5836	9CB4	05622	5836
6100	5837	9CB5	05623	5837
60F4	5838	9CB6	05624	5838
60FA	5839	9CB7	05625	5839
6103	583A	9CB8	05626	583A
6121	583B	9CB9	05627	583B
60FB	583C	9CBA	05628	583C
60F1	583D	9CBB	05629	583D
610D	583E	9CBC	05630	583E
610E	583F	9CBD	05631	583F
6147	5840	9CBE	05632	5840
613E	5841	9CBF	05633	5841
6128	5842	9CC0	05634	5842
6127	5843	9CC1	05635	5843
614A	5844	9CC2	05636	5844
613F	5845	9CC3	05637	5845
613C	5846	9CC4	05638	5846
612C	5847	9CC5	05639	5847
6134	5848	9CC6	05640	5848
613D	5849	9CC7	05641	5849
6142	584A	9CC8	05642	584A
6144	584B	9CC9	05643	584B
6173	584C	9CCA	05644	584C
6177	584D	9CCB	05645	584D
6158	584E	9CCC	05646	584E
6159	584F	9CCD	05647	584F
615A	5850	9CCE	05648	5850
616B	5851	9CCF	05649	5851
6174	5852	9CD0	05650	5852
616F	5853	9CD1	05651	5853
6165	5854	9CD2	05652	5854
6171	5855	9CD3	05653	5855
615F	5856	9CD4	05654	5856
615D	5857	9CD5	05655	5857
6153	5858	9CD6	05656	5858
6175	5859	9CD7	05657	5859
6199	585A	9CD8	05658	585A
6196	585B	9CD9	05659	585B
6187	585C	9CDA	05660	585C
61AC	585D	9CDB	05661	585D
6194	585E	9CDC	05662	585E
619A	585F	9CDD	05663	585F
618A	5860	9CDE	05664	5860
6191	5861	9CDF	05665	5861
E370				5862
61AE	5863	9CE1	05667	5863
61CC	5864	9CE2	05668	5864
61CA	5865	9CE3	05669	5865
61C9	5866	9CE4	05670	5866
61F7	5867	9CE5	05671	5867
61C8	5868	9CE6	05672	5868
61C3	5869	9CE7	05673	5869
61C6	586A	9CE8	05674	586A
61BA	586B	9CE9	05675	586B
61CB	586C	9CEA	05676	586C
7F79	586D	9CEB	05677	586D
61CD	586E	9CEC	05678	586E
61E6	586F	9CED	05679	586F
61E3	5870	9CEE	05680	5870
61F6	5871	9CEF	05681	5871
61FA	5872	9CF0	05682	5872
61F4	5873	9CF1	05683	5873
61FF	5874	9CF2	05684	5874
61FD	5875	9CF3	05685	5875
61FC	5876	9CF4	05686	5876
61FE	5877	9CF5	05687	5877
6200	5878	9CF6	05688	5878
6208	5879	9CF7	05689	5879
6209	587A	9CF8	05690	587A
620D	587B	9CF9	05691	587B
620C	587C	9CFA	05692	587C
6214	587D	9CFB	05693	587D
621B	587E	9CFC	05694	587E
621E	5921	9D40	05701	5921
6221	5922	9D41	05702	5922
622A	5923	9D42	05703	5923
622E	5924	9D43	05704	5924
6230	5925	9D44	05705	5925
6232	5926	9D45	05706	5926
6233	5927	9D46	05707	5927
6241	5928	9D47	05708	5928
624E	5929	9D48	05709	5929
625E	592A	9D49	05710	592A
6263	592B	9D4A	05711	592B
625B	592C	9D4B	05712	592C
E371				592D
E372				592E
627C	592F	9D4E	05715	592F
6282	5930	9D4F	05716	5930
6289	5931	9D50	05717	5931
627E	5932	9D51	05718	5932
6292	5933	9D52	05719	5933
6293	5934	9D53	05720	5934
6296	5935	9D54	05721	5935
62D4	5936	9D55	05722	5936
6283	5937	9D56	05723	5937
6294	5938	9D57	05724	5938
62D7	5939	9D58	05725	5939
62D1	593A	9D59	05726	593A
62BB	593B	9D5A	05727	593B
62CF	593C	9D5B	05728	593C
62FF	593D	9D5C	05729	593D
62C6	593E	9D5D	05730	593E
64D4	593F	9D5E	05731	593F
62C8	5940	9D5F	05732	5940
62DC	5941	9D60	05733	5941
62CC	5942	9D61	05734	5942
62CA	5943	9D62	05735	5943
62C2	5944	9D63	05736	5944
62C7	5945	9D64	05737	5945
629B	5946	9D65	05738	5946
62C9	5947	9D66	05739	5947
630C	5948	9D67	05740	5948
62EE	5949	9D68	05741	5949
62F1	594A	9D69	05742	594A
6327	594B	9D6A	05743	594B
6302	594C	9D6B	05744	594C
6308	594D	9D6C	05745	594D
62EF	594E	9D6D	05746	594E
62F5	594F	9D6E	05747	594F
6350	5950	9D6F	05748	5950
633E	5951	9D70	05749	5951
634D	5952	9D71	05750	5952
641C	5953	9D72	05751	5953
634F	5954	9D73	05752	5954
6396	5955	9D74	05753	5955
638E	5956	9D75	05754	5956
6380	5957	9D76	05755	5957
63AB	5958	9D77	05756	5958
6376	5959	9D78	05757	5959
63A3	595A	9D79	05758	595A
638F	595B	9D7A	05759	595B
6389	595C	9D7B	05760	595C
639F	595D	9D7C	05761	595D
63B5	595E	9D7D	05762	595E
636B	595F	9D7E	05763	595F
E373				5960
63BE	5961	9D81	05765	5961
63E9	5962	9D82	05766	5962
63C0	5963	9D83	05767	5963
E374				5964
63E3	5965	9D85	05769	5965
63C9	5966	9D86	05770	5966
63D2	5967	9D87	05771	5967
63F6	5968	9D88	05772	5968
E375				5969
6416	596A	9D8A	05774	596A
6434	596B	9D8B	05775	596B
E376				596C
6413	596D	9D8D	05777	596D
6426	596E	9D8E	05778	596E
6436	596F	9D8F	05779	596F
651D	5970	9D90	05780	5970
6417	5971	9D91	05781	5971
6428	5972	9D92	05782	5972
640F	5973	9D93	05783	5973
6467	5974	9D94	05784	5974
E377				5975
6476	5976	9D96	05786	5976
644E	5977	9D97	05787	5977
64B9	3349	8A68	01941	5978
6495	5979	9D99	05789	5979
6493	597A	9D9A	05790	597A
E378				597B
64A9	597C	9D9C	05792	597C
6488	597D	9D9D	05793	597D
64BC	597E	9D9E	05794	597E
64DA	5A21	9D9F	05801	5A21
64D2	5A22	9DA0	05802	5A22
64C5	5A23	9DA1	05803	5A23
64C7	5A24	9DA2	05804	5A24
64BB	5A25	9DA3	05805	5A25
64D8	5A26	9DA4	05806	5A26
64C2	5A27	9DA5	05807	5A27
64F1	5A28	9DA6	05808	5A28
64E7	5A29	9DA7	05809	5A29
8209	5A2A	9DA8	05810	5A2A
64E0	5A2B	9DA9	05811	5A2B
64E1	5A2C	9DAA	05812	5A2C
62AC	5A2D	9DAB	05813	5A2D
64E3	5A2E	9DAC	05814	5A2E
64EF	5A2F	9DAD	05815	5A2F
652C	5A30	9DAE	05816	5A30
64F6	5A31	9DAF	05817	5A31
64F4	5A32	9DB0	05818	5A32
64F2	5A33	9DB1	05819	5A33
64FA	5A34	9DB2	05820	5A34
6500	5A35	9DB3	05821	5A35
64FD	5A36	9DB4	05822	5A36
6518	5A37	9DB5	05823	5A37
651C	5A38	9DB6	05824	5A38
6522				5A39
6524	5A3A	9DB8	05826	5A3A
6523	5A3B	9DB9	05827	5A3B
652B	5A3C	9DBA	05828	5A3C
6534	5A3D	9DBB	05829	5A3D
6535	5A3E	9DBC	05830	5A3E
6537	5A3F	9DBD	05831	5A3F
6536	5A40	9DBE	05832	5A40
6538	5A41	9DBF	05833	5A41
754B	5A42	9DC0	05834	5A42
6548	5A43	9DC1	05835	5A43
6556	5A44	9DC2	05836	5A44
6555	5A45	9DC3	05837	5A45
654D	5A46	9DC4	05838	5A46
6558	5A47	9DC5	05839	5A47
655E	5A48	9DC6	05840	5A48
655D	5A49	9DC7	05841	5A49
6572	5A4A	9DC8	05842	5A4A
6578	5A4B	9DC9	05843	5A4B
6582	5A4C	9DCA	05844	5A4C
E379				5A4D
8B8A	5A4E	9DCC	05846	5A4E
659B	5A4F	9DCD	05847	5A4F
659F	5A50	9DCE	05848	5A50
65AB	5A51	9DCF	05849	5A51
65B7	5A52	9DD0	05850	5A52
65C3	5A53	9DD1	05851	5A53
65C6	5A54	9DD2	05852	5A54
65C1	5A55	9DD3	05853	5A55
65C4	5A56	9DD4	05854	5A56
65CC	5A57	9DD5	05855	5A57
65D2	5A58	9DD6	05856	5A58
65DB	5A59	9DD7	05857	5A59
65D9	5A5A	9DD8	05858	5A5A
65E0	5A5B	9DD9	05859	5A5B
65E1	5A5C	9DDA	05860	5A5C
65F1	5A5D	9DDB	05861	5A5D
6772	5A5E	9DDC	05862	5A5E
660A	5A5F	9DDD	05863	5A5F
6603	5A60	9DDE	05864	5A60
E37A				5A61
6773	5A62	9DE0	05866	5A62
6635	5A63	9DE1	05867	5A63
E37B				5A64
6634	5A65	9DE3	05869	5A65
661C	5A66	9DE4	05870	5A66
664F	5A67	9DE5	05871	5A67
6644	5A68	9DE6	05872	5A68
6649	5A69	9DE7	05873	5A69
6641	5A6A	9DE8	05874	5A6A
665E	5A6B	9DE9	05875	5A6B
665D	5A6C	9DEA	05876	5A6C
6664	5A6D	9DEB	05877	5A6D
6667	5A6E	9DEC	05878	5A6E
6668	5A6F	9DED	05879	5A6F
665F	5A70	9DEE	05880	5A70
6662	5A71	9DEF	05881	5A71
6670	5A72	9DF0	05882	5A72
6683	5A73	9DF1	05883	5A73
6688	5A74	9DF2	05884	5A74
668E	5A75	9DF3	05885	5A75
6689	5A76	9DF4	05886	5A76
6684	5A77	9DF5	05887	5A77
6698	5A78	9DF6	05888	5A78
669D	5A79	9DF7	05889	5A79
66C1	5A7A	9DF8	05890	5A7A
66B9	5A7B	9DF9	05891	5A7B
66C9	5A7C	9DFA	05892	5A7C
66BE	5A7D	9DFB	05893	5A7D
66BC	5A7E	9DFC	05894	5A7E
66C4	5B21	9E40	05901	5B21
66B8	5B22	9E41	05902	5B22
66D6	5B23	9E42	05903	5B23
66DA	5B24	9E43	05904	5B24
66E0	5B25	9E44	05905	5B25
663F	5B26	9E45	05906	5B26
66E6	5B27	9E46	05907	5B27
66E9	5B28	9E47	05908	5B28
66F0	5B29	9E48	05909	5B29
66F5	5B2A	9E49	05910	5B2A
66F7	5B2B	9E4A	05911	5B2B
670F	5B2C	9E4B	05912	5B2C
6716	5B2D	9E4C	05913	5B2D
671E	5B2E	9E4D	05914	5B2E
6726	5B2F	9E4E	05915	5B2F
6727	5B30	9E4F	05916	5B30
9738	5B31	9E50	05917	5B31
672E	5B32	9E51	05918	5B32
673F	5B33	9E52	05919	5B33
6736	5B34	9E53	05920	5B34
6741	5B35	9E54	05921	5B35
6738	5B36	9E55	05922	5B36
6737	5B37	9E56	05923	5B37
6746	5B38	9E57	05924	5B38
675E	5B39	9E58	05925	5B39
6760	5B3A	9E59	05926	5B3A
6759	5B3B	9E5A	05927	5B3B
6763	5B3C	9E5B	05928	5B3C
6764	5B3D	9E5C	05929	5B3D
6789	5B3E	9E5D	05930	5B3E
6770	5B3F	9E5E	05931	5B3F
67A9	5B40	9E5F	05932	5B40
677C	5B41	9E60	05933	5B41
676A	5B42	9E61	05934	5B42
678C	5B43	9E62	05935	5B43
678B	5B44	9E63	05936	5B44
E37C				5B45
67A1	5B46	9E65	05938	5B46
6785	5B47	9E66	05939	5B47
67B7	5B48	9E67	05940	5B48
67EF	5B49	9E68	05941	5B49
E37D				5B4A
67EC	5B4B	9E6A	05943	5B4B
67B3	5B4C	9E6B	05944	5B4C
67E9	5B4D	9E6C	05945	5B4D
67B8	5B4E	9E6D	05946	5B4E
67E4	5B4F	9E6E	05947	5B4F
67DE	5B50	9E6F	05948	5B50
67DD	5B51	9E70	05949	5B51
67E2	5B52	9E71	05950	5B52
67EE	5B53	9E72	05951	5B53
67B9	5B54	9E73	05952	5B54
67CE	5B55	9E74	05953	5B55
67C6	5B56	9E75	05954	5B56
67E7	5B57	9E76	05955	5B57
6867	4930	954F	04116	5B58
681E	5B59	9E78	05957	5B59
6846	5B5A	9E79	05958	5B5A
6829	5B5B	9E7A	05959	5B5B
6840	5B5C	9E7B	05960	5B5C
684D	5B5D	9E7C	05961	5B5D
6832	5B5E	9E7D	05962	5B5E
684E	5B5F	9E7E	05963	5B5F
68B3	5B60	9E80	05964	5B60
682B	5B61	9E81	05965	5B61
6859	5B62	9E82	05966	5B62
6863	5B63	9E83	05967	5B63
6877	5B64	9E84	05968	5B64
687F	5B65	9E85	05969	5B65
689F	5B66	9E86	05970	5B66
688F	5B67	9E87	05971	5B67
68AD	5B68	9E88	05972	5B68
6894	5B69	9E89	05973	5B69
689D	5B6A	9E8A	05974	5B6A
E37E				5B6B
E37F				5B6C
68BC	456E	938E	03778	5B6D
68B9	5B6E	9E8E	05978	5B6E
6874	5B6F	9E8F	05979	5B6F
68B5	5B70	9E90	05980	5B70
68A0	5B71	9E91	05981	5B71
68BA	5B72	9E92	05982	5B72
690F	5B73	9E93	05983	5B73
E380				5B74
687E	5B75	9E95	05985	5B75
6901	5B76	9E96	05986	5B76
68CA	5B77	9E97	05987	5B77
6908	5B78	9E98	05988	5B78
68D8	5B79	9E99	05989	5B79
6922	5B7A	9E9A	05990	5B7A
6926	5B7B	9E9B	05991	5B7B
68E1	5B7C	9E9C	05992	5B7C
690C	5B7D	9E9D	05993	5B7D
68CD	5B7E	9E9E	05994	5B7E
68D4	5C21	9E9F	06001	5C21
68E7	5C22	9EA0	06002	5C22
68D5	5C23	9EA1	06003	5C23
6936	5C24	9EA2	06004	5C24
6912	5C25	9EA3	06005	5C25
6904	5C26	9EA4	06006	5C26
E381				5C27
68E3	5C28	9EA6	06008	5C28
6925	5C29	9EA7	06009	5C29
68F9	5C2A	9EA8	06010	5C2A
68E0	5C2B	9EA9	06011	5C2B
68EF	5C2C	9EAA	06012	5C2C
6928	5C2D	9EAB	06013	5C2D
692A	5C2E	9EAC	06014	5C2E
691A	5C2F	9EAD	06015	5C2F
6923	5C30	9EAE	06016	5C30
6921	5C31	9EAF	06017	5C31
68C6	5C32	9EB0	06018	5C32
6979	5C33	9EB1	06019	5C33
6977	5C34	9EB2	06020	5C34
695C	5C35	9EB3	06021	5C35
6978	5C36	9EB4	06022	5C36
696B	5C37	9EB5	06023	5C37
6954	5C38	9EB6	06024	5C38
697E	5C39	9EB7	06025	5C39
696E	5C3A	9EB8	06026	5C3A
6939	5C3B	9EB9	06027	5C3B
6974	5C3C	9EBA	06028	5C3C
693D	5C3D	9EBB	06029	5C3D
6959	5C3E	9EBC	06030	5C3E
6930	5C3F	9EBD	06031	5C3F
E382				5C40
695E	5C41	9EBF	06033	5C41
695D	5C42	9EC0	06034	5C42
6981	5C43	9EC1	06035	5C43
696A	5C44	9EC2	06036	5C44
69B2	5C45	9EC3	06037	5C45
69AE	5C46	9EC4	06038	5C46
69D0	5C47	9EC5	06039	5C47
69BF	5C48	9EC6	06040	5C48
69C1	5C49	9EC7	06041	5C49
69D3	5C4A	9EC8	06042	5C4A
69BE	5C4B	9EC9	06043	5C4B
69CE	5C4C	9ECA	06044	5C4C
5BE8	5C4D	9ECB	06045	5C4D
69CA	5C4E	9ECC	06046	5C4E
69DD	5C4F	9ECD	06047	5C4F
69BB	5C50	9ECE	06048	5C50
69C3	5C51	9ECF	06049	5C51
69A7	5C52	9ED0	06050	5C52
6A2E	5C53	9ED1	06051	5C53
6991	5C54	9ED2	06052	5C54
69A0	5C55	9ED3	06053	5C55
699C	5C56	9ED4	06054	5C56
6995	5C57	9ED5	06055	5C57
69B4	5C58	9ED6	06056	5C58
69DE	5C59	9ED7	06057	5C59
69E8	5C5A	9ED8	06058	5C5A
6A02	5C5B	9ED9	06059	5C5B
6A1B	5C5C	9EDA	06060	5C5C
69FF	5C5D	9EDB	06061	5C5D
6B0A	5C5E	9EDC	06062	5C5E
69F9	5C5F	9EDD	06063	5C5F
69F2	5C60	9EDE	06064	5C60
69E7	5C61	9EDF	06065	5C61
6A05	5C62	9EE0	06066	5C62
69B1	5C63	9EE1	06067	5C63
6A1E	5C64	9EE2	06068	5C64
69ED	5C65	9EE3	06069	5C65
6A14	5C66	9EE4	06070	5C66
69EB	5C67	9EE5	06071	5C67
6A0A	5C68	9EE6	06072	5C68
6A12	5C69	9EE7	06073	5C69
6AC1	5C6A	9EE8	06074	5C6A
6A23	5C6B	9EE9	06075	5C6B
6A13	5C6C	9EEA	06076	5C6C
6A44	5C6D	9EEB	06077	5C6D
6A0C	5C6E	9EEC	06078	5C6E
6A72	5C6F	9EED	06079	5C6F
6A36	5C70	9EEE	06080	5C70
6A78	5C71	9EEF	06081	5C71
6A47	5C72	9EF0	06082	5C72
6A62	5C73	9EF1	06083	5C73
6A59	5C74	9EF2	06084	5C74
6A66	5C75	9EF3	06085	5C75
6A48	5C76	9EF4	06086	5C76
6A38	5C77	9EF5	06087	5C77
6A22	5C78	9EF6	06088	5C78
E383				5C79
6A8D	5C7A	9EF8	06090	5C7A
6AA0	5C7B	9EF9	06091	5C7B
6A84	5C7C	9EFA	06092	5C7C
6AA2	5C7D	9EFB	06093	5C7D
6AA3	5C7E	9EFC	06094	5C7E
6A97	5D21	9F40	06101	5D21
8617	5D22	9F41	06102	5D22
6ABB	5D23	9F42	06103	5D23
6AC3	5D24	9F43	06104	5D24
6AC2	5D25	9F44	06105	5D25
6AB8	5D26	9F45	06106	5D26
6AB3	5D27	9F46	06107	5D27
6AAC	5D28	9F47	06108	5D28
E384				5D29
6AD1	5D2A	9F49	06110	5D2A
6ADF	5D2B	9F4A	06111	5D2B
6AAA	5D2C	9F4B	06112	5D2C
6ADA	5D2D	9F4C	06113	5D2D
6AEA	5D2E	9F4D	06114	5D2E
6AFB	5D2F	9F4E	06115	5D2F
6B05	5D30	9F4F	06116	5D30
8616	5D31	9F50	06117	5D31
6AFA	5D32	9F51	06118	5D32
6B12	5D33	9F52	06119	5D33
6B16	5D34	9F53	06120	5D34
9B31	5D35	9F54	06121	5D35
6B1F	5D36	9F55	06122	5D36
6B38	5D37	9F56	06123	5D37
6B37	5D38	9F57	06124	5D38
76DC	5D39	9F58	06125	5D39
6B39	5D3A	9F59	06126	5D3A
98EE	5D3B	9F5A	06127	5D3B
6B47	5D3C	9F5B	06128	5D3C
6B43	5D3D	9F5C	06129	5D3D
6B49	5D3E	9F5D	06130	5D3E
6B50	5D3F	9F5E	06131	5D3F
6B59	5D40	9F5F	06132	5D40
6B54	5D41	9F60	06133	5D41
6B5B	5D42	9F61	06134	5D42
6B5F	5D43	9F62	06135	5D43
6B61	5D44	9F63	06136	5D44
6B78	5D45	9F64	06137	5D45
6B79	5D46	9F65	06138	5D46
6B7F	5D47	9F66	06139	5D47
6B80	5D48	9F67	06140	5D48
6B84	5D49	9F68	06141	5D49
6B83	5D4A	9F69	06142	5D4A
6B8D	5D4B	9F6A	06143	5D4B
6B98	5D4C	9F6B	06144	5D4C
6B95	5D4D	9F6C	06145	5D4D
6B9E	5D4E	9F6D	06146	5D4E
6BA4	5D4F	9F6E	06147	5D4F
6BAA	5D50	9F6F	06148	5D50
6BAB	5D51	9F70	06149	5D51
6BAF	5D52	9F71	06150	5D52
6BB2	5D53	9F72	06151	5D53
6BB1	5D54	9F73	06152	5D54
6BB3	5D55	9F74	06153	5D55
6BB7	5D56	9F75	06154	5D56
6BBC	5D57	9F76	06155	5D57
6BC6	5D58	9F77	06156	5D58
6BCB	5D59	9F78	06157	5D59
6BD3	5D5A	9F79	06158	5D5A
6BDF	5D5B	9F7A	06159	5D5B
6BEC	5D5C	9F7B	06160	5D5C
6BEB	5D5D	9F7C	06161	5D5D
6BF3	5D5E	9F7D	06162	5D5E
6BEF	5D5F	9F7E	06163	5D5F
9EBE	5D60	9F80	06164	5D60
E385				5D61
6C13	5D62	9F82	06166	5D62
6C14	5D63	9F83	06167	5D63
6C1B	5D64	9F84	06168	5D64
6C24	5D65	9F85	06169	5D65
6C23	5D66	9F86	06170	5D66
6C5E	5D67	9F87	06171	5D67
6C55	5D68	9F88	06172	5D68
6C62	5D69	9F89	06173	5D69
6C6A	5D6A	9F8A	06174	5D6A
6C82	5D6B	9F8B	06175	5D6B
6C8D	5D6C	9F8C	06176	5D6C
6C9A	5D6D	9F8D	06177	5D6D
6C81	5D6E	9F8E	06178	5D6E
6C9B	5D6F	9F8F	06179	5D6F
6C7E	5D70	9F90	06180	5D70
6C68	5D71	9F91	06181	5D71
6C73	5D72	9F92	06182	5D72
6C92	5D73	9F93	06183	5D73
6C90	5D74	9F94	06184	5D74
6CC4	5D75	9F95	06185	5D75
6CF1	5D76	9F96	06186	5D76
6CD3	5D77	9F97	06187	5D77
6CBD	5D78	9F98	06188	5D78
6CD7	5D79	9F99	06189	5D79
6CC5	5D7A	9F9A	06190	5D7A
6CDD	5D7B	9F9B	06191	5D7B
6CAE	5D7C	9F9C	06192	5D7C
6CB1	5D7D	9F9D	06193	5D7D
6CBE	5D7E	9F9E	06194	5D7E
6CBA	5E21	9F9F	06201	5E21
6CDB	5E22	9FA0	06202	5E22
6CEF	5E23	9FA1	06203	5E23
6CD9	5E24	9FA2	06204	5E24
6CEA	5E25	9FA3	06205	5E25
6D1F	5E26	9FA4	06206	5E26
884D	5E27	9FA5	06207	5E27
6D36	5E28	9FA6	06208	5E28
6D2B	5E29	9FA7	06209	5E29
6D3D	5E2A	9FA8	06210	5E2A
6D38	5E2B	9FA9	06211	5E2B
6D19	5E2C	9FAA	06212	5E2C
6D35	5E2D	9FAB	06213	5E2D
6D33	5E2E	9FAC	06214	5E2E
6D12	5E2F	9FAD	06215	5E2F
6D0C	5E30	9FAE	06216	5E30
6D63	5E31	9FAF	06217	5E31
6D93	5E32	9FB0	06218	5E32
6D64	5E33	9FB1	06219	5E33
6D5A	5E34	9FB2	06220	5E34
6D79	5E35	9FB3	06221	5E35
6D59	5E36	9FB4	06222	5E36
E386				5E37
6D95	5E38	9FB6	06224	5E38
6D9B	4573	9393	03783	5E39
6D85	5E3A	9FB8	06226	5E3A
6DF9	5E3B	9FB9	06227	5E3B
6E15	5E3C	9FBA	06228	5E3C
6E0A	5E3D	9FBB	06229	5E3D
6DB5	5E3E	9FBC	06230	5E3E
6DC7	5E3F	9FBD	06231	5E3F
6DE6	5E40	9FBE	06232	5E40
6DB8	5E41	9FBF	06233	5E41
6DC6	5E42	9FC0	06234	5E42
6DEC	5E43	9FC1	06235	5E43
6DDE	5E44	9FC2	06236	5E44
6DCC	5E45	9FC3	06237	5E45
6DE8	5E46	9FC4	06238	5E46
6DD2	5E47	9FC5	06239	5E47
6DC5	5E48	9FC6	06240	5E48
6DFA	5E49	9FC7	06241	5E49
6DD9	5E4A	9FC8	06242	5E4A
6DE4	5E4B	9FC9	06243	5E4B
6DD5	5E4C	9FCA	06244	5E4C
6DEA	5E4D	9FCB	06245	5E4D
6DEE	5E4E	9FCC	06246	5E4E
6E2D	5E4F	9FCD	06247	5E4F
E387				5E50
6E2E	5E51	9FCF	06249	5E51
6E19	5E52	9FD0	06250	5E52
6E72	5E53	9FD1	06251	5E53
6E5F	5E54	9FD2	06252	5E54
6E3E	5E55	9FD3	06253	5E55
E388				5E56
6E6B	5E57	9FD5	06255	5E57
6E2B	5E58	9FD6	06256	5E58
6E76	5E59	9FD7	06257	5E59
6E4D	5E5A	9FD8	06258	5E5A
6E1F	5E5B	9FD9	06259	5E5B
6E43	5E5C	9FDA	06260	5E5C
6E3A	5E5D	9FDB	06261	5E5D
6E4E	5E5E	9FDC	06262	5E5E
6E24	5E5F	9FDD	06263	5E5F
6EFF	5E60	9FDE	06264	5E60
E389				5E61
6E38	5E62	9FE0	06266	5E62
6E82	5E63	9FE1	06267	5E63
6EAA	5E64	9FE2	06268	5E64
6E98	5E65	9FE3	06269	5E65
6EC9	5E66	9FE4	06270	5E66
6EB7	5E67	9FE5	06271	5E67
6ED3	5E68	9FE6	06272	5E68
6EBD	5E69	9FE7	06273	5E69
6EAF	5E6A	9FE8	06274	5E6A
6EC4	5E6B	9FE9	06275	5E6B
6EB2	5E6C	9FEA	06276	5E6C
6ED4	5E6D	9FEB	06277	5E6D
6ED5	5E6E	9FEC	06278	5E6E
6E8F	5E6F	9FED	06279	5E6F
6EA5	5E70	9FEE	06280	5E70
6EC2	5E71	9FEF	06281	5E71
6E9F	5E72	9FF0	06282	5E72
6F41	5E73	9FF1	06283	5E73
6F11	5E74	9FF2	06284	5E74
E38A				5E75
6EEC	5E76	9FF4	06286	5E76
6EF8	5E77	9FF5	06287	5E77
E38B				5E78
6F3F	5E79	9FF7	06289	5E79
6EF2	5E7A	9FF8	06290	5E7A
6F31	5E7B	9FF9	06291	5E7B
6EEF	5E7C	9FFA	06292	5E7C
6F32	5E7D	9FFB	06293	5E7D
6ECC	5E7E	9FFC	06294	5E7E
E38C				5F21
6F13	5F22	E041	06302	5F22
6EF7	5F23	E042	06303	5F23
6F86	5F24	E043	06304	5F24
6F7A	5F25	E044	06305	5F25
6F78	5F26	E045	06306	5F26
6F81	5F27	E046	06307	5F27
6F80	5F28	E047	06308	5F28
6F6F	5F29	E048	06309	5F29
6F5B	5F2A	E049	06310	5F2A
6FF3	5F2B	E04A	06311	5F2B
6F6D	5F2C	E04B	06312	5F2C
6F82	5F2D	E04C	06313	5F2D
6F7C	5F2E	E04D	06314	5F2E
6F58	5F2F	E04E	06315	5F2F
6F8E	5F30	E04F	06316	5F30
6F91	5F31	E050	06317	5F31
6FC2	5F32	E051	06318	5F32
6F66	5F33	E052	06319	5F33
6FB3	5F34	E053	06320	5F34
6FA3	5F35	E054	06321	5F35
6FA1	5F36	E055	06322	5F36
6FA4	5F37	E056	06323	5F37
6FB9	5F38	E057	06324	5F38
6FC6	5F39	E058	06325	5F39
6FAA	5F3A	E059	06326	5F3A
6FDF	5F3B	E05A	06327	5F3B
6FD5	5F3C	E05B	06328	5F3C
6FEC	5F3D	E05C	06329	5F3D
6FD4	5F3E	E05D	06330	5F3E
6FD8	5F3F	E05E	06331	5F3F
6FF1	5F40	E05F	06332	5F40
6FEE	5F41	E060	06333	5F41
6FDB	5F42	E061	06334	5F42
7009	5F43	E062	06335	5F43
700B	5F44	E063	06336	5F44
6FFA	5F45	E064	06337	5F45
7011	5F46	E065	06338	5F46
7001	5F47	E066	06339	5F47
700F	5F48	E067	06340	5F48
6FFE	5F49	E068	06341	5F49
701B	5F4A	E069	06342	5F4A
701A	5F4B	E06A	06343	5F4B
6F74	5F4C	E06B	06344	5F4C
701D	5F4D	E06C	06345	5F4D
7018	5F4E	E06D	06346	5F4E
701F	5F4F	E06E	06347	5F4F
7030	5F50	E06F	06348	5F50
703E	5F51	E070	06349	5F51
7032	5F52	E071	06350	5F52
7051	5F53	E072	06351	5F53
7063	5F54	E073	06352	5F54
7099	5F55	E074	06353	5F55
7092	5F56	E075	06354	5F56
70AF	5F57	E076	06355	5F57
70F1	5F58	E077	06356	5F58
70AC	5F59	E078	06357	5F59
70B8	5F5A	E079	06358	5F5A
70B3	5F5B	E07A	06359	5F5B
70AE	5F5C	E07B	06360	5F5C
70DF	5F5D	E07C	06361	5F5D
70CB	5F5E	E07D	06362	5F5E
70DD	5F5F	E07E	06363	5F5F
70D9	5F60	E080	06364	5F60
7109	5F61	E081	06365	5F61
70FD	5F62	E082	06366	5F62
711C	5F63	E083	06367	5F63
7119	5F64	E084	06368	5F64
7165	5F65	E085	06369	5F65
7155	5F66	E086	06370	5F66
7188	5F67	E087	06371	5F67
7166	5F68	E088	06372	5F68
E38D				5F69
714C	5F6A	E08A	06374	5F6A
7156	5F6B	E08B	06375	5F6B
716C	5F6C	E08C	06376	5F6C
718F	5F6D	E08D	06377	5F6D
71FB	5F6E	E08E	06378	5F6E
7184	5F6F	E08F	06379	5F6F
7195	5F70	E090	06380	5F70
71A8	5F71	E091	06381	5F71
71AC	5F72	E092	06382	5F72
E38E				5F73
71B9	5F74	E094	06384	5F74
71BE	5F75	E095	06385	5F75
71D2	5F76	E096	06386	5F76
71C9	5F77	E097	06387	5F77
71D4	5F78	E098	06388	5F78
71CE	5F79	E099	06389	5F79
71E0	5F7A	E09A	06390	5F7A
71EC	5F7B	E09B	06391	5F7B
71E7	5F7C	E09C	06392	5F7C
71F5	5F7D	E09D	06393	5F7D
71FC	5F7E	E09E	06394	5F7E
71F9	6021	E09F	06401	6021
E38F				6022
720D	6023	E0A1	06403	6023
7210	6024	E0A2	06404	6024
721B	6025	E0A3	06405	6025
E390				6026
722D	6027	E0A5	06407	6027
722C	6028	E0A6	06408	6028
7230	6029	E0A7	06409	6029
7232	602A	E0A8	06410	602A
E391				602B
723C	602C	E0AA	06412	602C
723F	602D	E0AB	06413	602D
7240	602E	E0AC	06414	602E
7246	602F	E0AD	06415	602F
724B	6030	E0AE	06416	6030
7258	6031	E0AF	06417	6031
7274	6032	E0B0	06418	6032
727E	6033	E0B1	06419	6033
7282	6034	E0B2	06420	6034
7281	6035	E0B3	06421	6035
7287	6036	E0B4	06422	6036
7292	6037	E0B5	06423	6037
7296	6038	E0B6	06424	6038
72A2	6039	E0B7	06425	6039
72A7	603A	E0B8	06426	603A
72B9	603B	E0B9	06427	603B
72B2	603C	E0BA	06428	603C
72C3	603D	E0BB	06429	603D
72C6	603E	E0BC	06430	603E
72C4	603F	E0BD	06431	603F
72CE	6040	E0BE	06432	6040
72D2	6041	E0BF	06433	6041
72E2	6042	E0C0	06434	6042
72E0	6043	E0C1	06435	6043
E392				6044
72F9	6045	E0C3	06437	6045
72F7	6046	E0C4	06438	6046
500F	6047	E0C5	06439	6047
7317	6048	E0C6	06440	6048
730A	6049	E0C7	06441	6049
731C	604A	E0C8	06442	604A
7316	604B	E0C9	06443	604B
731D	604C	E0CA	06444	604C
7334	604D	E0CB	06445	604D
732F	604E	E0CC	06446	604E
7329	604F	E0CD	06447	604F
7325	6050	E0CE	06448	6050
733E	6051	E0CF	06449	6051
734E	6052	E0D0	06450	6052
734F	6053	E0D1	06451	6053
9ED8	6054	E0D2	06452	6054
7357	6055	E0D3	06453	6055
736A	6056	E0D4	06454	6056
7368	6057	E0D5	06455	6057
7370	6058	E0D6	06456	6058
7378	6059	E0D7	06457	6059
7375	605A	E0D8	06458	605A
737B	605B	E0D9	06459	605B
737A	605C	E0DA	06460	605C
73C8	605D	E0DB	06461	605D
73B3	605E	E0DC	06462	605E
73CE	605F	E0DD	06463	605F
73BB	6060	E0DE	06464	6060
73C0	6061	E0DF	06465	6061
E393				6062
73EE	6063	E0E1	06467	6063
73DE	6064	E0E2	06468	6064
74A2	6065	E0E3	06469	6065
7405	6066	E0E4	06470	6066
746F	6067	E0E5	06471	6067
7425	6068	E0E6	06472	6068
73F8	6069	E0E7	06473	6069
7432	606A	E0E8	06474	606A
743A	606B	E0E9	06475	606B
7455	606C	E0EA	06476	606C
743F	606D	E0EB	06477	606D
745F	606E	E0EC	06478	606E
7459	606F	E0ED	06479	606F
7441	6070	E0EE	06480	6070
E394				6071
7469	6072	E0F0	06482	6072
7470	6073	E0F1	06483	6073
7463	6074	E0F2	06484	6074
746A	6075	E0F3	06485	6075
7464	7424	EAA2	08404	6076
747E	6077	E0F5	06487	6077
748B	6078	E0F6	06488	6078
749E	6079	E0F7	06489	6079
74A7	607A	E0F8	06490	607A
74CA	607B	E0F9	06491	607B
74CF	607C	E0FA	06492	607C
74D4	607D	E0FB	06493	607D
73F1	607E	E0FC	06494	607E
74E0	6121	E140	06501	6121
74E3	6122	E141	06502	6122
74E7	6123	E142	06503	6123
74E9	6124	E143	06504	6124
74EE	6125	E144	06505	6125
74F2	6126	E145	06506	6126
74F0	6127	E146	06507	6127
74F1	6128	E147	06508	6128
74F8	6129	E148	06509	6129
74F7	612A	E149	06510	612A
E395				612B
7503	612C	E14B	06512	612C
7505	612D	E14C	06513	612D
750C	612E	E14D	06514	612E
750E	612F	E14E	06515	612F
E396				6130
E397				6131
7513	6132	E151	06518	6132
751E	6133	E152	06519	6133
E398				6134
752C	6135	E154	06521	6135
753C	6136	E155	06522	6136
7544	6137	E156	06523	6137
754D	6138	E157	06524	6138
754A	6139	E158	06525	6139
7549	613A	E159	06526	613A
755B	613B	E15A	06527	613B
7546	613C	E15B	06528	613C
755A	613D	E15C	06529	613D
7569	613E	E15D	06530	613E
7564	613F	E15E	06531	613F
7567	6140	E15F	06532	6140
756B	6141	E160	06533	6141
756D	6142	E161	06534	6142
7578	6143	E162	06535	6143
7576	6144	E163	06536	6144
7586	6145	E164	06537	6145
7587	6146	E165	06538	6146
7574	6147	E166	06539	6147
758A	6148	E167	06540	6148
7589	6149	E168	06541	6149
7582	614A	E169	06542	614A
7594	614B	E16A	06543	614B
759A	614C	E16B	06544	614C
759D	614D	E16C	06545	614D
75A5	614E	E16D	06546	614E
75A3	614F	E16E	06547	614F
75C2	6150	E16F	06548	6150
75B3	6151	E170	06549	6151
75C3	6152	E171	06550	6152
75B5	6153	E172	06551	6153
75BD	6154	E173	06552	6154
75B8	6155	E174	06553	6155
75BC	6156	E175	06554	6156
75B1	6157	E176	06555	6157
75CD	6158	E177	06556	6158
E399				6159
75D2	615A	E179	06558	615A
75D9	615B	E17A	06559	615B
75E3	615C	E17B	06560	615C
75DE	615D	E17C	06561	615D
75FE	615E	E17D	06562	615E
75FF	615F	E17E	06563	615F
75FC	6160	E180	06564	6160
7601	6161	E181	06565	6161
E39A				6162
75FA	6163	E183	06567	6163
75F2	6164	E184	06568	6164
75F3	6165	E185	06569	6165
760B	6166	E186	06570	6166
760D	6167	E187	06571	6167
E39B				6168
761F	6169	E189	06573	6169
7627	616A	E18A	06574	616A
7620	616B	E18B	06575	616B
7621	616C	E18C	06576	616C
7622	616D	E18D	06577	616D
7624	616E	E18E	06578	616E
7634	616F	E18F	06579	616F
7630	6170	E190	06580	6170
763B	6171	E191	06581	6171
7647	6172	E192	06582	6172
E39C				6173
7646	6174	E194	06584	6174
765C	6175	E195	06585	6175
7658	6176	E196	06586	6176
7661	6177	E197	06587	6177
7662	6178	E198	06588	6178
7668	6179	E199	06589	6179
7669	617A	E19A	06590	617A
766A	617B	E19B	06591	617B
7667	617C	E19C	06592	617C
766C	617D	E19D	06593	617D
7670	617E	E19E	06594	617E
E39D				6221
7676	6222	E1A0	06602	6222
7678	6223	E1A1	06603	6223
767C	6224	E1A2	06604	6224
7680	6225	E1A3	06605	6225
7683	6226	E1A4	06606	6226
7688	6227	E1A5	06607	6227
768B	6228	E1A6	06608	6228
E39E				6229
7696	622A	E1A8	06610	622A
E39F				622B
7699	622C	E1AA	06612	622C
769A	622D	E1AB	06613	622D
E3A0				622E
E3A1				622F
76B8	6230	E1AE	06616	6230
76B9	6231	E1AF	06617	6231
76BA	6232	E1B0	06618	6232
76C2	6233	E1B1	06619	6233
76CD	6234	E1B2	06620	6234
76D6	6235	E1B3	06621	6235
76D2	6236	E1B4	06622	6236
76DE	6237	E1B5	06623	6237
76E1	6238	E1B6	06624	6238
76E5	6239	E1B7	06625	6239
76E7	623A	E1B8	06626	623A
76EA	623B	E1B9	06627	623B
862F	623C	E1BA	06628	623C
76FB	623D	E1BB	06629	623D
7708	623E	E1BC	06630	623E
7707	623F	E1BD	06631	623F
7704	6240	E1BE	06632	6240
7729	6241	E1BF	06633	6241
7724	6242	E1C0	06634	6242
771E	6243	E1C1	06635	6243
7725	6244	E1C2	06636	6244
7726	6245	E1C3	06637	6245
771B	6246	E1C4	06638	6246
7737	6247	E1C5	06639	6247
7738	6248	E1C6	06640	6248
7747	6249	E1C7	06641	6249
775A	624A	E1C8	06642	624A
7768	624B	E1C9	06643	624B
776B	624C	E1CA	06644	624C
775B	624D	E1CB	06645	624D
7765	624E	E1CC	06646	624E
777F	624F	E1CD	06647	624F
777E	6250	E1CE	06648	6250
7779	6251	E1CF	06649	6251
778E	6252	E1D0	06650	6252
778B	6253	E1D1	06651	6253
7791	6254	E1D2	06652	6254
77A0	6255	E1D3	06653	6255
779E	6256	E1D4	06654	6256
77B0	6257	E1D5	06655	6257
77B6	6258	E1D6	06656	6258
77B9	6259	E1D7	06657	6259
77BF	625A	E1D8	06658	625A
77BC	625B	E1D9	06659	625B
77BD	625C	E1DA	06660	625C
77BB	625D	E1DB	06661	625D
77C7	625E	E1DC	06662	625E
77CD	625F	E1DD	06663	625F
E3A2				6260
77DA	6261	E1DF	06665	6261
E3A3				6262
77E3	6263	E1E1	06667	6263
77EE	6264	E1E2	06668	6264
77FC	6265	E1E3	06669	6265
780C	6266	E1E4	06670	6266
7812	6267	E1E5	06671	6267
783F	395C	8D7B	02560	6268
7820	6269	E1E7	06673	6269
783A	4557	9376	03755	626A
7845	626B	E1E9	06675	626B
788E	626C	E1EA	06676	626C
7874	626D	E1EB	06677	626D
7886	626E	E1EC	06678	626E
E3A4				626F
789A	6270	E1EE	06680	6270
788C	6271	E1EF	06681	6271
78A3	6272	E1F0	06682	6272
78B5	6273	E1F1	06683	6273
78AA	6274	E1F2	06684	6274
78AF	6275	E1F3	06685	6275
78D1	6276	E1F4	06686	6276
78C6	6277	E1F5	06687	6277
78CB	6278	E1F6	06688	6278
78D4	6279	E1F7	06689	6279
78BE	627A	E1F8	06690	627A
78BC	627B	E1F9	06691	627B
78C5	627C	E1FA	06692	627C
78CA	627D	E1FB	06693	627D
78EC	627E	E1FC	06694	627E
78E7	6321	E240	06701	6321
78DA	6322	E241	06702	6322
78FD	6323	E242	06703	6323
78F4	6324	E243	06704	6324
7907	6325	E244	06705	6325
7912	6326	E245	06706	6326
7911	6327	E246	06707	6327
7919	6328	E247	06708	6328
792C	6329	E248	06709	6329
792B	632A	E249	06710	632A
7940	632B	E24A	06711	632B
7960	632C	E24B	06712	632C
7957	632D	E24C	06713	632D
795F	632E	E24D	06714	632E
795A	632F	E24E	06715	632F
7955	6330	E24F	06716	6330
7953	6331	E250	06717	6331
797A	6332	E251	06718	6332
797F	6333	E252	06719	6333
798A	6334	E253	06720	6334
799D	6335	E254	06721	6335
79A7	6336	E255	06722	6336
9F4B	6337	E256	06723	6337
79AA	6338	E257	06724	6338
79AE	6339	E258	06725	6339
79B3	633A	E259	06726	633A
79B9	633B	E25A	06727	633B
79BA	633C	E25B	06728	633C
79C9	633D	E25C	06729	633D
79D5	633E	E25D	06730	633E
79E7	633F	E25E	06731	633F
79EC	6340	E25F	06732	6340
79E1	6341	E260	06733	6341
79E3	6342	E261	06734	6342
7A08	6343	E262	06735	6343
7A0D	6344	E263	06736	6344
7A18	6345	E264	06737	6345
E3A5				6346
7A20	6347	E266	06739	6347
7A1F	6348	E267	06740	6348
7980	6349	E268	06741	6349
E3A6				634A
7A3B	634B	E26A	06743	634B
7A3E	634C	E26B	06744	634C
7A37	634D	E26C	06745	634D
7A43	634E	E26D	06746	634E
7A57	634F	E26E	06747	634F
7A49	6350	E26F	06748	6350
7A61	6351	E270	06749	6351
7A62	6352	E271	06750	6352
7A69	6353	E272	06751	6353
E3A7				6354
7A70	6355	E274	06753	6355
7A79	6356	E275	06754	6356
7A7D	6357	E276	06755	6357
7A88	6358	E277	06756	6358
E3A8				6359
7A95	635A	E279	06758	635A
7A98	635B	E27A	06759	635B
7A96	635C	E27B	06760	635C
7AA9	635D	E27C	06761	635D
7AC3	3376	8A96	01986	635E
7AB0	635F	E27E	06763	635F
7AB6	6360	E280	06764	6360
7AC5	6361	E281	06765	6361
7AC4	6362	E282	06766	6362
7ABF	6363	E283	06767	6363
9083	6364	E284	06768	6364
7AC7	6365	E285	06769	6365
7ACA	6366	E286	06770	6366
7ACD	6367	E287	06771	6367
7ACF	6368	E288	06772	6368
7AD5	6369	E289	06773	6369
7AD3	636A	E28A	06774	636A
7AD9	636B	E28B	06775	636B
7ADA	636C	E28C	06776	636C
7ADD	636D	E28D	06777	636D
7AE1	636E	E28E	06778	636E
7AE2	636F	E28F	06779	636F
7AE6	6370	E290	06780	6370
7AED	6371	E291	06781	6371
7AF0	6372	E292	06782	6372
7B02	6373	E293	06783	6373
7B0F	6374	E294	06784	6374
7B0A	6375	E295	06785	6375
7B06	6376	E296	06786	6376
7B33	6377	E297	06787	6377
7B18	6378	E298	06788	6378
7B19	6379	E299	06789	6379
7B1E	637A	E29A	06790	637A
7B35	637B	E29B	06791	637B
7B28	637C	E29C	06792	637C
7B36	637D	E29D	06793	637D
7B50	637E	E29E	06794	637E
7B7A	6421	E29F	06801	6421
7B04	6422	E2A0	06802	6422
7B4D	6423	E2A1	06803	6423
7B0B	6424	E2A2	06804	6424
E3A9				6425
7B45	6426	E2A4	06806	6426
E3AA				6427
7B65	6428	E2A6	06808	6428
7B74	6429	E2A7	06809	6429
7B67	642A	E2A8	06810	642A
7B70	642B	E2A9	06811	642B
7B71	642C	E2AA	06812	642C
7B6C	642D	E2AB	06813	642D
7B6E	642E	E2AC	06814	642E
7B9D	642F	E2AD	06815	642F
7B98	6430	E2AE	06816	6430
7B9F	6431	E2AF	06817	6431
7B8D	6432	E2B0	06818	6432
7B9C	6433	E2B1	06819	6433
7B9A	6434	E2B2	06820	6434
7B8B	6435	E2B3	06821	6435
7B92	6436	E2B4	06822	6436
7B8F	6437	E2B5	06823	6437
7B5D	6438	E2B6	06824	6438
E3AB				6439
7BCB	643A	E2B8	06826	643A
7BC1	643B	E2B9	06827	643B
7BCC	643C	E2BA	06828	643C
7BCF	643D	E2BB	06829	643D
7BB4	643E	E2BC	06830	643E
7BC6	643F	E2BD	06831	643F
7BDD	6440	E2BE	06832	6440
7BE9	6441	E2BF	06833	6441
7C11	6442	E2C0	06834	6442
7C14	6443	E2C1	06835	6443
7BE6	6444	E2C2	06836	6444
7BE5	6445	E2C3	06837	6445
7BED	4F36	9855	04722	6446
7C00	6447	E2C5	06839	6447
7C07	6448	E2C6	06840	6448
7C13	6449	E2C7	06841	6449
7BF3	644A	E2C8	06842	644A
7BF7	644B	E2C9	06843	644B
7C17	644C	E2CA	06844	644C
7C0D	644D	E2CB	06845	644D
7BF6	644E	E2CC	06846	644E
7C23	644F	E2CD	06847	644F
7C27	6450	E2CE	06848	6450
7C2A	6451	E2CF	06849	6451
7C1F	6452	E2D0	06850	6452
7C37	6453	E2D1	06851	6453
7C2B	6454	E2D2	06852	6454
7C3D	6455	E2D3	06853	6455
7C4C	6456	E2D4	06854	6456
7C43	6457	E2D5	06855	6457
7C54	6458	E2D6	06856	6458
7C4F	6459	E2D7	06857	6459
7C40	645A	E2D8	06858	645A
7C50	645B	E2D9	06859	645B
7C58	645C	E2DA	06860	645C
7C5F	645D	E2DB	06861	645D
7C64	645E	E2DC	06862	645E
7C56	645F	E2DD	06863	645F
7C65	6460	E2DE	06864	6460
7C6C	6461	E2DF	06865	6461
7C75	6462	E2E0	06866	6462
7C83	6463	E2E1	06867	6463
E3AC				6464
7CA4	6465	E2E3	06869	6465
7CAD	6466	E2E4	06870	6466
7CA2	6467	E2E5	06871	6467
7CAB	6468	E2E6	06872	6468
7CA1	6469	E2E7	06873	6469
7CA8	646A	E2E8	06874	646A
E3AD				646B
7CB2	646C	E2EA	06876	646C
7CB1	646D	E2EB	06877	646D
E3AE				646E
7CB9	646F	E2ED	06879	646F
7CBD	6470	E2EE	06880	6470
7CC0	6471	E2EF	06881	6471
7CC5	6472	E2F0	06882	6472
7CC2	6473	E2F1	06883	6473
7CD8	6474	E2F2	06884	6474
7CD2	6475	E2F3	06885	6475
7CDC	6476	E2F4	06886	6476
7CE2	6477	E2F5	06887	6477
9B3B	6478	E2F6	06888	6478
7CEF	6479	E2F7	06889	6479
7CF2	647A	E2F8	06890	647A
7CF4	647B	E2F9	06891	647B
7CF6	647C	E2FA	06892	647C
7CFA	647D	E2FB	06893	647D
7D06	647E	E2FC	06894	647E
7D02	6521	E340	06901	6521
7D1C	6522	E341	06902	6522
7D15	6523	E342	06903	6523
E3AF				6524
7D45	6525	E344	06905	6525
7D4B	6526	E345	06906	6526
7D2E	6527	E346	06907	6527
7D32	6528	E347	06908	6528
7D3F	6529	E348	06909	6529
7D35	652A	E349	06910	652A
7D46	652B	E34A	06911	652B
7D73	652C	E34B	06912	652C
7D56	652D	E34C	06913	652D
7D4E	652E	E34D	06914	652E
7D72	652F	E34E	06915	652F
7D68	6530	E34F	06916	6530
7D6E	6531	E350	06917	6531
7D4F	6532	E351	06918	6532
7D63	6533	E352	06919	6533
7D93	6534	E353	06920	6534
7D89	6535	E354	06921	6535
7D5B	6536	E355	06922	6536
E3B0				6537
7D7D	6538	E357	06924	6538
E3B1				6539
7DBA	653A	E359	06926	653A
E3B2				653B
7DA3	653C	E35B	06928	653C
E3B3				653D
7DC7	653E	E35D	06930	653E
7DBD	653F	E35E	06931	653F
7DAB	6540	E35F	06932	6540
E3B4				6541
7DA2	6542	E361	06934	6542
7DAF	6543	E362	06935	6543
7DDC	6544	E363	06936	6544
7DB8	6545	E364	06937	6545
E3B5				6546
7DB0	6547	E366	06939	6547
7DD8	6548	E367	06940	6548
E3B6				6549
7DE4	654A	E369	06942	654A
7DDE	654B	E36A	06943	654B
7DFB	654C	E36B	06944	654C
7DF2	654D	E36C	06945	654D
7DE1	654E	E36D	06946	654E
7E05	654F	E36E	06947	654F
7E0A	6550	E36F	06948	6550
7E23	6551	E370	06949	6551
7E21	6552	E371	06950	6552
7E12	6553	E372	06951	6553
7E31	6554	E373	06952	6554
7E1F	6555	E374	06953	6555
7E09	6556	E375	06954	6556
7E0B	6557	E376	06955	6557
7E22	6558	E377	06956	6558
7E46	6559	E378	06957	6559
7E66	655A	E379	06958	655A
7E3B	655B	E37A	06959	655B
7E35	655C	E37B	06960	655C
7E39	655D	E37C	06961	655D
7E43	655E	E37D	06962	655E
7E37	655F	E37E	06963	655F
7E32	6560	E380	06964	6560
7E3A	6561	E381	06965	6561
7E67	6562	E382	06966	6562
7E5D	6563	E383	06967	6563
7E56	6564	E384	06968	6564
7E5E	6565	E385	06969	6565
7E59	6566	E386	06970	6566
7E5A	6567	E387	06971	6567
7E79	6568	E388	06972	6568
7E6A	6569	E389	06973	6569
7E69	656A	E38A	06974	656A
7E7C	656B	E38B	06975	656B
7E7B	656C	E38C	06976	656C
7E83	656D	E38D	06977	656D
7DD5	656E	E38E	06978	656E
7E7D	656F	E38F	06979	656F
8FAE	6570	E390	06980	6570
7E7F	6571	E391	06981	6571
7E88	6572	E392	06982	6572
7E89	6573	E393	06983	6573
7E8C	6574	E394	06984	6574
7E92	6575	E395	06985	6575
7E90	6576	E396	06986	6576
7E93	6577	E397	06987	6577
7E94	6578	E398	06988	6578
7E96	6579	E399	06989	6579
7E8E	657A	E39A	06990	657A
7E9B	657B	E39B	06991	657B
7E9C	657C	E39C	06992	657C
7F38	657D	E39D	06993	657D
7F3A	657E	E39E	06994	657E
7F45	6621	E39F	07001	6621
7F4C	6622	E3A0	07002	6622
7F4D	6623	E3A1	07003	6623
7F4E	6624	E3A2	07004	6624
7F50	6625	E3A3	07005	6625
7F51	6626	E3A4	07006	6626
7F55	6627	E3A5	07007	6627
7F54	6628	E3A6	07008	6628
7F58	6629	E3A7	07009	6629
7F5F	662A	E3A8	07010	662A
7F60	662B	E3A9	07011	662B
7F68	662C	E3AA	07012	662C
7F69	662D	E3AB	07013	662D
7F67	662E	E3AC	07014	662E
7F78	662F	E3AD	07015	662F
7F82	6630	E3AE	07016	6630
7F86	6631	E3AF	07017	6631
7F83	6632	E3B0	07018	6632
7F88	6633	E3B1	07019	6633
7F87	6634	E3B2	07020	6634
7F8C	6635	E3B3	07021	6635
7F94	6636	E3B4	07022	6636
7F9E	6637	E3B5	07023	6637
7F9D	6638	E3B6	07024	6638
7F9A	6639	E3B7	07025	6639
7FA3	663A	E3B8	07026	663A
7FAF	663B	E3B9	07027	663B
7FB2	663C	E3BA	07028	663C
7FB9	663D	E3BB	07029	663D
7FAE	663E	E3BC	07030	663E
7FB6	663F	E3BD	07031	663F
7FB8	6640	E3BE	07032	6640
8B71	6641	E3BF	07033	6641
7FC5	6642	E3C0	07034	6642
7FC6	6643	E3C1	07035	6643
7FCA	6644	E3C2	07036	6644
7FD5	6645	E3C3	07037	6645
E3B7				6646
7FE1	6647	E3C5	07039	6647
7FE6	6648	E3C6	07040	6648
7FE9	6649	E3C7	07041	6649
7FF3	664A	E3C8	07042	664A
7FF9	664B	E3C9	07043	664B
98DC	664C	E3CA	07044	664C
8006	664D	E3CB	07045	664D
8004	664E	E3CC	07046	664E
800B	664F	E3CD	07047	664F
8012	6650	E3CE	07048	6650
8018	6651	E3CF	07049	6651
8019	6652	E3D0	07050	6652
801C	6653	E3D1	07051	6653
8021	6654	E3D2	07052	6654
8028	6655	E3D3	07053	6655
803F	6656	E3D4	07054	6656
803B	6657	E3D5	07055	6657
804A	6658	E3D6	07056	6658
8046	6659	E3D7	07057	6659
8052	665A	E3D8	07058	665A
8058	665B	E3D9	07059	665B
E3B8				665C
805F	665D	E3DB	07061	665D
8062	665E	E3DC	07062	665E
8068	665F	E3DD	07063	665F
8073	6660	E3DE	07064	6660
8072	6661	E3DF	07065	6661
E3B9				6662
E3BA				6663
8079	6664	E3E2	07068	6664
807D	6665	E3E3	07069	6665
807F	6666	E3E4	07070	6666
8084	6667	E3E5	07071	6667
8086	6668	E3E6	07072	6668
8085	6669	E3E7	07073	6669
809B	666A	E3E8	07074	666A
8093	666B	E3E9	07075	666B
809A	666C	E3EA	07076	666C
80AD	666D	E3EB	07077	666D
5190	666E	E3EC	07078	666E
80AC	666F	E3ED	07079	666F
80DB	6670	E3EE	07080	6670
80E5	6671	E3EF	07081	6671
80D9	6672	E3F0	07082	6672
80DD	6673	E3F1	07083	6673
80C4	6674	E3F2	07084	6674
80DA	6675	E3F3	07085	6675
80D6	6676	E3F4	07086	6676
8109	6677	E3F5	07087	6677
80EF	6678	E3F6	07088	6678
80F1	6679	E3F7	07089	6679
811B	667A	E3F8	07090	667A
8129	667B	E3F9	07091	667B
8123	667C	E3FA	07092	667C
812F	667D	E3FB	07093	667D
814B	667E	E3FC	07094	667E
968B	6721	E440	07101	6721
8146	6722	E441	07102	6722
813E	6723	E442	07103	6723
E3BB				6724
8151	6725	E444	07105	6725
80FC	6726	E445	07106	6726
E3BC				6727
816E	6728	E447	07108	6728
8165	6729	E448	07109	6729
8166	672A	E449	07110	672A
8174	672B	E44A	07111	672B
8183	672C	E44B	07112	672C
8188	672D	E44C	07113	672D
818A	672E	E44D	07114	672E
8180	672F	E44E	07115	672F
8182	6730	E44F	07116	6730
81A0	6731	E450	07117	6731
8195	6732	E451	07118	6732
81A4	6733	E452	07119	6733
81A3	6734	E453	07120	6734
815F	6735	E454	07121	6735
8193	6736	E455	07122	6736
81A9	6737	E456	07123	6737
81B0	6738	E457	07124	6738
81B5	6739	E458	07125	6739
81BE	673A	E459	07126	673A
81B8	673B	E45A	07127	673B
81BD	673C	E45B	07128	673C
81C0	673D	E45C	07129	673D
81C2	673E	E45D	07130	673E
81BA	673F	E45E	07131	673F
81C9	6740	E45F	07132	6740
81CD	6741	E460	07133	6741
81D1	6742	E461	07134	6742
81D9	6743	E462	07135	6743
81D8	6744	E463	07136	6744
81C8	6745	E464	07137	6745
81DA	6746	E465	07138	6746
81DF	6747	E466	07139	6747
81E0	6748	E467	07140	6748
81E7	6749	E468	07141	6749
81FA	674A	E469	07142	674A
81FB	674B	E46A	07143	674B
81FE	674C	E46B	07144	674C
8201	674D	E46C	07145	674D
8202	674E	E46D	07146	674E
8205	674F	E46E	07147	674F
8207	6750	E46F	07148	6750
820A	6751	E470	07149	6751
820D	6752	E471	07150	6752
8210	6753	E472	07151	6753
8216	6754	E473	07152	6754
8229	6755	E474	07153	6755
822B	6756	E475	07154	6756
8238	6757	E476	07155	6757
8233	6758	E477	07156	6758
8240	6759	E478	07157	6759
8259	675A	E479	07158	675A
8258	675B	E47A	07159	675B
825D	675C	E47B	07160	675C
825A	675D	E47C	07161	675D
825F	675E	E47D	07162	675E
8264	675F	E47E	07163	675F
8262	6760	E480	07164	6760
E3BD				6761
826A	6762	E482	07166	6762
826B	6763	E483	07167	6763
E3BE				6764
8271	6765	E485	07169	6765
8277	6766	E486	07170	6766
8278	6767	E487	07171	6767
E3BF				6768
E3C0				6769
8292	676A	E48A	07174	676A
82AB	676B	E48B	07175	676B
829F	676C	E48C	07176	676C
82BB	676D	E48D	07177	676D
82AC	676E	E48E	07178	676E
82E1	676F	E48F	07179	676F
82E3	6770	E490	07180	6770
82DF	6771	E491	07181	6771
E3C1				6772
82F4	6773	E493	07183	6773
82F3	6774	E494	07184	6774
82FA	6775	E495	07185	6775
8393	6776	E496	07186	6776
8303	6777	E497	07187	6777
82FB	6778	E498	07188	6778
82F9	6779	E499	07189	6779
82DE	677A	E49A	07190	677A
8306	677B	E49B	07191	677B
82DC	677C	E49C	07192	677C
8309	677D	E49D	07193	677D
82D9	677E	E49E	07194	677E
8335	6821	E49F	07201	6821
8334	6822	E4A0	07202	6822
8316	6823	E4A1	07203	6823
8332	6824	E4A2	07204	6824
8331	6825	E4A3	07205	6825
8340	6826	E4A4	07206	6826
8339	6827	E4A5	07207	6827
8350	6828	E4A6	07208	6828
8345	6829	E4A7	07209	6829
832F	682A	E4A8	07210	682A
832B	682B	E4A9	07211	682B
8317	682C	E4AA	07212	682C
8318	682D	E4AB	07213	682D
8385	682E	E4AC	07214	682E
E3C2				682F
83AA	6830	E4AE	07216	6830
839F	6831	E4AF	07217	6831
83A2	6832	E4B0	07218	6832
8396	6833	E4B1	07219	6833
8323	6834	E4B2	07220	6834
838E	6835	E4B3	07221	6835
8387	6836	E4B4	07222	6836
838A	6837	E4B5	07223	6837
837C	6838	E4B6	07224	6838
83B5	6839	E4B7	07225	6839
8373	683A	E4B8	07226	683A
E3C3				683B
83A0	683C	E4BA	07228	683C
8389	683D	E4BB	07229	683D
83A8	683E	E4BC	07230	683E
83F4	683F	E4BD	07231	683F
8413	6840	E4BE	07232	6840
83EB	6841	E4BF	07233	6841
83CE	6842	E4C0	07234	6842
83FD	6843	E4C1	07235	6843
8403	6844	E4C2	07236	6844
83D8	6845	E4C3	07237	6845
840B	6846	E4C4	07238	6846
83C1	6847	E4C5	07239	6847
83F7	6848	E4C6	07240	6848
8407	6849	E4C7	07241	6849
83E0	684A	E4C8	07242	684A
83F2	684B	E4C9	07243	684B
840D	684C	E4CA	07244	684C
8422	684D	E4CB	07245	684D
8420	684E	E4CC	07246	684E
83BD	684F	E4CD	07247	684F
8438	6850	E4CE	07248	6850
8506	6851	E4CF	07249	6851
83FB	6852	E4D0	07250	6852
846D	6853	E4D1	07251	6853
842A	6854	E4D2	07252	6854
843C	6855	E4D3	07253	6855
855A	6856	E4D4	07254	6856
8484	6857	E4D5	07255	6857
8477	6858	E4D6	07256	6858
846B	6859	E4D7	07257	6859
84AD	685A	E4D8	07258	685A
846E	685B	E4D9	07259	685B
8482	685C	E4DA	07260	685C
8469	685D	E4DB	07261	685D
8446	685E	E4DC	07262	685E
842C	685F	E4DD	07263	685F
846F	6860	E4DE	07264	6860
8479	6861	E4DF	07265	6861
8435	6862	E4E0	07266	6862
84CA	6863	E4E1	07267	6863
8462	6864	E4E2	07268	6864
84B9	6865	E4E3	07269	6865
84BF	6866	E4E4	07270	6866
849F	6867	E4E5	07271	6867
84D9	6868	E4E6	07272	6868
84CD	6869	E4E7	07273	6869
84BB	686A	E4E8	07274	686A
84DA	686B	E4E9	07275	686B
84D0	686C	E4EA	07276	686C
E3C4				686D
84C6	686E	E4EC	07278	686E
84D6	686F	E4ED	07279	686F
84A1	6870	E4EE	07280	6870
8521	6871	E4EF	07281	6871
84FF	6872	E4F0	07282	6872
84F4	6873	E4F1	07283	6873
8517	6874	E4F2	07284	6874
8518	6875	E4F3	07285	6875
852C	6876	E4F4	07286	6876
851F	6877	E4F5	07287	6877
8515	6878	E4F6	07288	6878
8514	6879	E4F7	07289	6879
84FC	687A	E4F8	07290	687A
8540	687B	E4F9	07291	687B
8563	687C	E4FA	07292	687C
8558	687D	E4FB	07293	687D
8548	687E	E4FC	07294	687E
8541	6921	E540	07301	6921
854A	3C49	8EC7	02841	6922
854B	6923	E542	07303	6923
8555	6924	E543	07304	6924
8580	6925	E544	07305	6925
85A4	6926	E545	07306	6926
8588	6927	E546	07307	6927
8591	6928	E547	07308	6928
858A	6929	E548	07309	6929
85A8	692A	E549	07310	692A
856D	692B	E54A	07311	692B
8594	692C	E54B	07312	692C
859B	692D	E54C	07313	692D
85AE	4C79	96F7	04489	692E
8587	692F	E54E	07315	692F
859C	6930	E54F	07316	6930
8577	6931	E550	07317	6931
857E	6932	E551	07318	6932
8590	6933	E552	07319	6933
85C9	6934	E553	07320	6934
85BA	6935	E554	07321	6935
85CF	6936	E555	07322	6936
85B9	6937	E556	07323	6937
85D0	6938	E557	07324	6938
85D5	6939	E558	07325	6939
85DD	693A	E559	07326	693A
85E5	693B	E55A	07327	693B
E3C5				693C
85F9	693D	E55C	07329	693D
860A	693E	E55D	07330	693E
8613	693F	E55E	07331	693F
860B	6940	E55F	07332	6940
85FE	6941	E560	07333	6941
85FA	6942	E561	07334	6942
8606	6943	E562	07335	6943
8622	6944	E563	07336	6944
861A	6945	E564	07337	6945
8630	6946	E565	07338	6946
863F	6947	E566	07339	6947
864D	6948	E567	07340	6948
4E55	6949	E568	07341	6949
E3C6				694A
865F	694B	E56A	07343	694B
8667	694C	E56B	07344	694C
8671	694D	E56C	07345	694D
8693	694E	E56D	07346	694E
86A3	694F	E56E	07347	694F
86A9	6950	E56F	07348	6950
86AA	6951	E570	07349	6951
868B	6952	E571	07350	6952
868C	6953	E572	07351	6953
86B6	6954	E573	07352	6954
86AF	6955	E574	07353	6955
86C4	6956	E575	07354	6956
86C6	6957	E576	07355	6957
86B0	6958	E577	07356	6958
86C9	6959	E578	07357	6959
86CE	3342	8A61	01934	695A
86AB	695B	E57A	07359	695B
86D4	695C	E57B	07360	695C
86DE	695D	E57C	07361	695D
86E9	695E	E57D	07362	695E
86EC	695F	E57E	07363	695F
E3C7				6960
86DB	6961	E581	07365	6961
86EF	6962	E582	07366	6962
E3C8				6963
8706	6964	E584	07368	6964
8708	6965	E585	07369	6965
8700	6966	E586	07370	6966
8703	6967	E587	07371	6967
86FB	6968	E588	07372	6968
E3C9				6969
8709	696A	E58A	07374	696A
870D	696B	E58B	07375	696B
86F9	696C	E58C	07376	696C
870A	696D	E58D	07377	696D
8734	696E	E58E	07378	696E
873F	696F	E58F	07379	696F
8737	6970	E590	07380	6970
873B	6971	E591	07381	6971
8725	6972	E592	07382	6972
8729	6973	E593	07383	6973
871A	6974	E594	07384	6974
8760	6975	E595	07385	6975
875F	6976	E596	07386	6976
8778	6977	E597	07387	6977
874C	6978	E598	07388	6978
874E	6979	E599	07389	6979
8774	697A	E59A	07390	697A
8757	697B	E59B	07391	697B
8768	697C	E59C	07392	697C
876E	697D	E59D	07393	697D
8759	697E	E59E	07394	697E
E3CA				6A21
8763	6A22	E5A0	07402	6A22
876A	6A23	E5A1	07403	6A23
877F	4768	9488	03972	6A24
87A2	6A25	E5A3	07405	6A25
879F	6A26	E5A4	07406	6A26
E3CB				6A27
87AF	6A28	E5A6	07408	6A28
87CB	6A29	E5A7	07409	6A29
87BD	6A2A	E5A8	07410	6A2A
87C0	6A2B	E5A9	07411	6A2B
87D0	6A2C	E5AA	07412	6A2C
96D6	6A2D	E5AB	07413	6A2D
87AB	6A2E	E5AC	07414	6A2E
E3CC				6A2F
87B3	6A30	E5AE	07416	6A30
87C7	6A31	E5AF	07417	6A31
87C6	6A32	E5B0	07418	6A32
87BB	6A33	E5B1	07419	6A33
87EF	6A34	E5B2	07420	6A34
87F2	6A35	E5B3	07421	6A35
87E0	6A36	E5B4	07422	6A36
880F	6A37	E5B5	07423	6A37
880D	6A38	E5B6	07424	6A38
87FE	6A39	E5B7	07425	6A39
87F6	6A3A	E5B8	07426	6A3A
87F7	6A3B	E5B9	07427	6A3B
880E	6A3C	E5BA	07428	6A3C
87D2	6A3D	E5BB	07429	6A3D
8811	6A3E	E5BC	07430	6A3E
8816	6A3F	E5BD	07431	6A3F
8815	6A40	E5BE	07432	6A40
8822	6A41	E5BF	07433	6A41
8821	6A42	E5C0	07434	6A42
8831	6A43	E5C1	07435	6A43
8836	6A44	E5C2	07436	6A44
8839	6A45	E5C3	07437	6A45
8827	6A46	E5C4	07438	6A46
883B	6A47	E5C5	07439	6A47
8844	6A48	E5C6	07440	6A48
E3CD				6A49
8852	6A4A	E5C8	07442	6A4A
8859	6A4B	E5C9	07443	6A4B
885E	6A4C	E5CA	07444	6A4C
8862	6A4D	E5CB	07445	6A4D
886B	6A4E	E5CC	07446	6A4E
8881	6A4F	E5CD	07447	6A4F
887E	6A50	E5CE	07448	6A50
E3CE				6A51
8875	6A52	E5D0	07450	6A52
887D	6A53	E5D1	07451	6A53
88B5	6A54	E5D2	07452	6A54
8872	6A55	E5D3	07453	6A55
8882	6A56	E5D4	07454	6A56
8897	6A57	E5D5	07455	6A57
8892	6A58	E5D6	07456	6A58
88AE	6A59	E5D7	07457	6A59
8899	6A5A	E5D8	07458	6A5A
88A2	6A5B	E5D9	07459	6A5B
888D	6A5C	E5DA	07460	6A5C
88A4	6A5D	E5DB	07461	6A5D
88B0	6A5E	E5DC	07462	6A5E
88BF	6A5F	E5DD	07463	6A5F
88B1	6A60	E5DE	07464	6A60
88C3	6A61	E5DF	07465	6A61
88C4	6A62	E5E0	07466	6A62
88D4	6A63	E5E1	07467	6A63
88D8	6A64	E5E2	07468	6A64
88D9	6A65	E5E3	07469	6A65
88DD	6A66	E5E4	07470	6A66
88F9	6A67	E5E5	07471	6A67
8902	6A68	E5E6	07472	6A68
88FC	6A69	E5E7	07473	6A69
88F4	6A6A	E5E8	07474	6A6A
88E8	6A6B	E5E9	07475	6A6B
88F2	6A6C	E5EA	07476	6A6C
8904	6A6D	E5EB	07477	6A6D
890C	6A6E	E5EC	07478	6A6E
E3CF				6A6F
8913	6A70	E5EE	07480	6A70
8943	6A71	E5EF	07481	6A71
891E	6A72	E5F0	07482	6A72
8925	6A73	E5F1	07483	6A73
892A	6A74	E5F2	07484	6A74
E3D0				6A75
8941	6A76	E5F4	07486	6A76
8944	6A77	E5F5	07487	6A77
893B	6A78	E5F6	07488	6A78
8936	6A79	E5F7	07489	6A79
8938	6A7A	E5F8	07490	6A7A
894C	6A7B	E5F9	07491	6A7B
891D	6A7C	E5FA	07492	6A7C
8960	6A7D	E5FB	07493	6A7D
895E	6A7E	E5FC	07494	6A7E
8966	6B21	E640	07501	6B21
8964	6B22	E641	07502	6B22
896D	6B23	E642	07503	6B23
E3D1				6B24
E3D2				6B25
8974	6B26	E645	07506	6B26
8977	6B27	E646	07507	6B27
897E	6B28	E647	07508	6B28
8983	6B29	E648	07509	6B29
8988	6B2A	E649	07510	6B2A
898A	6B2B	E64A	07511	6B2B
8993	6B2C	E64B	07512	6B2C
8998	6B2D	E64C	07513	6B2D
89A1	6B2E	E64D	07514	6B2E
89A9	6B2F	E64E	07515	6B2F
E3D3				6B30
89AC	6B31	E650	07517	6B31
E3D4				6B32
89B2	6B33	E652	07519	6B33
89BA	6B34	E653	07520	6B34
89BD	6B35	E654	07521	6B35
89BF	6B36	E655	07522	6B36
89C0	6B37	E656	07523	6B37
89DA	6B38	E657	07524	6B38
89DC	6B39	E658	07525	6B39
89DD	6B3A	E659	07526	6B3A
89E7	6B3B	E65A	07527	6B3B
89F4	6B3C	E65B	07528	6B3C
89F8	6B3D	E65C	07529	6B3D
8A03	6B3E	E65D	07530	6B3E
8A16	6B3F	E65E	07531	6B3F
8A10	6B40	E65F	07532	6B40
8A0C	6B41	E660	07533	6B41
E3D5				6B42
E3D6				6B43
8A25	6B44	E663	07536	6B44
8A36	6B45	E664	07537	6B45
8A41	6B46	E665	07538	6B46
8A5B	6B47	E666	07539	6B47
8A52	6B48	E667	07540	6B48
8A46	6B49	E668	07541	6B49
8A48	6B4A	E669	07542	6B4A
8A7C	6B4B	E66A	07543	6B4B
8A6D	6B4C	E66B	07544	6B4C
8A6C	6B4D	E66C	07545	6B4D
8A62	6B4E	E66D	07546	6B4E
8A85	6B4F	E66E	07547	6B4F
8A82	6B50	E66F	07548	6B50
8A84	6B51	E670	07549	6B51
8AA8	6B52	E671	07550	6B52
8AA1	6B53	E672	07551	6B53
8A91	6B54	E673	07552	6B54
8AA5	6B55	E674	07553	6B55
8AA6	6B56	E675	07554	6B56
8A9A	6B57	E676	07555	6B57
8AA3	6B58	E677	07556	6B58
8AC4	6B59	E678	07557	6B59
8ACD	6B5A	E679	07558	6B5A
8AC2	6B5B	E67A	07559	6B5B
8ADA	6B5C	E67B	07560	6B5C
8ACC	3452	8AD0	02050	6B5D
8AF3	6B5E	E67D	07562	6B5E
8AE7	6B5F	E67E	07563	6B5F
8AE4	6B60	E680	07564	6B60
8AF1	6B61	E681	07565	6B61
8B14	6B62	E682	07566	6B62
8AE0	6B63	E683	07567	6B63
8AE2	6B64	E684	07568	6B64
8AF7	6B65	E685	07569	6B65
E3D7				6B66
8ADB	6B67	E687	07571	6B67
8B0C	6B68	E688	07572	6B68
8B07	6B69	E689	07573	6B69
8B1A	6B6A	E68A	07574	6B6A
8AE1	6B6B	E68B	07575	6B6B
8B16	6B6C	E68C	07576	6B6C
8B10	6B6D	E68D	07577	6B6D
8B17	6B6E	E68E	07578	6B6E
8B20	6B6F	E68F	07579	6B6F
8B33	6B70	E690	07580	6B70
97AB	6B71	E691	07581	6B71
8B26	6B72	E692	07582	6B72
8B2B	6B73	E693	07583	6B73
8B3E	6B74	E694	07584	6B74
8B28	6B75	E695	07585	6B75
8B41	6B76	E696	07586	6B76
8B4C	6B77	E697	07587	6B77
8B4F	6B78	E698	07588	6B78
8B4E	6B79	E699	07589	6B79
8B49	6B7A	E69A	07590	6B7A
8B56	6B7B	E69B	07591	6B7B
8B5B	6B7C	E69C	07592	6B7C
8B5A	6B7D	E69D	07593	6B7D
8B6B	6B7E	E69E	07594	6B7E
8B5F	6C21	E69F	07601	6C21
8B6C	6C22	E6A0	07602	6C22
8B6F	6C23	E6A1	07603	6C23
8B74	6C24	E6A2	07604	6C24
8B7D	6C25	E6A3	07605	6C25
8B80	6C26	E6A4	07606	6C26
8B8C	6C27	E6A5	07607	6C27
8B8E	6C28	E6A6	07608	6C28
8B92	6C29	E6A7	07609	6C29
8B93	6C2A	E6A8	07610	6C2A
8B96	6C2B	E6A9	07611	6C2B
8B99	6C2C	E6AA	07612	6C2C
8B9A	6C2D	E6AB	07613	6C2D
8C3A	6C2E	E6AC	07614	6C2E
E3D8				6C2F
8C3F	6C30	E6AE	07616	6C30
8C48	6C31	E6AF	07617	6C31
8C4C	6C32	E6B0	07618	6C32
8C4E	6C33	E6B1	07619	6C33
8C50	6C34	E6B2	07620	6C34
8C55	6C35	E6B3	07621	6C35
8C62	6C36	E6B4	07622	6C36
8C6C	6C37	E6B5	07623	6C37
8C78	6C38	E6B6	07624	6C38
8C7A	6C39	E6B7	07625	6C39
8C82	6C3A	E6B8	07626	6C3A
8C89	6C3B	E6B9	07627	6C3B
8C85	6C3C	E6BA	07628	6C3C
8C8A	6C3D	E6BB	07629	6C3D
8C8D	6C3E	E6BC	07630	6C3E
8C8E	6C3F	E6BD	07631	6C3F
8C94	6C40	E6BE	07632	6C40
8C7C	6C41	E6BF	07633	6C41
8C98	6C42	E6C0	07634	6C42
621D	6C43	E6C1	07635	6C43
E3D9				6C44
8CAA	6C45	E6C3	07637	6C45
8CBD	6C46	E6C4	07638	6C46
8CB2	6C47	E6C5	07639	6C47
8CB3	6C48	E6C6	07640	6C48
8CAE	6C49	E6C7	07641	6C49
8CB6	6C4A	E6C8	07642	6C4A
8CC8	6C4B	E6C9	07643	6C4B
8CC1	6C4C	E6CA	07644	6C4C
8CCE	4128	9147	03308	6C4D
8CE3	6C4E	E6CC	07646	6C4E
8CDA	6C4F	E6CD	07647	6C4F
8CFD	6C50	E6CE	07648	6C50
8CFA	6C51	E6CF	07649	6C51
8CFB	6C52	E6D0	07650	6C52
8D04	6C53	E6D1	07651	6C53
8D05	6C54	E6D2	07652	6C54
8D0A	6C55	E6D3	07653	6C55
8D07	6C56	E6D4	07654	6C56
E3DA				6C57
8D0D	6C58	E6D6	07656	6C58
8D10	6C59	E6D7	07657	6C59
9F4E	6C5A	E6D8	07658	6C5A
8D13	6C5B	E6D9	07659	6C5B
8CCD	6C5C	E6DA	07660	6C5C
8D14	6C5D	E6DB	07661	6C5D
8D16	6C5E	E6DC	07662	6C5E
8D67	6C5F	E6DD	07663	6C5F
8D6D	6C60	E6DE	07664	6C60
8D71	6C61	E6DF	07665	6C61
8D73	6C62	E6E0	07666	6C62
8D81	6C63	E6E1	07667	6C63
8D99	6C64	E6E2	07668	6C64
8DC2	6C65	E6E3	07669	6C65
8DBE	6C66	E6E4	07670	6C66
8DBA	6C67	E6E5	07671	6C67
8DCF	6C68	E6E6	07672	6C68
E3DB				6C69
8DD6	6C6A	E6E8	07674	6C6A
8DCC	6C6B	E6E9	07675	6C6B
8DDB	6C6C	E6EA	07676	6C6C
8DCB	6C6D	E6EB	07677	6C6D
8DEA	6C6E	E6EC	07678	6C6E
8DEB	6C6F	E6ED	07679	6C6F
8DDF	6C70	E6EE	07680	6C70
8DE3	6C71	E6EF	07681	6C71
8DFC	6C72	E6F0	07682	6C72
8E08	6C73	E6F1	07683	6C73
E3DC				6C74
8DFF	6C75	E6F3	07685	6C75
8E1D	6C76	E6F4	07686	6C76
8E1E	6C77	E6F5	07687	6C77
8E10	6C78	E6F6	07688	6C78
8E1F	6C79	E6F7	07689	6C79
8E42	6C7A	E6F8	07690	6C7A
8E35	6C7B	E6F9	07691	6C7B
E3DD				6C7C
8E34	6C7D	E6FB	07693	6C7D
8E4A	6C7E	E6FC	07694	6C7E
8E47	6D21	E740	07701	6D21
8E49	6D22	E741	07702	6D22
8E4C	6D23	E742	07703	6D23
8E50	6D24	E743	07704	6D24
8E48	6D25	E744	07705	6D25
8E59	6D26	E745	07706	6D26
8E64	6D27	E746	07707	6D27
8E60	6D28	E747	07708	6D28
8E2A	6D29	E748	07709	6D29
8E63	6D2A	E749	07710	6D2A
8E55	6D2B	E74A	07711	6D2B
8E76	6D2C	E74B	07712	6D2C
8E72	6D2D	E74C	07713	6D2D
8E7C	6D2E	E74D	07714	6D2E
8E81	6D2F	E74E	07715	6D2F
8E87	6D30	E74F	07716	6D30
8E85	6D31	E750	07717	6D31
8E84	6D32	E751	07718	6D32
8E8B	6D33	E752	07719	6D33
8E8A	6D34	E753	07720	6D34
8E93	6D35	E754	07721	6D35
8E91	6D36	E755	07722	6D36
8E94	6D37	E756	07723	6D37
8E99	6D38	E757	07724	6D38
8EAA	6D39	E758	07725	6D39
E3DE				6D3A
8EAC	6D3B	E75A	07727	6D3B
8EB0	6D3C	E75B	07728	6D3C
8EC6	6D3D	E75C	07729	6D3D
8EB1	6D3E	E75D	07730	6D3E
8EBE	6D3F	E75E	07731	6D3F
8EC5	6D40	E75F	07732	6D40
8EC8	6D41	E760	07733	6D41
8ECB	6D42	E761	07734	6D42
8EDB	6D43	E762	07735	6D43
8EE3	6D44	E763	07736	6D44
8EFC	6D45	E764	07737	6D45
8EFB	6D46	E765	07738	6D46
8EEB	6D47	E766	07739	6D47
8EFE	6D48	E767	07740	6D48
8F0A	6D49	E768	07741	6D49
8F05	6D4A	E769	07742	6D4A
8F15	6D4B	E76A	07743	6D4B
8F12	6D4C	E76B	07744	6D4C
8F19	6D4D	E76C	07745	6D4D
8F13	6D4E	E76D	07746	6D4E
8F1C	6D4F	E76E	07747	6D4F
8F1F	6D50	E76F	07748	6D50
8F1B	6D51	E770	07749	6D51
8F0C	6D52	E771	07750	6D52
8F26	6D53	E772	07751	6D53
E3DF				6D54
8F3B	6D55	E774	07753	6D55
8F39	6D56	E775	07754	6D56
8F45	6D57	E776	07755	6D57
8F42	6D58	E777	07756	6D58
8F3E	6D59	E778	07757	6D59
8F4C	6D5A	E779	07758	6D5A
8F49	6D5B	E77A	07759	6D5B
8F46	6D5C	E77B	07760	6D5C
8F4E	6D5D	E77C	07761	6D5D
8F57	6D5E	E77D	07762	6D5E
8F5C	6D5F	E77E	07763	6D5F
8F62	6D60	E780	07764	6D60
8F63	6D61	E781	07765	6D61
8F64	6D62	E782	07766	6D62
8F9C	6D63	E783	07767	6D63
8F9F	6D64	E784	07768	6D64
8FA3	6D65	E785	07769	6D65
8FAD	6D66	E786	07770	6D66
8FAF	6D67	E787	07771	6D67
8FB7	6D68	E788	07772	6D68
8FDA	6D69	E789	07773	6D69
8FE5	6D6A	E78A	07774	6D6A
8FE2	6D6B	E78B	07775	6D6B
E3E0				6D6C
8FEF	6D6D	E78D	07777	6D6D
E3E1				6D6E
8FF4	6D6F	E78F	07779	6D6F
9005	6D70	E790	07780	6D70
8FF9	6D71	E791	07781	6D71
8FFA	6D72	E792	07782	6D72
9011	6D73	E793	07783	6D73
9015	6D74	E794	07784	6D74
E3E2				6D75
900D	6D76	E796	07786	6D76
901E	6D77	E797	07787	6D77
9016	6D78	E798	07788	6D78
900B	6D79	E799	07789	6D79
9027	6D7A	E79A	07790	6D7A
9036	6D7B	E79B	07791	6D7B
9035	6D7C	E79C	07792	6D7C
9039	6D7D	E79D	07793	6D7D
8FF8	6D7E	E79E	07794	6D7E
904F	6E21	E79F	07801	6E21
9050	6E22	E7A0	07802	6E22
9051	6E23	E7A1	07803	6E23
9052	6E24	E7A2	07804	6E24
900E	6E25	E7A3	07805	6E25
9049	6E26	E7A4	07806	6E26
E3E3				6E27
9056	6E28	E7A6	07808	6E28
E3E4				6E29
905E	6E2A	E7A8	07810	6E2A
9068	6E2B	E7A9	07811	6E2B
906F	6E2C	E7AA	07812	6E2C
9076	6E2D	E7AB	07813	6E2D
96A8	6E2E	E7AC	07814	6E2E
9072	6E2F	E7AD	07815	6E2F
9082	6E30	E7AE	07816	6E30
907D	6E31	E7AF	07817	6E31
9081	6E32	E7B0	07818	6E32
E3E5				6E33
908A	6E34	E7B2	07820	6E34
9089	6E35	E7B3	07821	6E35
908F	6E36	E7B4	07822	6E36
90A8	6E37	E7B5	07823	6E37
90AF	6E38	E7B6	07824	6E38
90B1	6E39	E7B7	07825	6E39
90B5	6E3A	E7B8	07826	6E3A
90E2	6E3B	E7B9	07827	6E3B
E3E6				6E3C
E3E7				6E3D
90DB	6E3E	E7BC	07830	6E3E
9102	6E3F	E7BD	07831	6E3F
9112	6E40	E7BE	07832	6E40
9119	6E41	E7BF	07833	6E41
9132	6E42	E7C0	07834	6E42
9130	6E43	E7C1	07835	6E43
914A	6E44	E7C2	07836	6E44
9156	6E45	E7C3	07837	6E45
9158	6E46	E7C4	07838	6E46
9163	6E47	E7C5	07839	6E47
9165	6E48	E7C6	07840	6E48
9169	6E49	E7C7	07841	6E49
9173	6E4A	E7C8	07842	6E4A
9172	6E4B	E7C9	07843	6E4B
918B	6E4C	E7CA	07844	6E4C
9189	6E4D	E7CB	07845	6E4D
9182	6E4E	E7CC	07846	6E4E
91A2	6E4F	E7CD	07847	6E4F
91AB	6E50	E7CE	07848	6E50
91AF	6E51	E7CF	07849	6E51
91AA	6E52	E7D0	07850	6E52
91B5	6E53	E7D1	07851	6E53
91B4	6E54	E7D2	07852	6E54
91BA	6E55	E7D3	07853	6E55
91C0	6E56	E7D4	07854	6E56
91C1	6E57	E7D5	07855	6E57
91C9	6E58	E7D6	07856	6E58
91CB	6E59	E7D7	07857	6E59
91D0	6E5A	E7D8	07858	6E5A
91D6	6E5B	E7D9	07859	6E5B
91DF	6E5C	E7DA	07860	6E5C
91E1	6E5D	E7DB	07861	6E5D
91DB	6E5E	E7DC	07862	6E5E
E3E8				6E5F
E3E9				6E60
91F6	6E61	E7DF	07865	6E61
921E	6E62	E7E0	07866	6E62
91FF	6E63	E7E1	07867	6E63
9214	6E64	E7E2	07868	6E64
922C	6E65	E7E3	07869	6E65
9215	6E66	E7E4	07870	6E66
9211	6E67	E7E5	07871	6E67
925E	6E68	E7E6	07872	6E68
9257	6E69	E7E7	07873	6E69
9245	6E6A	E7E8	07874	6E6A
9249	6E6B	E7E9	07875	6E6B
9264	6E6C	E7EA	07876	6E6C
9248	6E6D	E7EB	07877	6E6D
9295	6E6E	E7EC	07878	6E6E
923F	6E6F	E7ED	07879	6E6F
924B	6E70	E7EE	07880	6E70
9250	6E71	E7EF	07881	6E71
929C	6E72	E7F0	07882	6E72
9296	6E73	E7F1	07883	6E73
E3EA				6E74
929B	6E75	E7F3	07885	6E75
925A	6E76	E7F4	07886	6E76
92CF	6E77	E7F5	07887	6E77
92B9	6E78	E7F6	07888	6E78
92B7	6E79	E7F7	07889	6E79
92E9	6E7A	E7F8	07890	6E7A
930F	6E7B	E7F9	07891	6E7B
92FA	6E7C	E7FA	07892	6E7C
9344	6E7D	E7FB	07893	6E7D
932E	6E7E	E7FC	07894	6E7E
9319	6F21	E840	07901	6F21
9322	6F22	E841	07902	6F22
931A	6F23	E842	07903	6F23
9323	6F24	E843	07904	6F24
933A	6F25	E844	07905	6F25
9335	6F26	E845	07906	6F26
933B	6F27	E846	07907	6F27
935C	6F28	E847	07908	6F28
9360	6F29	E848	07909	6F29
937C	6F2A	E849	07910	6F2A
E3EB				6F2B
9356	6F2C	E84B	07912	6F2C
93B0	6F2D	E84C	07913	6F2D
93AC	6F2E	E84D	07914	6F2E
93AD	6F2F	E84E	07915	6F2F
9394	6F30	E84F	07916	6F30
93B9	6F31	E850	07917	6F31
93D6	6F32	E851	07918	6F32
93D7	6F33	E852	07919	6F33
93E8	6F34	E853	07920	6F34
93E5	6F35	E854	07921	6F35
93D8	6F36	E855	07922	6F36
93C3	6F37	E856	07923	6F37
93DD	6F38	E857	07924	6F38
93D0	6F39	E858	07925	6F39
93C8	6F3A	E859	07926	6F3A
93E4	6F3B	E85A	07927	6F3B
941A	6F3C	E85B	07928	6F3C
9414	6F3D	E85C	07929	6F3D
9413	6F3E	E85D	07930	6F3E
9403	6F3F	E85E	07931	6F3F
9407	6F40	E85F	07932	6F40
9410	6F41	E860	07933	6F41
9436	6F42	E861	07934	6F42
942B	6F43	E862	07935	6F43
9435	6F44	E863	07936	6F44
9421	6F45	E864	07937	6F45
943A	6F46	E865	07938	6F46
9441	6F47	E866	07939	6F47
9452	6F48	E867	07940	6F48
9444	6F49	E868	07941	6F49
945B	6F4A	E869	07942	6F4A
9460	6F4B	E86A	07943	6F4B
9462	6F4C	E86B	07944	6F4C
945E	6F4D	E86C	07945	6F4D
946A	6F4E	E86D	07946	6F4E
9229	6F4F	E86E	07947	6F4F
9470	6F50	E86F	07948	6F50
9475	6F51	E870	07949	6F51
E3EC				6F52
947D	6F53	E872	07951	6F53
945A	6F54	E873	07952	6F54
947C	6F55	E874	07953	6F55
947E	6F56	E875	07954	6F56
9481	6F57	E876	07955	6F57
E3ED				6F58
9582	6F59	E878	07957	6F59
9587	6F5A	E879	07958	6F5A
958A	6F5B	E87A	07959	6F5B
E3EE				6F5C
9596	6F5D	E87C	07961	6F5D
9598	6F5E	E87D	07962	6F5E
9599	6F5F	E87E	07963	6F5F
95A0	6F60	E880	07964	6F60
95A8	6F61	E881	07965	6F61
95A7	6F62	E882	07966	6F62
95AD	6F63	E883	07967	6F63
95BC	6F64	E884	07968	6F64
E3EF				6F65
95B9	6F66	E886	07970	6F66
95BE	6F67	E887	07971	6F67
95CA	6F68	E888	07972	6F68
6FF6	6F69	E889	07973	6F69
95C3	6F6A	E88A	07974	6F6A
95CD	6F6B	E88B	07975	6F6B
95CC	6F6C	E88C	07976	6F6C
95D5	6F6D	E88D	07977	6F6D
95D4	6F6E	E88E	07978	6F6E
95D6	6F6F	E88F	07979	6F6F
95DC	6F70	E890	07980	6F70
95E1	6F71	E891	07981	6F71
95E5	6F72	E892	07982	6F72
95E2	6F73	E893	07983	6F73
9621	6F74	E894	07984	6F74
9628	6F75	E895	07985	6F75
962E	6F76	E896	07986	6F76
962F	6F77	E897	07987	6F77
9642	6F78	E898	07988	6F78
964C	6F79	E899	07989	6F79
964F	6F7A	E89A	07990	6F7A
964B	6F7B	E89B	07991	6F7B
9677	6F7C	E89C	07992	6F7C
965C	6F7D	E89D	07993	6F7D
965E	6F7E	E89E	07994	6F7E
965D	7021	E89F	08001	7021
965F	7022	E8A0	08002	7022
9666	7023	E8A1	08003	7023
9672	7024	E8A2	08004	7024
966C	7025	E8A3	08005	7025
968D	7026	E8A4	08006	7026
9698	7027	E8A5	08007	7027
9695	7028	E8A6	08008	7028
9697	7029	E8A7	08009	7029
96AA	702A	E8A8	08010	702A
96A7	702B	E8A9	08011	702B
96B1	702C	E8AA	08012	702C
96B2	702D	E8AB	08013	702D
96B0	702E	E8AC	08014	702E
96B4	702F	E8AD	08015	702F
96B6	7030	E8AE	08016	7030
96B8	7031	E8AF	08017	7031
96B9	7032	E8B0	08018	7032
E3F0				7033
96CB	7034	E8B2	08020	7034
96C9	7035	E8B3	08021	7035
96CD	7036	E8B4	08022	7036
894D	7037	E8B5	08023	7037
96DC	7038	E8B6	08024	7038
970D	7039	E8B7	08025	7039
96D5	703A	E8B8	08026	703A
96F9	703B	E8B9	08027	703B
9704	703C	E8BA	08028	703C
E3F1				703D
9708	703E	E8BC	08030	703E
9713	703F	E8BD	08031	703F
970E	7040	E8BE	08032	7040
9711	7041	E8BF	08033	7041
970F	7042	E8C0	08034	7042
9716	7043	E8C1	08035	7043
9719	7044	E8C2	08036	7044
E3F2				7045
972A	7046	E8C4	08038	7046
9730	7047	E8C5	08039	7047
9739	7048	E8C6	08040	7048
973D	7049	E8C7	08041	7049
973E	704A	E8C8	08042	704A
9744	704B	E8C9	08043	704B
9746	704C	E8CA	08044	704C
9748	704D	E8CB	08045	704D
9742	704E	E8CC	08046	704E
9749	704F	E8CD	08047	704F
975C	7050	E8CE	08048	7050
E3F3				7051
9764	7052	E8D0	08050	7052
9766	7053	E8D1	08051	7053
9768	7054	E8D2	08052	7054
52D2	7055	E8D3	08053	7055
E3F4				7056
E3F5				7057
9779	7058	E8D6	08056	7058
9785	7059	E8D7	08057	7059
977C	705A	E8D8	08058	705A
9781	705B	E8D9	08059	705B
977A	705C	E8DA	08060	705C
9786	705D	E8DB	08061	705D
978B	705E	E8DC	08062	705E
978F	705F	E8DD	08063	705F
9790	7060	E8DE	08064	7060
979C	7061	E8DF	08065	7061
97A8	7062	E8E0	08066	7062
97A6	7063	E8E1	08067	7063
97A3	7064	E8E2	08068	7064
97B3	7065	E8E3	08069	7065
97B4	7066	E8E4	08070	7066
97C3	7067	E8E5	08071	7067
97C6	7068	E8E6	08072	7068
97C8	7069	E8E7	08073	7069
97CB	706A	E8E8	08074	706A
97DC	706B	E8E9	08075	706B
97ED	706C	E8EA	08076	706C
9F4F	706D	E8EB	08077	706D
97F2	706E	E8EC	08078	706E
7ADF	706F	E8ED	08079	706F
97F6	7070	E8EE	08080	7070
97F5	7071	E8EF	08081	7071
980F	7072	E8F0	08082	7072
E3F6				7073
981A	375B	8C7A	02359	7074
E3F7				7075
9821	7076	E8F4	08086	7076
9837	7077	E8F5	08087	7077
983D	7078	E8F6	08088	7078
9846	7079	E8F7	08089	7079
984F	707A	E8F8	08090	707A
984B	707B	E8F9	08091	707B
986B	707C	E8FA	08092	707C
986F	707D	E8FB	08093	707D
9870	707E	E8FC	08094	707E
9871	7121	E940	08101	7121
9874	7122	E941	08102	7122
E3F8				7123
98AA	7124	E943	08104	7124
98AF	7125	E944	08105	7125
98B1	7126	E945	08106	7126
98B6	7127	E946	08107	7127
98C4	7128	E947	08108	7128
98C3	7129	E948	08109	7129
98C6	712A	E949	08110	712A
98E9	712B	E94A	08111	712B
98EB	712C	E94B	08112	712C
E3F9				712D
9909	712E	E94D	08114	712E
9912	712F	E94E	08115	712F
9914	7130	E94F	08116	7130
9918	7131	E950	08117	7131
9921	7132	E951	08118	7132
991D	7133	E952	08119	7133
991E	7134	E953	08120	7134
E3FA				7135
9920	7136	E955	08122	7136
992C	7137	E956	08123	7137
992E	7138	E957	08124	7138
993D	7139	E958	08125	7139
993E	713A	E959	08126	713A
9942	713B	E95A	08127	713B
9949	713C	E95B	08128	713C
9945	713D	E95C	08129	713D
9950	713E	E95D	08130	713E
994B	713F	E95E	08131	713F
9951	7140	E95F	08132	7140
9952	7141	E960	08133	7141
994C	7142	E961	08134	7142
9955	7143	E962	08135	7143
9997	7144	E963	08136	7144
9998	7145	E964	08137	7145
99A5	7146	E965	08138	7146
99AD	7147	E966	08139	7147
99AE	7148	E967	08140	7148
E3FB				7149
99DF	714A	E969	08142	714A
E3FC				714B
99DD	714C	E96B	08144	714C
99D8	714D	E96C	08145	714D
99D1	714E	E96D	08146	714E
99ED	714F	E96E	08147	714F
E3FD				7150
99F1	7151	E970	08149	7151
99F2	7152	E971	08150	7152
99FB	7153	E972	08151	7153
99F8	7154	E973	08152	7154
9A01	7155	E974	08153	7155
9A0F	7156	E975	08154	7156
9A05	7157	E976	08155	7157
99E2	7158	E977	08156	7158
9A19	7159	E978	08157	7159
9A2B	715A	E979	08158	715A
E3FE				715B
9A45	715C	E97B	08160	715C
9A42	715D	E97C	08161	715D
9A40	715E	E97D	08162	715E
9A43	715F	E97E	08163	715F
9A3E	7160	E980	08164	7160
9A55	7161	E981	08165	7161
9A4D	7162	E982	08166	7162
9A5B	7163	E983	08167	7163
9A57	7164	E984	08168	7164
E3FF				7165
9A62	7166	E986	08170	7166
9A65	7167	E987	08171	7167
9A64	7168	E988	08172	7168
9A69	7169	E989	08173	7169
9A6B	716A	E98A	08174	716A
9A6A	716B	E98B	08175	716B
9AAD	716C	E98C	08176	716C
9AB0	716D	E98D	08177	716D
9ABC	716E	E98E	08178	716E
9AC0	716F	E98F	08179	716F
9ACF	7170	E990	08180	7170
9AD1	7171	E991	08181	7171
9AD3	7172	E992	08182	7172
9AD4	7173	E993	08183	7173
9ADE	7174	E994	08184	7174
9ADF	7175	E995	08185	7175
9AE2	7176	E996	08186	7176
9AE3	7177	E997	08187	7177
9AE6	7178	E998	08188	7178
9AEF	7179	E999	08189	7179
9AEB	717A	E99A	08190	717A
9AEE	717B	E99B	08191	717B
9AF4	717C	E99C	08192	717C
9AF1	717D	E99D	08193	717D
9AF7	717E	E99E	08194	717E
9AFB	7221	E99F	08201	7221
9B06	7222	E9A0	08202	7222
9B18	7223	E9A1	08203	7223
9B1A	7224	E9A2	08204	7224
9B1F	7225	E9A3	08205	7225
9B22	7226	E9A4	08206	7226
9B23	7227	E9A5	08207	7227
9B25	7228	E9A6	08208	7228
9B27	7229	E9A7	08209	7229
9B28	722A	E9A8	08210	722A
9B29	722B	E9A9	08211	722B
9B2A	722C	E9AA	08212	722C
E400				722D
9B2F	722E	E9AC	08214	722E
9B32	722F	E9AD	08215	722F
9B44	7230	E9AE	08216	7230
9B43	7231	E9AF	08217	7231
9B4F	7232	E9B0	08218	7232
E401				7233
9B4E	7234	E9B2	08220	7234
9B51	7235	E9B3	08221	7235
9B58	7236	E9B4	08222	7236
9B74	7237	E9B5	08223	7237
9B93	7238	E9B6	08224	7238
9B83	7239	E9B7	08225	7239
9B91	723A	E9B8	08226	723A
9B96	723B	E9B9	08227	723B
E402				723C
9B9F	723D	E9BB	08229	723D
9BA0	723E	E9BC	08230	723E
9BA8	723F	E9BD	08231	723F
9BB4	7240	E9BE	08232	7240
9BC0	7241	E9BF	08233	7241
9BCA	7242	E9C0	08234	7242
9BB9	7243	E9C1	08235	7243
9BC6	7244	E9C2	08236	7244
9BCF	7245	E9C3	08237	7245
9BD1	7246	E9C4	08238	7246
9BD2	7247	E9C5	08239	7247
9BE3	7248	E9C6	08240	7248
9BE2	7249	E9C7	08241	7249
9BE4	724A	E9C8	08242	724A
9BD4	724B	E9C9	08243	724B
9BE1	724C	E9CA	08244	724C
9BF5	3033	88B1	01619	724D
E403				724E
E404				724F
9BF0	7250	E9CE	08248	7250
9C15	7251	E9CF	08249	7251
9C14	7252	E9D0	08250	7252
9C09	7253	E9D1	08251	7253
9C13	7254	E9D2	08252	7254
9C0C	7255	E9D3	08253	7255
9C06	7256	E9D4	08254	7256
9C08	7257	E9D5	08255	7257
9C12	7258	E9D6	08256	7258
9C0A	7259	E9D7	08257	7259
9C04	725A	E9D8	08258	725A
9C2E	725B	E9D9	08259	725B
9C1B	725C	E9DA	08260	725C
E405				725D
9C24	725E	E9DC	08262	725E
9C21	725F	E9DD	08263	725F
9C30	7260	E9DE	08264	7260
9C47	7261	E9DF	08265	7261
9C32	7262	E9E0	08266	7262
9C46	7263	E9E1	08267	7263
9C3E	7264	E9E2	08268	7264
9C5A	7265	E9E3	08269	7265
9C60	7266	E9E4	08270	7266
9C67	7267	E9E5	08271	7267
E406				7268
9C78	7269	E9E7	08273	7269
9CE7	726A	E9E8	08274	726A
9CEC	726B	E9E9	08275	726B
9CF0	726C	E9EA	08276	726C
9D09	726D	E9EB	08277	726D
9D08	726E	E9EC	08278	726E
9CEB	726F	E9ED	08279	726F
9D03	7270	E9EE	08280	7270
9D06	7271	E9EF	08281	7271
9D2A	7272	E9F0	08282	7272
9D26	7273	E9F1	08283	7273
9D2C	3229	89A7	01809	7274
9D23	7275	E9F3	08285	7275
9D1F	7276	E9F4	08286	7276
9D44	7277	E9F5	08287	7277
9D15	7278	E9F6	08288	7278
9D12	7279	E9F7	08289	7279
E407				727A
9D3F	727B	E9F9	08291	727B
9D3E	727C	E9FA	08292	727C
9D46	727D	E9FB	08293	727D
9D48	727E	E9FC	08294	727E
9D5D	7321	EA40	08301	7321
9D5E	7322	EA41	08302	7322
9D64	7323	EA42	08303	7323
9D51	7324	EA43	08304	7324
9D50	7325	EA44	08305	7325
9D59	7326	EA45	08306	7326
9D72	7327	EA46	08307	7327
9D89	7328	EA47	08308	7328
9D87	7329	EA48	08309	7329
9DAB	732A	EA49	08310	732A
9D6F	732B	EA4A	08311	732B
9D7A	732C	EA4B	08312	732C
9D9A	732D	EA4C	08313	732D
9DA4	732E	EA4D	08314	732E
9DA9	732F	EA4E	08315	732F
9DB2	7330	EA4F	08316	7330
9DC4	7331	EA50	08317	7331
9DC1	7332	EA51	08318	7332
9DBB	7333	EA52	08319	7333
9DB8	7334	EA53	08320	7334
9DBA	7335	EA54	08321	7335
9DC6	7336	EA55	08322	7336
9DCF	7337	EA56	08323	7337
9DC2	7338	EA57	08324	7338
E408				7339
9DD3	733A	EA59	08326	733A
9DF8	733B	EA5A	08327	733B
9DE6	733C	EA5B	08328	733C
9DED	733D	EA5C	08329	733D
9DEF	733E	EA5D	08330	733E
9DFD	733F	EA5E	08331	733F
9E1A	7340	EA5F	08332	7340
9E1B	7341	EA60	08333	7341
9E1E	7342	EA61	08334	7342
9E75	7343	EA62	08335	7343
9E79	7344	EA63	08336	7344
9E7D	7345	EA64	08337	7345
9E81	7346	EA65	08338	7346
9E88	7347	EA66	08339	7347
9E8B	7348	EA67	08340	7348
9E8C	7349	EA68	08341	7349
9E92	734A	EA69	08342	734A
9E95	734B	EA6A	08343	734B
9E91	734C	EA6B	08344	734C
9E9D	734D	EA6C	08345	734D
9EA5	734E	EA6D	08346	734E
9EA9	734F	EA6E	08347	734F
9EB8	7350	EA6F	08348	7350
E409				7351
9EAD	7352	EA71	08350	7352
E40A				7353
9ECC	7354	EA73	08352	7354
9ECE	7355	EA74	08353	7355
9ECF	7356	EA75	08354	7356
9ED0	7357	EA76	08355	7357
9ED4	7358	EA77	08356	7358
9EDC	7359	EA78	08357	7359
9EDE	735A	EA79	08358	735A
9EDD	735B	EA7A	08359	735B
9EE0	735C	EA7B	08360	735C
9EE5	735D	EA7C	08361	735D
9EE8	735E	EA7D	08362	735E
9EEF	735F	EA7E	08363	735F
9EF4	7360	EA80	08364	7360
9EF6	7361	EA81	08365	7361
9EF7	7362	EA82	08366	7362
9EF9	7363	EA83	08367	7363
9EFB	7364	EA84	08368	7364
9EFC	7365	EA85	08369	7365
9EFD	7366	EA86	08370	7366
E40B				7367
E40C				7368
76B7	7369	EA89	08373	7369
9F15	736A	EA8A	08374	736A
9F21	736B	EA8B	08375	736B
E40D				736C
9F3E	736D	EA8D	08377	736D
9F4A	736E	EA8E	08378	736E
9F52	736F	EA8F	08379	736F
9F54	7370	EA90	08380	7370
9F63	7371	EA91	08381	7371
9F5F	7372	EA92	08382	7372
9F60	7373	EA93	08383	7373
9F61	7374	EA94	08384	7374
9F66	7375	EA95	08385	7375
9F67	7376	EA96	08386	7376
9F6C	7377	EA97	08387	7377
9F6A	7378	EA98	08388	7378
9F77	7379	EA99	08389	7379
9F72	737A	EA9A	08390	737A
9F76	737B	EA9B	08391	737B
9F95	737C	EA9C	08392	737C
E40E				737D
9FA0	737E	EA9E	08394	737E
E40F				A1A1
E410				A1A2
E411				A1A3
E412				A1A4
E413				A1A5
E414				A1A6
E415				A1A7
E416				A1A8
E417				A1A9
E418				A1AA
E419				A1AB
2488				A1AC
2489				A1AD
248A				A1AE
248B				A1AF
248C				A1B0
248D				A1B1
248E				A1B2
248F				A1B3
2490				A1B4
E41A				A1B5
E41B				A1B6
E41C				A1B7
E41D				A1B8
E41E				A1B9
E41F				A1BA
E420				A1BB
E421				A1BC
E422				A1BD
E423				A1BE
E424				A1BF
E425				A1C0
E426				A1C1
E427				A1C2
E428				A1C3
E429				A1C4
E42A				A1C5
E42B				A1C6
E42C				A1C7
E42D				A1C8
2474				A1C9
2475				A1CA
2476				A1CB
2477				A1CC
2478				A1CD
2479				A1CE
247A				A1CF
247B				A1D0
247C				A1D1
247D				A1D2
247E				A1D3
247F				A1D4
2480				A1D5
2481				A1D6
2482				A1D7
2483				A1D8
2484				A1D9
2485				A1DA
2486				A1DB
2487				A1DC
24B6				A1DD
24B7				A1DE
24B8				A1DF
24B9				A1E0
24BA				A1E1
24BB				A1E2
24BC				A1E3
24BD				A1E4
24BE				A1E5
24BF				A1E6
24C0				A1E7
24D0				A1E8
24D1				A1E9
24D2				A1EA
24D3				A1EB
24D4				A1EC
24D5				A1ED
24D6				A1EE
24D7				A1EF
24D8				A1F0
24D9				A1F1
24DA				A1F2
32D1				A1F3
32FA				A1F4
32E9				A1F5
32E5				A1F6
32ED				A1F7
32EC				A1F8
32E3				A1F9
32E0				A1FA
32F7				A1FB
32E6				A1FC
32D0				A1FD
32D2				A1FE
32D3				A2A1
32D4				A2A2
32D5				A2A3
32D6				A2A4
32D7				A2A5
32D8				A2A6
32D9				A2A7
E42E				A2A8
E42F				A2A9
E430				A2AA
E431				A2AB
E432				A2AC
E433				A2AD
E434				A2AE
E435				A2AF
E436				A2B0
E437				A2B1
E438				A2B2
E439				A2B3
E43A				A2B4
E43B				A2B5
E43C				A2B6
E43D				A2B7
E43E				A2B8
E43F				A2B9
E440				A2BA
E441				A2BB
E442				A2BC
2170		FA40	11501	A2BD
2171		FA41	11502	A2BE
2172		FA42	11503	A2BF
2173		FA43	11504	A2C0
2174		FA44	11505	A2C1
2175		FA45	11506	A2C2
2176		FA46	11507	A2C3
2177		FA47	11508	A2C4
2178		FA48	11509	A2C5
2179		FA49	11510	A2C6
E443				A2C7
E444				A2C8
E445				A2C9
E446				A2CA
E447				A2CB
E448				A2CC
E449				A2CD
E44A				A2CE
E44B				A2CF
E44C				A2D0
216A				A2D1
216B				A2D2
E44D				A2D3
E44E				A2D4
E44F				A2D5
E450				A2D6
E451				A2D7
E452				A2D8
E453				A2D9
E454				A2DA
217A				A2DB
217B				A2DC
E455				A2DD
E456				A2DE
E457				A2DF
E458				A2E0
E459				A2E1
E45A				A2E2
E45B				A2E3
E45C				A2E4
3280				A2E5
3281				A2E6
3282				A2E7
3283				A2E8
3284				A2E9
3285				A2EA
3286				A2EB
3287				A2EC
3288				A2ED
3289				A2EE
3220				A2EF
3221				A2F0
3222				A2F1
3223				A2F2
3224				A2F3
3225				A2F4
3226				A2F5
3227				A2F6
3228				A2F7
3229				A2F8
E45D				A2F9
2153				A2FA
2154				A2FB
E45E				A2FC
E45F				A2FD
E460				A3A1
E461				A3A2
E462				A3A3
E463				A3A4
E464				A3A5
E465				A3A6
E466				A3A7
E467				A3A8
E468				A3A9
E469				A3AA
E46A				A3AB
E46B				A3AC
E46C				A3AD
E46D				A3AE
E46E				A3AF
E46F				A3B0
E470				A3B1
E471				A3B2
E472				A3B3
E473				A3B4
E474				A3B5
E475				A3B6
E476				A3B7
E477				A3B8
E478				A3B9
E479				A3BA
24C1				A3BB
24C2				A3BC
24C3				A3BD
24C4				A3BE
24C5				A3BF
24C6				A3C0
24C7				A3C1
24C8				A3C2
24C9				A3C3
24CA				A3C4
24CB				A3C5
24CC				A3C6
24CD				A3C7
24CE				A3C8
24CF				A3C9
E47A				A3CA
E47B				A3CB
E47C				A3CC
E47D				A3CD
E47E				A3CE
E47F				A3CF
E480				A3D0
E481				A3D1
E482				A3D2
E483				A3D3
E484				A3D4
E485				A3D5
E486				A3D6
E487				A3D7
E488				A3D8
E489				A3D9
E48A				A3DA
E48B				A3DB
E48C				A3DC
21C4				A3DD
2194				A3DE
E48D				A3DF
2195				A3E0
2197				A3E1
2196				A3E2
2199				A3E3
2198				A3E4
E48E				A3E5
E48F				A3E6
E490				A3E7
E491				A3E8
E492				A3E9
E493				A3EA
25C1				A3EB
25B7				A3EC
25C0				A3ED
25B6				A3EE
2776				A3EF
2777				A3F0
2778				A3F1
2779				A3F2
277A				A3F3
277B				A3F4
277C				A3F5
277D				A3F6
277E				A3F7
277F				A3F8
E494				A3F9
E495				A3FA
201B				A3FB
201F				A3FC
249C				A4A1
249D				A4A2
249E				A4A3
249F				A4A4
24A0				A4A5
24A1				A4A6
24A2				A4A7
24A3				A4A8
24A4				A4A9
24A5				A4AA
24A6				A4AB
24A7				A4AC
24A8				A4AD
24A9				A4AE
24AA				A4AF
24AB				A4B0
24AC				A4B1
24AD				A4B2
24AE				A4B3
24AF				A4B4
24B0				A4B5
24B1				A4B6
24B2				A4B7
24B3				A4B8
24B4				A4B9
24B5				A4BA
24DB				A4BB
24DC				A4BC
24DD				A4BD
24DE				A4BE
24DF				A4BF
24E0				A4C0
24E1				A4C1
24E2				A4C2
24E3				A4C3
24E4				A4C4
24E5				A4C5
24E6				A4C6
24E7				A4C7
24E8				A4C8
24E9				A4C9
E496				A4CA
E497				A4CB
E498				A4CC
E499				A4CD
E49A				A4CE
E49B				A4CF
E49C				A4D0
E49D				A4D1
E49E				A4D2
E49F				A4D3
E4A0				A4D4
E4A1				A4D5
E4A2				A4D6
E4A3				A4D7
E4A4				A4D8
E4A5				A4D9
E4A6				A4DA
E4A7				A4DB
E4A8				A4DC
E4A9				A4DD
E4AA				A4DE
212B	2272	81F0	00282	A4DF
217F				A4E0
33CC				A4E1
E4AB				A4E2
339F				A4E3
33A0				A4E4
33A2				A4E5
33CA				A4E6
E4AC				A4E7
E4AD				A4E8
33A4				A4E9
33A5				A4EA
E4AE				A4EB
E4AF				A4EC
E4B0				A4ED
E4B1				A4EE
E4B2				A4EF
E4B3				A4F0
E4B4				A4F1
E4B5				A4F2
E4B6				A4F3
3390				A4F4
2109				A4F5
E4B7				A4F6
33CB				A4F7
33BE				A4F8
E4B8				A4F9
3388				A4FA
E4B9				A4FB
E4BA				A4FC
33D9				A4FD
2030	2273	81F1	00283	A4FE
00C1				A5A1
00C2				A5A2
04D2				A5A3
00C7				A5A4
00C9				A5A5
00CA				A5A6
00CD				A5A7
00CE				A5A8
00D1				A5A9
00D3				A5AA
00D4				A5AB
00D6				A5AC
0152				A5AD
00DA				A5AE
00DB				A5AF
00DC				A5B0
00DD				A5B1
00E1				A5B2
00E0				A5B3
00E2				A5B4
00E4				A5B5
00E7				A5B6
00E9				A5B7
00E8				A5B8
00EA				A5B9
00EB				A5BA
00ED				A5BB
00EE				A5BC
00CF				A5BD
00F1				A5BE
00F3				A5BF
00F4				A5C0
00F6				A5C1
0153				A5C2
E4BB				A5C3
00FA				A5C4
00F9				A5C5
00FB				A5C6
00FC				A5C7
00FF				A5C8
E4BC				A5C9
2202	225F	81DD	00263	A5CA
0454				A5CB
E4BD				A5CC
E4BE				A5CD
E4BF				A5CE
2310				A5CF
2227	224A	81C8	00242	A5D0
2228	224B	81C9	00243	A5D1
2282	223E	81BC	00230	A5D2
2283	223F	81BD	00231	A5D3
2208	223A	81B8	00226	A5D4
220B	223B	81B9	00227	A5D5
2213				A5D6
2243				A5D7
2264				A5D8
2265				A5D9
2272				A5DA
2273				A5DB
221D	2267	81E5	00271	A5DC
223D	2266	81E4	00270	A5DD
E4C0				A5DE
2225	2142	8161	00134	A5DF
2262				A5E0
226A	2263	81E1	00267	A5E1
226B	2264	81E2	00268	A5E2
2277				A5E3
2276				A5E4
E4C1				A5E5
E4C2				A5E6
2200	224F	81CD	00247	A5E7
2203	2250	81CE	00248	A5E8
2295				A5E9
2296				A5EA
2297				A5EB
E4C3				A5EC
229C				A5ED
2020	2277	81F5	00287	A5EE
2021	2278	81F6	00288	A5EF
00B6	2279	81F7	00289	A5F0
E4C4				A5F1
2663				A5F2
2666				A5F3
2665				A5F4
2660				A5F5
E4C5				A5F6
E4C6				A5F7
E4C7				A5F8
E4C8				A5F9
E4C9				A5FA
FFE4		FA55	11522	A5FB
E4CA				A5FC
E4CB				A5FD
E4CC				A5FE
3302				A6A1
E4CD				A6A2
3308				A6A3
E4CE				A6A4
E4CF				A6A5
330C				A6A6
E4D0				A6A7
3310				A6A8
3315				A6A9
E4D1				A6AA
3317				A6AB
E4D2				A6AC
331F				A6AD
E4D3				A6AE
3325				A6AF
E4D4				A6B0
332D				A6B1
3333				A6B2
E4D5				A6B3
333D				A6B4
E4D6				A6B5
E4D7				A6B6
E4D8				A6B7
3343				A6B8
E4D9				A6B9
E4DA				A6BA
3348				A6BB
334B				A6BC
E4DB				A6BD
E4DC				A6BE
E4DD				A6BF
E4DE				A6C0
E4DF				A6C1
3321				A6C2
E4E0				A6C3
3337				A6C4
E4E1				A6C5
E4E2				A6C6
3352				A6C7
3354				A6C8
E4E3				A6C9
323E				A6CA
3234				A6CB
3233				A6CC
3236				A6CD
E4E4				A6CE
3242				A6CF
3243				A6D0
323D				A6D1
323F				A6D2
3238				A6D3
3230				A6D4
322A				A6D5
322B				A6D6
322C				A6D7
322D				A6D8
322E				A6D9
322F				A6DA
3241				A6DB
3237				A6DC
323A				A6DD
329E				A6DE
3299				A6DF
3007	213B	815A	00127	A6F0
2642	2169	8189	00173	A6F1
2286	223C	81BA	00228	A6F4
2287	223D	81BB	00229	A6F5
21D2	224D	81CB	00245	A6F6
21D4	224E	81CC	00246	A6F7
2312	225E	81DC	00262	A6F8
2207	2260	81DE	00264	A6F9
222C	226A	81E8	00274	A6FA
266F	2274	81F2	00284	A6FB
266D	2275	81F3	00285	A6FC
266A	2276	81F4	00286	A6FD
25EF	227E	81FC	00294	A6FE
E4E5				B0A1
E4E6				B0A2
E4E7				B0A3
E4E8				B0A4
E4E9				B0A5
E4EA				B0A6
E4EB				B0A7
E4EC				B0A8
E4ED				B0A9
E4EE				B0AA
E4EF				B0AB
E4F0				B0AC
E4F1				B0AD
E4F2				B0AE
E4F3				B0AF
E4F4				B0B0
E4F5				B0B1
E4F6				B0B2
E4F7				B0B3
E4F8				B0B4
E4F9				B0B5
E4FA				B0B6
E4FB				B0B7
E4FC				B0B8
E4FD				B0B9
E4FE				B0BA
E4FF				B0BB
E500				B0BC
E501				B0BD
E502				B0BE
E503				B0BF
E504				B0C0
E505				B0C1
E506				B0C2
E507				B0C3
E508				B0C4
E509				B0C5
E50A				B0C6
E50B				B0C7
E50C				B0C8
E50D				B0C9
E50E				B0CA
E50F				B0CB
E510				B0CC
E511				B0CD
E512				B0CE
E513				B0CF
E514				B0D0
E515				B0D1
E516				B0D2
E517				B0D3
E518				B0D4
E519				B0D5
E51A				B0D6
E51B				B0D7
E51C				B0D8
E51D				B0D9
E51E				B0DA
E51F				B0DB
E520				B0DC
E521				B0DD
E522				B0DE
E523				B0DF
E524				B0E0
E525				B0E1
E526				B0E2
E527				B0E3
E528				B0E4
E529				B0E5
E52A				B0E6
E52B				B0E7
E52C				B0E8
E52D				B0E9
E52E				B0EA
E52F				B0EB
E530				B0EC
E531				B0ED
E532				B0EE
E533				B0EF
E534				B0F0
E535				B0F1
E536				B0F2
E537				B0F3
E538				B0F4
E539				B0F5
E53A				B0F6
E53B				B0F7
E53C				B0F8
E53D				B0F9
E53E				B0FA
E53F				B0FB
E540				B0FC
E541				B0FD
E542				B0FE
E543				B1A1
E544				B1A2
E545				B1A3
E546				B1A4
E547				B1A5
E548				B1A6
E549				B1A7
E54A				B1A8
E54B				B1A9
E54C				B1AA
E54D				B1AB
E54E				B1AC
E54F				B1AD
E550				B1AE
E551				B1AF
E552				B1B0
E553				B1B1
E554				B1B2
E555				B1B3
E556				B1B4
E557				B1B5
E558				B1B6
E559				B1B7
E55A				B1B8
E55B				B1B9
E55C				B1BA
E55D				B1BB
E55E				B1BC
E55F				B1BD
E560				B1BE
E561				B1BF
E562				B1C0
E563				B1C1
E564				B1C2
E565				B1C3
E566				B1C4
E567				B1C5
E568				B1C6
E569				B1C7
E56A				B1C8
E56B				B1C9
E56C				B1CA
E56D				B1CB
E56E				B1CC
E56F				B1CD
E570				B1CE
E571				B1CF
E572				B1D0
E573				B1D1
E574				B1D2
E575				B1D3
E576				B1D4
E577				B1D5
E578				B1D6
E579				B1D7
E57A				B1D8
E57B				B1D9
E57C				B1DA
4E02				B2A1
E57D				B2A2
E57E				B2A3
EB69				B2A4
E57F				B2A5
E580				B2A6
5F0E				B2A7
E581				B2A8
E582				B2A9
E583				B2AA
4E28		FA68	11541	B2AB
4E2B				B2AC
4E30				B2AD
E584				B2AE
4E2E				B2AF
E585				B2B0
4E44				B2B1
4E74				B2B2
E586				B2B3
4E8D				B2B4
E587				B2B5
E588				B2B6
E589				B2B7
E58A				B2B8
E58B				B2B9
E58C				B2BA
4EB9				B2BB
E58D				B2BC
EB6A				B2BD
E58E				B2BE
4ED0				B2BF
E58F				B2C0
E590				B2C1
4EE1		FA69	11542	B2C2
E591				B2C3
4EE8				B2C4
E592				B2C5
4F0B				B2C6
4F08				B2C7
4EFF				B2C8
E593				B2C9
E594				B2CA
4F12				B2CB
4EF1				B2CC
4F00		FA6B	11544	B2CD
4F03		FA6C	11545	B2CE
4F15				B2CF
4EFD				B2D0
E595				B2D1
4EF5				B2D2
EB6B				B2D3
4F54				B2D4
4F3E				B2D5
4F56		FA6E	11547	B2D6
4F48				B2D7
4F3B				B2D8
4F4B				B2D9
4F42				B2DA
4F39		FA6D	11546	B2DB
E596				B2DC
E597				B2DD
4F7E				B2DE
4F90				B2DF
4F97				B2E0
4F94		FA72	11551	B2E1
4FA0	3622	8BA0	02202	B2E2
E598				B2E3
E599				B2E4
4F9F				B2E5
4F6E				B2E6
4F8A		FA70	11549	B2E7
4F93				B2E8
4F78				B2E9
4F92		FA6F	11548	B2EA
4F6A				B2EB
E59A				B2EC
E59B				B2ED
4FC5				B2EE
4FC1				B2EF
77E6				B2F0
4FCF				B2F1
4FCB				B2F2
E59C				B2F3
4FBE				B2F4
4FCD		FA73	11552	B2F5
4FC9		FA61	11534	B2F6
4FE2				B2F7
E59D				B2F8
4FB2				B2F9
E59E				B2FA
E59F				B2FB
E5A0				B2FC
501B				B2FD
5018				B2FE
5022		FA75	11554	B3A1
5013				B3A2
5000				B3A3
501C				B3A4
502E				B3A5
501E		FA77	11556	B3A6
4FF2				B3A7
E5A1				B3A8
E5A2				B3A9
4FFD				B3AA
5027				B3AB
5035				B3AC
5032				B3AD
500C				B3AE
500E				B3AF
E5A3				B3B0
E5A4				B3B1
5040		FA74	11553	B3B2
4FFF		FA76	11555	B3B3
E5A5				B3B4
506A				B3B5
506D				B3B6
E5A6				B3B7
E5A7				B3B8
E5A8				B3B9
5057				B3BA
5059				B3BB
5063				B3BC
E5A9				B3BD
5071				B3BE
5066				B3BF
5046		FA78	11557	B3C0
E5AA				B3C1
E5AB				B3C2
5070		FA79	11558	B3C3
5042		FA7A	11559	B3C4
E5AC				B3C5
5094		FA7B	11560	B3C6
509E				B3C7
E5AD				B3C8
E5AE				B3C9
E5AF				B3CA
508F				B3CB
E5B0				B3CC
50C4				B3CD
E5B1				B3CE
50CC				B3CF
E5B2				B3D0
50E9				B3D1
EB6C				B3D2
50E6				B3D3
E5B3				B3D4
50D8		FA7D	11562	B3D5
50D3				B3D6
50F2				B3D7
50D0				B3D8
E5B4				B3D9
5106				B3DA
E5B5				B3DB
50EF				B3DC
510C				B3DD
E5B6				B3DE
5107				B3DF
50E8				B3E0
5103				B3E1
5110				B3E2
511E				B3E3
E5B7				B3E4
5132	4C59	96D7	04457	B3E5
E5B8				B3E6
E5B9				B3E7
514A		FA7E	11563	B3E8
E5BA				B3E9
E5BB				B3EA
7097				B3EB
5155				B3EC
E5BC				B3ED
515F				B3EE
E5BD				B3EF
5164		FA80	11564	B3F0
E5BE				B3F1
E5BF				B3F2
E5C0				B3F3
E5C1				B3F4
517E				B3F5
5189	5147	9966	04939	B3F6
519D		FA81	11565	B3F7
51A1				B3F8
E5C2				B3F9
E5C3				B3FA
E5C4				B3FB
E5C5				B3FC
E5C6				B3FD
E5C7				B3FE
51BC				B4A1
51BE		FA82	11566	B4A2
51BF				B4A3
E5C8				B4A4
E5C9				B4A5
E5CA				B4A6
E5CB				B4A7
51DC	7425	EAA3	08405	B4A8
E5CC				B4A9
51DE				B4AA
51EE				B4AB
51F4				B4AC
5202				B4AD
5201				B4AE
E5CD				B4AF
E5CE				B4B0
E5CF				B4B1
E5D0				B4B2
5216				B4B3
5215		FA84	11568	B4B4
E5D1				B4B5
5222				B4B6
E5D2				B4B7
5231				B4B8
5249				B4B9
5257				B4BA
525A				B4BB
5265	476D	948D	03977	B4BC
E5D3				B4BD
E5D4				B4BE
5275	414F	916E	03347	B4BF
E5D5				B4C0
529C		FA85	11569	B4C1
E5D6				B4C2
52A6		FA86	11570	B4C3
52A7				B4C4
E5D7				B4C5
E5D8				B4C6
52AF		FB77	11756	B4C7
E5D9				B4C8
52B6				B4C9
52B8				B4CA
52B7				B4CB
52C8				B4CC
E5DA				B4CD
E5DB				B4CE
E5DC				B4CF
52C0		FA87	11571	B4D0
52D1				B4D1
E5DD				B4D2
52CF				B4D3
E5DE				B4D4
607F				B4D5
E5DF				B4D6
52D4				B4D7
52DB		FA88	11572	B4D8
E5E0				B4D9
E5E1				B4DA
52E5				B4DB
52EA				B4DC
E5E2				B4DD
52E8				B4DE
E5E3				B4DF
E5E4				B4E0
52F1				B4E1
8FA6				B4E2
52F4				B4E3
52F6				B4E4
E5E5				B4E5
E5E6				B4E6
E5E7				B4E7
5300		FA89	11573	B4E8
5303				B4E9
E5E8				B4EA
E5E9				B4EB
EB6D				B4EC
530A				B4ED
E5EA				B4EE
EB6E				B4EF
5328				B4F0
E5EB				B4F1
533C				B4F2
5342				B4F3
E5EC				B4F4
E5ED				B4F5
E5EE				B4F6
E5EF				B4F7
E5F0				B4F8
684C				B4F9
5359				B4FA
E5F1				B4FB
E5F2				B4FC
E5F3				B4FD
E5F4				B4FE
E5F5				B5A1
E5F6				B5A2
5361				B5A3
5363				B5A4
E5F7				B5A5
536C				B5A6
5372		FA8C	11576	B5A7
E5F8				B5A8
E5F9				B5A9
537F	362A	8BA8	02210	B5AA
E5FA				B5AB
E5FB				B5AC
5393		FA8D	11577	B5AD
E5FC				B5AE
E5FD				B5AF
539D				B5B0
53B2		FA8E	11578	B5B1
64EA				B5B2
53B8				B5B3
53BA				B5B4
53C5				B5B5
E5FE				B5B6
E5FF				B5B7
E600				B5B8
E601				B5B9
53D5				B5BA
53DA				B5BB
53DD		FA8F	11579	B5BC
51A3				B5BD
E602				B5BE
E603				B5BF
53F1	3C38	8EB6	02824	B5C0
E604				B5C1
E605				B5C2
E606				B5C3
E607				B5C4
5434				B5C5
E608				B5C6
E609				B5C7
E60A				B5C8
E60B				B5C9
544D				B5CA
5447				B5CB
547F				B5CC
E60C				B5CD
5466				B5CE
EB6F				B5CF
5464				B5D0
546D				B5D1
5496				B5D2
549C		FA91	11581	B5D3
E60D				B5D4
E60E				B5D5
E60F				B5D6
E610				B5D7
54BF				B5D8
54C6				B5D9
54A1				B5DA
E611				B5DB
54A9		FA93	11583	B5DC
54B1				B5DD
E612				B5DE
5509				B5DF
54FF		FA94	11584	B5E0
5501				B5E1
E613				B5E2
550E				B5E3
54E8	3E25	8FA3	03005	B5E4
608A		FABE	11632	B5E5
E614				B5E6
554D				B5E7
5547				B5E8
5560				B5E9
5550				B5EA
5561				B5EB
554A				B5EC
559D	3365	8A85	01969	B5ED
5586		FA95	11585	B5EE
E615				B5EF
55AD				B5F0
E616				B5F1
5591				B5F2
E617				B5F3
E618				B5F4
E619				B5F5
55C3				B5F6
55DB				B5F7
55E4	5350	9A6F	05148	B5F8
E61A				B5F9
55CE				B5FA
55E9				B5FB
E61B				B5FC
EBBC				B5FD
560F				B5FE
E61C				B6A1
560E				B6A2
5611				B6A3
5610				B6A4
5608				B6A5
564C	4139	9158	03325	B6A6
E61D				B6A7
6198		FAC6	11640	B6A8
563B				B6A9
E61E				B6AA
564D				B6AB
563D				B6AC
5649				B6AD
563F				B6AE
E61F				B6AF
5642	313D	895C	01729	B6B0
E620				B6B1
566F				B6B2
5666				B6B3
5672				B6B4
5678	4655	93D3	03853	B6B5
E621				B6B6
E622				B6B7
E623				B6B8
5685				B6B9
568B				B6BA
569E				B6BB
8B46				B6BC
56AC				B6BD
56AD				B6BE
E624				B6BF
9FA2				B6C0
56CD				B6C1
56D9				B6C2
E625				B6C3
56E4				B6C4
E626				B6C5
570A				B6C6
E627				B6C7
E628				B6C8
E629				B6C9
E62A				B6CA
E62B				B6CB
572F				B6CC
572C				B6CD
E62C				B6CE
5733				B6CF
5734				B6D0
E62D				B6D1
962C				B6D2
574C				B6D3
E62E				B6D4
E62F				B6D5
EB70				B6D6
E630				B6D7
E631				B6D8
577B				B6D9
5C2D	3646	8BC4	02238	B6DA
5767				B6DB
E632				B6DC
E633				B6DD
5765		FA97	11587	B6DE
E634				B6DF
579A				B6E0
578C				B6E1
E635				B6E2
57AC		FA98	11588	B6E3
E636				B6E4
57C8		FA99	11589	B6E5
57C7		FA9A	11590	B6E6
E637				B6E7
E638				B6E8
57F8				B6E9
E639				B6EA
57F5				B6EB
57E4				B6EC
5809				B6ED
57DE				B6EE
E63A				B6EF
E63B				B6F0
9681				B6F1
5820				B6F2
581E				B6F3
9684				B6F4
E63C				B6F5
E63D				B6F6
E63E				B6F7
5839				B6F8
E63F				B6F9
5864				B6FA
E640				B6FB
E641				B6FC
5890				B6FD
587C				B6FE
E642				B7A1
5881				B7A2
5889				B7A3
587F				B7A4
E643				B7A5
588F				B7A6
E644				B7A7
EB71				B7A8
E645				B7A9
5880				B7AA
58A9				B7AB
58A0				B7AC
E646				B7AD
58B2		FA9E	11594	B7AE
58B1				B7AF
E647				B7B0
E648				B7B1
58CE				B7B2
58D4				B7B3
58DA				B7B4
E649				B7B5
58E9				B7B6
5906				B7B7
590B		FA9F	11601	B7B8
E64A				B7B9
590C				B7BA
E64B				B7BB
E64C				B7BC
E64D				B7BD
E64E				B7BE
E64F				B7BF
E650				B7C0
E651				B7C1
6598				B7C2
E652				B7C3
5924				B7C4
5933				B7C5
592F				B7C6
E653				B7C7
E654				B7C8
E655				B7C9
E656				B7CA
E657				B7CB
E658				B7CC
E659				B7CD
595D		FAA2	11604	B7CE
595B		FAA1	11603	B7CF
E65A				B7D0
E65B				B7D1
E65C				B7D2
E65D				B7D3
E65E				B7D4
E65F				B7D5
596B				B7D6
596D				B7D7
5975				B7D8
E660				B7D9
5979				B7DA
59A4		FAA4	11606	B7DB
5992				B7DC
598C				B7DD
598B				B7DE
599F				B7DF
E661				B7E0
59C1				B7E1
59CA				B7E2
59D2				B7E3
59BA		FAA5	11607	B7E4
59C3				B7E5
59C8				B7E6
59AF				B7E7
E662				B7E8
59E4				B7E9
59EE				B7EA
59DD				B7EB
59DE				B7EC
5A24				B7ED
5A23				B7EE
5A13				B7EF
5A63				B7F0
5A6D				B7F1
5A65				B7F2
5A55				B7F3
5A4C				B7F4
5A5E				B7F5
E663				B7F6
5A9F				B7F7
5A7E				B7F8
E664				B7F9
E665				B7FA
5A9E				B7FB
E666				B7FC
5A77				B7FD
5A9C				B7FE
5AAC				B8A1
5AC4				B8A2
E667				B8A3
5ABA				B8A4
5AC6				B8A5
E668				B8A6
E669				B8A7
5ACC	3779	8C99	02389	B8A8
E66A				B8A9
E66B				B8AA
E66C				B8AB
5B08				B8AC
5AF5				B8AD
5B34				B8AE
5B1B				B8AF
5B19				B8B0
E66D				B8B1
5B4B				B8B2
5B4C				B8B3
5B56		FAA6	11608	B8B4
E66E				B8B5
E66F				B8B6
E670				B8B7
E671				B8B8
5B7D				B8B9
E672				B8BA
5B7E				B8BB
5B7F				B8BC
5B81				B8BD
E673				B8BE
E674				B8BF
5B93				B8C0
5BAF				B8C1
5BBA				B8C2
E675				B8C3
5BB7				B8C4
E676				B8C5
E677				B8C6
5BAD				B8C7
E678				B8C8
E679				B8C9
5BC0		FAA7	11609	B8CA
5BC3	5563	9B83	05367	B8CB
752F		FAA8	11610	B8CC
5BD8		FAA9	11611	B8CD
E67A				B8CE
5BD7				B8CF
5BDA				B8D0
E67B				B8D1
5BE0				B8D2
E67C				B8D3
E67D				B8D4
E67E				B8D5
E67F				B8D6
E680				B8D7
5BF4				B8D8
E681				B8D9
E682				B8DA
E683				B8DB
E684				B8DC
E685				B8DD
E686				B8DE
5C0C				B8DF
E687				B8E0
E688				B8E1
5C1F				B8E2
5C1E		FAAB	11613	B8E3
E689				B8E4
5C29				B8E5
E68A				B8E6
5C4F	5622	9BA0	05402	B8E7
5C61	3C48	8EC6	02840	B8E8
E68B				B8E9
E68C				B8EA
5C70				B8EB
5C7A				B8EC
E68D				B8ED
E68E				B8EE
E68F				B8EF
E690				B8F0
5C87				B8F1
5C8F				B8F2
E691				B8F3
E692				B8F4
5CB5				B8F5
E693				B8F6
5CAA				B8F7
E694				B8F8
5CBA		FAAD	11615	B8F9
E695				B8FA
5CA6		FAAC	11614	B8FB
E696				B8FC
E697				B8FD
5C9F				B8FE
E698				B9A1
E699				B9A2
E69A				B9A3
5CD2				B9A4
5CC9				B9A5
E69B				B9A6
E69C				B9A7
9657				B9A8
EB72				B9A9
E69D				B9AA
5CF4				B9AB
5CEE				B9AC
E69E				B9AD
E69F				B9AE
E6A0				B9AF
5D12				B9B0
E6A1				B9B1
E6A2				B9B2
E6A3				B9B3
5D06				B9B4
5D2B				B9B5
E6A4				B9B6
E6A5				B9B7
E6A6				B9B8
E6A7				B9B9
E6A8				B9BA
E6A9				B9BB
5D34				B9BC
5D42		FAB2	11620	B9BD
5D43				B9BE
E6AA				B9BF
5D6D		FAB3	11621	B9C0
5D61				B9C1
E6AB				B9C2
E6AC				B9C3
5D83				B9C4
E6AD				B9C5
5D88				B9C6
5D8A				B9C7
E6AE				B9C8
E6AF				B9C9
5D7F				B9CA
E6B0				B9CB
E6B1				B9CC
5DA0				B9CD
5D94				B9CE
5D9F				B9CF
5D92				B9D0
E6B2				B9D1
5DA7				B9D2
5DB8		FAB4	11622	B9D3
8C73				B9D4
5DB9		FAB5	11623	B9D5
E6B3				B9D6
E6B4				B9D7
5DC3				B9D8
E6B5				B9D9
5DD0		FAB6	11624	B9DA
E6B6				B9DB
E6B7				B9DC
5DE0				B9DD
E6B8				B9DE
5DF9				B9DF
E6B9				B9E0
E6BA				B9E1
5DFD	4327	9246	03507	B9E2
5E00				B9E3
E6BB				B9E4
E6BC				B9E5
E6BD				B9E6
5E15				B9E7
5E18				B9E8
5E12				B9E9
E6BE				B9EA
5E2E				B9EB
E6BF				B9EC
E6C0				B9ED
E6C1				B9EE
E6C2				B9EF
5E58				B9F0
5E5E				B9F1
E6C3				B9F2
8946				B9F3
E6C4				B9F4
98BF				B9F5
E6C5				B9F6
E6C6				B9F7
79CA				B9F8
E6C7				B9F9
69A6				B9FA
E6C8				B9FB
E6C9				B9FC
E6CA				B9FD
E6CB				B9FE
E6CC				BAA1
E6CD				BAA2
5EA8				BAA3
E6CE				BAA4
E6CF				BAA5
E6D0				BAA6
5EBE				BAA7
E6D1				BAA8
5ECA	4F2D	984C	04713	BAA9
5ECB				BAAA
5ED5				BAAB
E6D2				BAAC
E6D3				BAAD
E6D4				BAAE
5EDE				BAAF
E6D5				BAB0
9F90				BAB1
E6D6				BAB2
E6D7				BAB3
E6D8				BAB4
E6D9				BAB5
E6DA				BAB6
5F06				BAB7
630A				BAB8
5F08				BAB9
5F07				BABA
6335				BABB
63DC				BABC
E6DB				BABD
5F1C				BABE
5F21		FAB7	11625	BABF
5F30				BAC0
5F34		FAB8	11626	BAC1
5F36				BAC2
E6DC				BAC3
5F3B				BAC4
5F3D				BAC5
5F40				BAC6
E6DD				BAC7
5F4D				BAC8
5F50				BAC9
E6DE				BACA
E6DF				BACB
E6E0				BACC
5F64				BACD
E6E1				BACE
E6E2				BACF
E6E3				BAD0
5F67		FAB9	11627	BAD1
E6E4				BAD2
E6E5				BAD3
E6E6				BAD4
5F75				BAD5
E6E7				BAD6
5F78				BAD7
E6E8				BAD8
5F7D				BAD9
4F9A		FA71	11550	BADA
5F8D				BADB
5F89				BADC
E6E9				BADD
5F9C				BADE
E6EA				BADF
E6EB				BAE0
E6EC				BAE1
E6ED				BAE2
E6EE				BAE3
E6EF				BAE4
E6F0				BAE5
E6F1				BAE6
E6F2				BAE7
5FAF				BAE8
5FAC				BAE9
5FB1				BAEA
E6F3				BAEB
EB73				BAEC
E6F4				BAED
5FC4				BAEE
E6F5				BAEF
5FC9				BAF0
5FD4				BAF1
E6F6				BAF2
5FD2				BAF3
5FEE				BAF4
5FE1				BAF5
E6F7				BAF6
5FEF				BAF7
5FE2				BAF8
E6F8				BAF9
5FF6				BAFA
E6F9				BAFB
E6FA				BAFC
5FEC				BAFD
E6FB				BAFE
5FE9				BBA1
6033				BBA2
600D				BBA3
6017				BBA4
5379				BBA5
6014				BBA6
E6FC				BBA7
E6FD				BBA8
E6FE				BBA9
6047				BBAA
E6FF				BBAB
6071				BBAC
6061				BBAD
E700				BBAE
6091				BBAF
E701				BBB0
EB74				BBB1
609D				BBB2
609E				BBB3
60A2				BBB4
E702				BBB5
E703				BBB6
6086				BBB7
E704				BBB8
E705				BBB9
E706				BBBA
E707				BBBB
60BB				BBBC
60D5		FAC0	11634	BBBD
60B1				BBBE
E708				BBBF
E709				BBC0
E70A				BBC1
60B0				BBC2
E70B				BBC3
60DE		FABF	11633	BBC4
E70C				BBC5
E70D				BBC6
E70E				BBC7
E70F				BBC8
EBBF				BBC9
611C				BBCA
6112				BBCB
6113				BBCC
611E				BBCD
60F2		FAC2	11636	BBCE
6114				BBCF
E710				BBD0
6102				BBD1
E711				BBD2
E712				BBD3
E713				BBD4
E714				BBD5
6137		FAC4	11638	BBD6
E715				BBD7
6130		FAC5	11639	BBD8
E716				BBD9
612B				BBDA
E717				BBDB
E718				BBDC
E719				BBDD
E71A				BBDE
617C				BBDF
617F				BBE0
6167	3745	8C64	02337	BBE1
E71B				BBE2
E71C				BBE3
E71D				BBE4
E71E				BBE5
61A8				BBE6
6197				BBE7
61AD				BBE8
6193				BBE9
618D				BBEA
E71F				BBEB
61C1				BBEC
E720				BBED
61BC				BBEE
61E2				BBEF
61D5				BBF0
61DF				BBF1
E721				BBF2
E722				BBF3
E723				BBF4
E724				BBF5
6207				BBF6
E725				BBF7
E726				BBF8
6213		FAC7	11641	BBF9
E727				BBFA
6222				BBFB
6220				BBFC
6229				BBFD
E728				BBFE
E729				BCA1
E72A				BCA2
6243				BCA3
6249	4862	94E0	04066	BCA4
624C				BCA5
6251				BCA6
6250				BCA7
6256				BCA8
E72B				BCA9
625A				BCAA
E72C				BCAB
62D5				BCAC
62D6				BCAD
62C4				BCAE
E72D				BCAF
E72E				BCB0
62A6		FAC8	11642	BCB1
62DA				BCB2
62D0	327D	89FB	01893	BCB3
6318				BCB4
6313				BCB5
E72F				BCB6
E730				BCB7
62FC				BCB8
6344				BCB9
6343				BCBA
6342				BCBB
6346				BCBC
634E				BCBD
6339				BCBE
634B				BCBF
E731				BCC0
E732				BCC1
E733				BCC2
6387				BCC3
6384				BCC4
6374				BCC5
637F				BCC6
63A6				BCC7
6365				BCC8
6BEE				BCC9
6366				BCCA
E734				BCCB
E735				BCCC
63C1				BCCD
63E5				BCCE
E736				BCCF
63D1				BCD0
E737				BCD1
E738				BCD2
6409				BCD3
E739				BCD4
6425				BCD5
642A				BCD6
E73A				BCD7
6473				BCD8
E73B				BCD9
6454				BCDA
649D		FACB	11645	BCDB
64CE		FACC	11646	BCDC
E73C				BCDD
64CA				BCDE
64D0				BCDF
64BE				BCE0
64D7				BCE1
64BF				BCE2
E73D				BCE3
E73E				BCE4
E73F				BCE5
E740				BCE6
E741				BCE7
6504				BCE8
6505	5A39	9DB7	05825	BCE9
6514				BCEA
6529				BCEB
6531				BCEC
E742				BCED
6547				BCEE
E743				BCEF
EB75				BCF0
E744				BCF1
E745				BCF2
655F				BCF3
6567				BCF4
E746				BCF5
E747				BCF6
E748				BCF7
E749				BCF8
6581				BCF9
6583	5A4D	9DCB	05845	BCFA
E74A				BCFB
E74B				BCFC
E74C				BCFD
E74D				BCFE
E74E				BDA1
E74F				BDA2
E750				BDA3
E751				BDA4
E752				BDA5
E753				BDA6
6595				BDA7
E754				BDA8
E755				BDA9
65BF				BDAA
E756				BDAB
65C2				BDAC
65C8				BDAD
E757				BDAE
E758				BDAF
E759				BDB0
65F0				BDB1
E75A				BDB2
65F2				BDB3
65F5				BDB4
6615		FACF	11649	BDB5
E75B				BDB6
6609		FAD1	11651	BDB7
65F9				BDB8
6604				BDB9
65FF				BDBA
6608				BDBB
6611				BDBC
6600		FACE	11648	BDBD
E75C				BDBE
6631		FA63	11536	BDBF
70A4				BDC0
6639				BDC1
662E		FAD2	11652	BDC2
6630				BDC3
661E		FAD3	11653	BDC4
E75D				BDC5
6621				BDC6
6624		FAD4	11654	BDC7
6622				BDC8
662B				BDC9
664C				BDCA
E75E				BDCB
E75F				BDCC
6645				BDCD
6651				BDCE
664A				BDCF
E760				BDD0
E761				BDD1
E762				BDD2
E763				BDD3
E764				BDD4
66FB		FA66	11539	BDD5
EB76				BDD6
6660				BDD7
E765				BDD8
6661				BDD9
EBB6				BDDA
E766				BDDB
E767				BDDC
665B				BDDD
6659		FAD7	11657	BDDE
E768				BDDF
6657		FAD6	11656	BDE0
E769				BDE1
E76A				BDE2
6677				BDE3
E76B				BDE4
E76C				BDE5
6673		FAD9	11659	BDE6
E76D				BDE7
667F				BDE8
666A				BDE9
666B				BDEA
E76E				BDEB
E76F				BDEC
E770				BDED
E771				BDEE
6678				BDEF
E772				BDF0
6692				BDF1
668D				BDF2
714A				BDF3
E773				BDF4
668B				BDF5
E774				BDF6
6699		FADA	11660	BDF7
669C				BDF8
E775				BDF9
669B				BDFA
669F				BDFB
E776				BDFC
669A				BDFD
66B1				BDFE
E777				BEA1
E778				BEA2
66B5				BEA3
E779				BEA4
66B2		FADC	11662	BEA5
E77A				BEA6
E77B				BEA7
66C0				BEA8
E77C				BEA9
66C8				BEAA
66BB				BEAB
66BF		FADD	11663	BEAC
E77D				BEAD
E77E				BEAE
E77F				BEAF
E780				BEB0
E781				BEB1
E782				BEB2
66E8				BEB3
E783				BEB4
E784				BEB5
66EB				BEB6
62FD				BEB7
E785				BEB8
6705				BEB9
E786				BEBA
E787				BEBB
670E		FADF	11665	BEBC
6713				BEBD
E788				BEBE
E789				BEBF
6733				BEC0
E78A				BEC1
6747				BEC2
674C				BEC3
E78B				BEC4
E78C				BEC5
6755				BEC6
675D				BEC7
6766		FAE1	11667	BEC8
E78D				BEC9
E78E				BECA
E78F				BECB
6793				BECC
677B				BECD
E790				BECE
6798				BECF
67A6	5B45	9E64	05937	BED0
678E				BED1
676C				BED2
6796				BED3
679B				BED4
E791				BED5
E792				BED6
6776				BED7
67D7				BED8
67BB		FAE2	11668	BED9
E793				BEDA
67D9				BEDB
E794				BEDC
67F9				BEDD
67F6				BEDE
67C5				BEDF
67F7				BEE0
67C2				BEE1
EB77				BEE2
E795				BEE3
67F5	3A74	8DF2	02684	BEE4
6852		FAE3	11669	BEE5
E796				BEE6
67BC				BEE7
67C0		FAE4	11670	BEE8
67B4	5B4A	9E69	05942	BEE9
681D				BEEA
81EC				BEEB
E797				BEEC
683B				BEED
E798				BEEE
E799				BEEF
6844		FAE6	11672	BEF0
6831				BEF1
6858				BEF2
E79A				BEF3
E79B				BEF4
E79C				BEF5
67BD				BEF6
E79D				BEF7
E79E				BEF8
68B2				BEF9
E79F				BEFA
6896				BEFB
6886				BEFC
68A9				BEFD
68A2	3E3F	8FBD	03031	BEFE
E7A0				BFA1
6879				BFA2
68AA				BFA3
6870				BFA4
689A				BFA5
E7A1				BFA6
E7A2				BFA7
E7A3				BFA8
68E8				BFA9
68EC				BFAA
68F0				BFAB
68D6				BFAC
68D0				BFAD
68FC				BFAE
68EB				BFAF
68F1				BFB0
68D9				BFB1
E7A4				BFB2
68ED				BFB3
E7A5				BFB4
68FD				BFB5
6911				BFB6
68C8		FA64	11537	BFB7
E7A6				BFB8
E7A7				BFB9
E7A8				BFBA
690A				BFBB
E7A9				BFBC
68DA	432A	9249	03510	BFBD
E7AA				BFBE
6942				BFBF
6968		FAE9	11675	BFC0
6963				BFC1
698A	3A67	8DE5	02671	BFC2
6935				BFC3
6980				BFC4
693B				BFC5
E7AB				BFC6
E7AC				BFC7
E7AD				BFC8
E7AE				BFC9
69B7				BFCA
EB78				BFCB
6996				BFCC
69D7				BFCD
69D9	4B6A	968A	04374	BFCE
E7AF				BFCF
6A0B	4875	94F3	04085	BFD0
E7B0				BFD1
69D6				BFD2
E7B1				BFD3
E7B2				BFD4
E7B3				BFD5
E7B4				BFD6
E7B5				BFD7
E7B6				BFD8
E7B7				BFD9
E7B8				BFDA
69F3				BFDB
E7B9				BFDC
E7BA				BFDD
E7BB				BFDE
6A11				BFDF
6A0F				BFE0
E7BC				BFE1
E7BD				BFE2
E7BE				BFE3
69E2		FAEC	11678	BFE4
6A30		FAED	11679	BFE5
E7BF				BFE6
6A5B				BFE7
6A32				BFE8
6A64				BFE9
6A50				BFEA
6A3D	432E	924D	03514	BFEB
6A46		FAEF	11681	BFEC
E7C0				BFED
E7C1				BFEE
6A45				BFEF
6A73		FAF0	11682	BFF0
E7C2				BFF1
6A9F				BFF2
E7C3				BFF3
6A89				BFF4
E7C4				BFF5
6ADB	367B	8BF9	02291	BFF6
E7C5				BFF7
6A7E		FAF1	11683	BFF8
E7C6				BFF9
6A91				BFFA
7F47		FB90	11780	BFFB
E7C7				BFFC
E7C8				BFFD
E7C9				BFFE
E7CA				C0A1
E7CB				C0A2
E7CC				C0A3
6AD5				C0A4
6AE4		FAF3	11685	C0A5
6AD6				C0A6
6AD0				C0A7
EB79				C0A8
6AF1				C0A9
6AE7				C0AA
6AEC				C0AB
E7CD				C0AC
E7CE				C0AD
E7CF				C0AE
E7D0				C0AF
6B06				C0B0
6B11				C0B1
6B10				C0B2
6B1B				C0B3
E7D1				C0B4
6B2F				C0B5
6B35				C0B6
E7D2				C0B7
E7D3				C0B8
E7D4				C0B9
6B46				C0BA
6B4A				C0BB
E7D5				C0BC
E7D6				C0BD
6B58				C0BE
6B56				C0BF
E7D7				C0C0
EB7A				C0C1
E7D8				C0C2
E7D9				C0C3
EB7B				C0C4
EB7C				C0C5
6B82				C0C6
6B85				C0C7
E7DA				C0C8
6BAD				C0C9
E7DB				C0CA
E7DC				C0CB
E7DD				C0CC
6BCF				C0CD
E7DE				C0CE
6BD6		FAF4	11686	C0CF
6BE1				C0D0
E7DF				C0D1
6BE7				C0D2
E7E0				C0D3
6C10				C0D4
6C33				C0D5
6C35				C0D6
E7E1				C0D7
E7E2				C0D8
6C3F		FAF5	11687	C0D9
E7E3				C0DA
E7E4				C0DB
6C59				C0DC
6C4D				C0DD
6C5C		FAF6	11688	C0DE
6C4F				C0DF
E7E5				C0E0
E7E6				C0E1
6C87				C0E2
EB7D				C0E3
6C85				C0E4
6C86		FAF7	11689	C0E5
E7E7				C0E6
6C89				C0E7
E7E8				C0E8
E7E9				C0E9
6C94				C0EA
6C6F		FAF8	11690	C0EB
6C78				C0EC
6C9C				C0ED
E7EA				C0EE
6C7B				C0EF
6C97				C0F0
6CC6				C0F1
6CD4				C0F2
6CEB				C0F3
6CDA		FAF9	11691	C0F4
6CB2				C0F5
E7EB				C0F6
6CEE				C0F7
6CE0				C0F8
EB7E				C0F9
6CE9				C0FA
6CD6				C0FB
6CE7				C0FC
6CD2				C0FD
E7EC				C0FE
6CD1				C1A1
6CEC				C1A2
E7ED				C1A3
E7EE				C1A4
6CE1	4B22	9641	04302	C1A5
6CF4				C1A6
6D04		FAFA	11692	C1A7
6D0E				C1A8
6D0A				C1A9
6D2E				C1AA
6D3C				C1AB
E7EF				C1AC
6D39				C1AD
6D27				C1AE
E7F0				C1AF
E7F1				C1B0
E7F2				C1B1
6D00				C1B2
E7F3				C1B3
E7F4				C1B4
E7F5				C1B5
6D87		FAFB	11693	C1B6
6D94				C1B7
6D97				C1B8
EB7F				C1B9
6D82				C1BA
6D92				C1BB
6D61				C1BC
6D65				C1BD
6D5F				C1BE
6D96		FB40	11701	C1BF
EB80				C1C0
E7F6				C1C1
E7F7				C1C2
EB81				C1C3
6D70				C1C4
E7F8				C1C5
E7F9				C1C6
6D6F		FAFC	11694	C1C7
57FF				C1C8
EB82				C1C9
6DAC		FB41	11702	C1CA
6DC8				C1CB
6DC4				C1CC
EB83				C1CD
6DBF				C1CE
6DDF				C1CF
6DD6				C1D0
6DFC		FB45	11706	C1D1
EB84				C1D2
6DE9				C1D3
6DE5				C1D4
E7FA				C1D5
E7FB				C1D6
5C5A				C1D7
6E00				C1D8
E7FC				C1D9
E7FD				C1DA
6DCE				C1DB
E7FE				C1DC
E7FF				C1DD
6DB7				C1DE
EB85				C1DF
E800				C1E0
6E36				C1E1
6DCA				C1E2
6E1A	3D6D	8F8D	02977	C1E3
EB86				C1E4
E801				C1E5
6E63				C1E6
6E23	5E56	9FD4	06254	C1E7
6E51				C1E8
6E44				C1E9
6E53				C1EA
E802				C1EB
E803				C1EC
6E8C	482E	94AC	04014	C1ED
E804				C1EE
6E57				C1EF
6E3C		FB49	11710	C1F0
6E5E				C1F1
E805				C1F2
6E5C		FB47	11708	C1F3
E806				C1F4
E807				C1F5
6E73				C1F6
E808				C1F7
E809				C1F8
E80A				C1F9
E80B				C1FA
6E39		FB46	11707	C1FB
E80C				C1FC
EB87				C1FD
6EC3				C1FE
6EB1				C2A1
6EC7				C2A2
E80D				C2A3
6EA2	306E	88EC	01678	C2A4
E80E				C2A5
E80F				C2A6
E810				C2A7
6EBF		FB4A	11711	C2A8
6E99				C2A9
E811				C2AA
E812				C2AB
E813				C2AC
E814				C2AD
E815				C2AE
6E9D	3942	8D61	02534	C2AF
9816				C2B0
E816				C2B1
E817				C2B2
6F2A				C2B3
6F1A				C2B4
E818				C2B5
6EF9				C2B6
6F3C				C2B7
E819				C2B8
6F3B				C2B9
E81A				C2BA
6F33				C2BB
E81B				C2BC
E81C				C2BD
E81D				C2BE
E81E				C2BF
E81F				C2C0
E820				C2C1
E821				C2C2
6F90				C2C3
6F94				C2C4
E822				C2C5
6F5D				C2C6
6F62				C2C7
6F8C				C2C8
6F8D				C2C9
E823				C2CA
E824				C2CB
6F97	3442	8AC0	02034	C2CC
E825				C2CD
6F88		FB4B	11712	C2CE
6F7E				C2CF
6F96				C2D0
6F59				C2D1
E826				C2D2
E827				C2D3
6F7D				C2D4
E828				C2D5
E829				C2D6
EB88				C2D7
6F8B				C2D8
6F9A				C2D9
6FAE				C2DA
6FA7				C2DB
E82A				C2DC
E82B				C2DD
E82C				C2DE
6FB5		FB4C	11713	C2DF
6FAF				C2E0
6F5E				C2E1
E82D				C2E2
E82E				C2E3
6FF5		FB4D	11714	C2E4
6FF0				C2E5
701E	4654	93D2	03852	C2E6
E82F				C2E7
6FDA				C2E8
E830				C2E9
6FF9				C2EA
E831				C2EB
6FEF	4275	91F3	03485	C2EC
E832				C2ED
7007		FB4F	11716	C2EE
7005		FB4E	11715	C2EF
6FFD				C2F0
EB89				C2F1
702F				C2F2
E833				C2F3
703C				C2F4
705D				C2F5
7064				C2F6
706C				C2F7
706E				C2F8
7098				C2F9
7095				C2FA
7085		FB51	11718	C2FB
70AB		FB52	11719	C2FC
70B7				C2FD
E834				C2FE
70DC				C3A1
70D8				C3A2
70D3				C3A3
70D4				C3A4
70E4				C3A5
7104		FB54	11721	C3A6
E835				C3A7
710F		FB53	11720	C3A8
E836				C3A9
E837				C3AA
E838				C3AB
7107				C3AC
E839				C3AD
7131				C3AE
711E				C3AF
E83A				C3B0
E83B				C3B1
712F				C3B2
E83C				C3B3
E83D				C3B4
E83E				C3B5
E83F				C3B6
7152				C3B7
715C		FB55	11722	C3B8
7146		FB56	11723	C3B9
7199	7426	EAA4	08406	C3BA
E840				C3BB
7147		FB57	11724	C3BC
7141				C3BD
E841				C3BE
E842				C3BF
714B				C3C0
7160				C3C1
E843				C3C2
E844				C3C3
E845				C3C4
EB8A				C3C5
7185				C3C6
7192				C3C7
7180				C3C8
7179				C3C9
71A0				C3CA
E846				C3CB
E847				C3CC
E848				C3CD
E849				C3CE
E84A				C3CF
E84B				C3D0
71C4				C3D1
E84C				C3D2
71D7	5F73	E093	06383	C3D3
71BF				C3D4
71FE		FB5A	11727	C3D5
7200				C3D6
721D				C3D7
E84D				C3D8
E84E				C3D9
E84F				C3DA
E850				C3DB
E851				C3DC
E852				C3DD
7243				C3DE
E853				C3DF
7256				C3E0
E854				C3E1
725C				C3E2
7263				C3E3
7278				C3E4
727F				C3E5
729B				C3E6
72AD				C3E7
72AE				C3E8
EB8B				C3E9
E855				C3EA
72B1		FB5B	11728	C3EB
EB8C				C3EC
72C7				C3ED
72BE		FB5C	11729	C3EE
72CC				C3EF
72FB				C3F0
E856				C3F1
E857				C3F2
7319				C3F3
730B				C3F4
7305				C3F5
E858				C3F6
7304				C3F7
E859				C3F8
7331				C3F9
7328				C3FA
7324		FB5D	11730	C3FB
733D				C3FC
E85A				C3FD
735D				C3FE
7360				C4A1
E85B				C4A2
7367				C4A3
736B				C4A4
7377		FB5F	11732	C4A5
737C				C4A6
7381				C4A7
7385				C4A8
7386				C4A9
E85C				C4AA
EB8D				C4AB
738E				C4AC
E85D				C4AD
7395				C4AE
E85E				C4AF
7398				C4B0
E85F				C4B1
739E				C4B2
739C				C4B3
EB8E				C4B4
E860				C4B5
E861				C4B6
73CA	3B39	8E58	02725	C4B7
73C9		FB61	11734	C4B8
73CB				C4B9
73BD		FB60	11733	C4BA
73B9				C4BB
73CF				C4BC
73C5				C4BD
73C6				C4BE
73D9				C4BF
73E9				C4C0
73D6		FB62	11735	C4C1
73E3		FB63	11736	C4C2
E862				C4C3
73F9				C4C4
E863				C4C5
73D2		FB64	11737	C4C6
7407		FB65	11738	C4C7
E864				C4C8
E865				C4C9
73FA				C4CA
E866				C4CB
7404				C4CC
7401				C4CD
EB8F				C4CE
E867				C4CF
E868				C4D0
742F				C4D1
7426		FB67	11740	C4D2
742A		FB68	11741	C4D3
7439				C4D4
7428				C4D5
7424				C4D6
742E		FB6A	11743	C4D7
E869				C4D8
741B				C4D9
73F7				C4DA
7429		FB69	11742	C4DB
742B				C4DC
741A				C4DD
742D				C4DE
E86A				C4DF
E86B				C4E0
E86C				C4E1
E86D				C4E2
7457				C4E3
E86E				C4E4
744B				C4E5
7446				C4E6
7452				C4E7
7476	6076	E0F4	06486	C4E8
E86F				C4E9
E870				C4EA
7471				C4EB
7467				C4EC
E871				C4ED
E872				C4EE
7466				C4EF
7486				C4F0
7480				C4F1
7487				C4F2
7489		FB6C	11745	C4F3
E873				C4F4
E874				C4F5
E875				C4F6
E876				C4F7
E877				C4F8
749C				C4F9
7499				C4FA
749F		FB6D	11746	C4FB
74A8				C4FC
74AB				C4FD
E878				C4FE
74A6				C5A1
7490				C5A2
E879				C5A3
74BF				C5A4
74B9				C5A5
74C9				C5A6
E87A				C5A7
74C8				C5A8
74DA				C5A9
E87B				C5AA
74F4				C5AB
EB90				C5AC
7511	3979	8D99	02589	C5AD
7521				C5AE
7540				C5AF
7550				C5B0
E87C				C5B1
E87D				C5B2
7571				C5B3
756F		FB6F	11748	C5B4
EB91				C5B5
E87E				C5B6
7579				C5B7
757E				C5B8
7581				C5B9
E87F				C5BA
E880				C5BB
7590				C5BC
7592				C5BD
7593				C5BE
75C0				C5BF
E881				C5C0
75B4				C5C1
75E4				C5C2
E882				C5C3
E883				C5C4
75F9				C5C5
7600				C5C6
7613				C5C7
7616				C5C8
7625				C5C9
7619				C5CA
7633				C5CB
762D				C5CC
7652	4C7E	96FC	04494	C5CD
E884				C5CE
7665				C5CF
766D				C5D0
7671				C5D1
E885				C5D2
E886				C5D3
E887				C5D4
E888				C5D5
E889				C5D6
7693	622B	E1A9	06611	C5D7
66A0		FADB	11661	C5D8
769C		FB71	11750	C5D9
EB92				C5DA
769B		FB73	11752	C5DB
EBBD				C5DC
76A5				C5DD
76A4				C5DE
76A6		FB74	11753	C5DF
76A7				C5E0
76AA				C5E1
76AD				C5E2
E88A				C5E3
76C1				C5E4
E88B				C5E5
76CE				C5E6
76CC				C5E7
E88C				C5E8
7C0B				C5E9
7C20				C5EA
76F1				C5EB
7722				C5EC
7A85				C5ED
7719				C5EE
E88D				C5EF
7736				C5F0
E88E				C5F1
7746		FB76	11755	C5F2
774D				C5F3
E88F				C5F4
776A				C5F5
775F				C5F6
775C				C5F7
E890				C5F8
E891				C5F9
96CE	7033	E8B1	08019	C5FA
7772				C5FB
E892				C5FC
77A2				C5FD
779A				C5FE
7795				C6A1
77AA				C6A2
77AF				C6A3
EB93				C6A4
77DE				C6A5
E893				C6A6
77E4				C6A7
E894				C6A8
E895				C6A9
6998		FAEB	11677	C6AA
77F4				C6AB
77FB				C6AC
E896				C6AD
7806				C6AE
782D				C6AF
7822				C6B0
7821		FB78	11757	C6B1
7823				C6B2
E897				C6B3
7843				C6B4
784E		FB79	11758	C6B5
7848				C6B6
EB94				C6B7
786E				C6B8
785C				C6B9
7868				C6BA
7860				C6BB
7863				C6BC
7864		FB7A	11759	C6BD
E898				C6BE
7891	486A	94E8	04074	C6BF
E899				C6C0
E89A				C6C1
788F				C6C2
E89B				C6C3
787A		FB7B	11760	C6C4
78A4				C6C5
789E				C6C6
78AD				C6C7
E89C				C6C8
78C8				C6C9
E89D				C6CA
E89E				C6CB
78E1				C6CC
E89F				C6CD
78DB				C6CE
78E0				C6CF
78FB				C6D0
78F7				C6D1
78FF				C6D2
7900				C6D3
E8A0				C6D4
78F6				C6D5
790C				C6D6
7929				C6D7
E8A1				C6D8
7931				C6D9
EB95				C6DA
7934				C6DB
793B				C6DC
E8A2				C6DD
E8A3				C6DE
E8A4				C6DF
E8A5				C6E0
E8A6				C6E1
E8A7				C6E2
E8A8				C6E3
E8A9				C6E4
7947	3540	8B5F	02132	C6E5
E8AA				C6E6
795B				C6E7
795C				C6E8
E8AB				C6E9
E8AC				C6EA
E8AD				C6EB
E8AE				C6EC
E8AF				C6ED
E8B0				C6EE
E8B1				C6EF
E8B2				C6F0
E8B3				C6F1
E8B4				C6F2
E8B5				C6F3
E8B6				C6F4
E8B7				C6F5
796B				C6F6
E8B8				C6F7
7967				C6F8
E8B9				C6F9
7972				C6FA
7977	4578	9398	03788	C6FB
E8BA				C6FC
E8BB				C6FD
E8BC				C6FE
E8BD				C7A1
798B				C7A2
7994		FB81	11765	C7A3
E8BE				C7A4
7996				C7A5
E8BF				C7A6
E8C0				C7A7
E8C1				C7A8
E8C2				C7A9
799B		FB83	11767	C7AA
E8C3				C7AB
79A8				C7AC
E8C4				C7AD
E8C5				C7AE
E8C6				C7AF
E8C7				C7B0
79B0	4729	9448	03909	C7B1
79B8				C7B2
79C2				C7B3
E8C8				C7B4
E8C9				C7B5
79C8				C7B6
E8CA				C7B7
E8CB				C7B8
EB96				C7B9
79D6				C7BA
79EB				C7BB
79DA				C7BC
E8CC				C7BD
E8CD				C7BE
79EA				C7BF
79F1				C7C0
79F8				C7C1
E8CE				C7C2
E8CF				C7C3
7A0A				C7C4
7A0C				C7C5
7A02				C7C6
7A03				C7C7
E8D0				C7C8
7A1B				C7C9
7A11				C7CA
7A1E				C7CB
E8D1				C7CC
E8D2				C7CD
E8D3				C7CE
E8D4				C7CF
7A3D	374E	8C6D	02346	C7D0
7A39				C7D1
7A38				C7D2
7CD3				C7D3
7A45				C7D4
E8D5				C7D5
7A4C				C7D6
7A5D				C7D7
E8D6				C7D8
E8D7				C7D9
E8D8				C7DA
7A5F				C7DB
E8D9				C7DC
E8DA				C7DD
7A6A				C7DE
E8DB				C7DF
7076				C7E0
E8DC				C7E1
E8DD				C7E2
E8DE				C7E3
E8DF				C7E4
7AA0				C7E5
E8E0				C7E6
E8E1				C7E7
7AAC				C7E8
E8E2				C7E9
E8E3				C7EA
E8E4				C7EB
E8E5				C7EC
E8E6				C7ED
7ABC				C7EE
E8E7				C7EF
E8E8				C7F0
7ACE				C7F1
7AD1		FB84	11768	C7F2
E8E9				C7F3
E8EA				C7F4
EB97				C7F5
E8EB				C7F6
E8EC				C7F7
7AEB		FB87	11771	C7F8
E8ED				C7F9
E8EE				C7FA
E8EF				C7FB
E8F0				C7FC
E8F1				C7FD
7AFD				C7FE
E8F2				C8A1
7B14				C8A2
7B34				C8A3
7B31				C8A4
7B2E				C8A5
7B2D				C8A6
EB98				C8A7
E8F3				C8A8
7B3D				C8A9
7B47				C8AA
E8F4				C8AB
E8F5				C8AC
E8F6				C8AD
7B60				C8AE
7B69				C8AF
7B6D				C8B0
E8F7				C8B1
EB99				C8B2
E8F8				C8B3
7B64				C8B4
7B66				C8B5
7B6A				C8B6
7B6F				C8B7
E8F9				C8B8
7B91				C8B9
E8FA				C8BA
7B9E		FB88	11772	C8BB
7B9B				C8BC
E8FB				C8BD
E8FC				C8BE
E8FD				C8BF
7BB2				C8C0
E8FE				C8C1
E8FF				C8C2
E900				C8C3
E901				C8C4
7BBA				C8C5
7BBC				C8C6
E902				C8C7
7BD9				C8C8
7BDA				C8C9
E903				C8CA
7BD7				C8CB
E904				C8CC
7BD4				C8CD
E905				C8CE
7C0F				C8CF
7C09				C8D0
EB9A				C8D1
7BFE				C8D2
E906				C8D3
7C31				C8D4
E907				C8D5
E908				C8D6
E909				C8D7
7C5C				C8D8
7C59				C8D9
E90A				C8DA
7C63				C8DB
7C6D				C8DC
7C6E				C8DD
79D4				C8DE
7C79				C8DF
7C94				C8E0
E90B				C8E1
E90C				C8E2
E90D				C8E3
E90E				C8E4
7CA0				C8E5
E90F				C8E6
E910				C8E7
E911				C8E8
7CB7				C8E9
7CBA				C8EA
E912				C8EB
7CC8				C8EC
7CC9				C8ED
E913				C8EE
E914				C8EF
7CDD				C8F0
7CD9				C8F1
E915				C8F2
E916				C8F3
E917				C8F4
E918				C8F5
7D08				C8F6
7D03				C8F7
7D09				C8F8
7D1D				C8F9
E919				C8FA
7D13				C8FB
7D23				C8FC
7D16				C8FD
7D0F				C8FE
E91A				C9A1
E91B				C9A2
7D47				C9A3
7D41				C9A4
7D3E				C9A5
7D31				C9A6
7D48		FB8A	11774	C9A7
7D2D				C9A8
7D4D				C9A9
7D6A				C9AA
7D53				C9AB
E91C				C9AC
7D67				C9AD
7D8C				C9AE
E91D				C9AF
7D88				C9B0
7D8B				C9B1
7DA6				C9B2
7DC4				C9B3
7DC5				C9B4
7DC2				C9B5
EB9B				C9B6
E91E				C9B7
7DB3				C9B8
7DB7		FB8C	11776	C9B9
E91F				C9BA
E920				C9BB
E921				C9BC
EBC0				C9BD
E922				C9BE
E923				C9BF
7DD7				C9C0
E924				C9C1
7DF1				C9C2
7E15				C9C3
7E08				C9C4
7E11				C9C5
7E20				C9C6
E925				C9C7
7E1D				C9C8
E926				C9C9
7E1C				C9CA
E927				C9CB
E928				C9CC
7E27				C9CD
E929				C9CE
E92A				C9CF
7E36				C9D0
7E47				C9D1
E92B				C9D2
7E4D	3D2B	8F4A	02911	C9D3
7E62				C9D4
7E52		FB8F	11779	C9D5
E92C				C9D6
7E7E				C9D7
7E8A		FA5C	11529	C9D8
7E86				C9D9
E92D				C9DA
7E91				C9DB
530B				C9DC
74E8				C9DD
7F43				C9DE
7F44				C9DF
E92E				C9E0
E92F				C9E1
E930				C9E2
7F52				C9E3
E931				C9E4
E932				C9E5
7F5D				C9E6
7F61				C9E7
7F63				C9E8
E933				C9E9
EBB9				C9EA
8989				C9EB
E934				C9EC
7F91				C9ED
E935				C9EE
7F97				C9EF
7242				C9F0
E936				C9F1
7FA1		FB91	11781	C9F2
7FAA				C9F3
7FAD				C9F4
E937				C9F5
E938				C9F6
7FBF				C9F7
7FCF				C9F8
7FCE				C9F9
7FD4	6646	E3C4	07038	C9FA
7FDF				C9FB
7FE0	3F69	9089	03173	C9FC
7FEC				C9FD
7FE5				C9FE
7FEF				CAA1
7FFA				CAA2
EB9C				CAA3
E939				CAA4
800A				CAA5
8014				CAA6
E93A				CAA7
8024				CAA8
E93B				CAA9
8026				CAAA
8030				CAAB
803E				CAAC
803A				CAAD
E93C				CAAE
E93D				CAAF
8075				CAB0
E93E				CAB1
E93F				CAB2
E940				CAB3
80A6				CAB4
80E0				CAB5
E941				CAB6
80D8				CAB7
80FE				CAB8
E942				CAB9
E943				CABA
EB9D				CABB
E944				CABC
E945				CABD
E946				CABE
8118				CABF
8116				CAC0
E947				CAC1
E948				CAC2
814A				CAC3
813A				CAC4
E949				CAC5
E94A				CAC6
E94B				CAC7
E94C				CAC8
E94D				CAC9
7527				CACA
819B				CACB
E94E				CACC
E94F				CACD
81E4				CACE
E950				CACF
E951				CAD0
81F6				CAD1
81F8				CAD2
81F9				CAD3
E952				CAD4
E953				CAD5
E954				CAD6
E955				CAD7
E956				CAD8
821D				CAD9
8221				CADA
8222				CADB
8228				CADC
8234				CADD
E957				CADE
8246				CADF
EB9E				CAE0
824E				CAE1
E958				CAE2
E959				CAE3
8274				CAE4
8C54				CAE5
8279				CAE6
E95A				CAE7
E95B				CAE8
E95C				CAE9
E95D				CAEA
E95E				CAEB
E95F				CAEC
E960				CAED
828E				CAEE
E961				CAEF
E962				CAF0
E963				CAF1
E964				CAF2
E965				CAF3
E966				CAF4
E967				CAF5
E968				CAF6
E969				CAF7
E96A				CAF8
E96B				CAF9
E96C				CAFA
E96D				CAFB
E96E				CAFC
E96F				CAFD
E970				CAFE
E971				CBA1
E972				CBA2
E973				CBA3
82B0				CBA4
82A9				CBA5
82B7				CBA6
E974				CBA7
82BA				CBA8
E975				CBA9
E976				CBAA
E977				CBAB
E978				CBAC
E979				CBAD
E97A				CBAE
E97B				CBAF
E97C				CBB0
E97D				CBB1
E97E				CBB2
82D2	6772	E492	07182	CBB3
8301		FB93	11783	CBB4
82FE				CBB5
E97F				CBB6
E980				CBB7
E981				CBB8
E982				CBB9
E983				CBBA
E984				CBBB
E985				CBBC
E986				CBBD
E987				CBBE
E988				CBBF
E989				CBC0
E98A				CBC1
E98B				CBC2
E98C				CBC3
E98D				CBC4
E98E				CBC5
E98F				CBC6
E990				CBC7
E991				CBC8
E992				CBC9
E993				CBCA
E994				CBCB
E995				CBCC
E996				CBCD
8355				CBCE
E997				CBCF
834A	3755	8C74	02353	CBD0
E998				CBD1
E999				CBD2
E99A				CBD3
837F		FB95	11785	CBD4
E99B				CBD5
E99C				CBD6
E99D				CBD7
E99E				CBD8
E99F				CBD9
E9A0				CBDA
E9A1				CBDB
E9A2				CBDC
E9A3				CBDD
E9A4				CBDE
E9A5				CBDF
E9A6				CBE0
83A7				CBE1
E9A7				CBE2
E9A8				CBE3
838D				CBE4
E9A9				CBE5
E9AA				CBE6
E9AB				CBE7
E9AC				CBE8
E9AD				CBE9
E9AE				CBEA
E9AF				CBEB
E9B0				CBEC
E9B1				CBED
E9B2				CBEE
845B	336B	8A8B	01975	CBEF
E9B3				CBF0
E9B4				CBF1
E9B5				CBF2
E9B6				CBF3
E9B7				CBF4
E9B8				CBF5
E9B9				CBF6
83F6		FB97	11787	CBF7
83C0				CBF8
8411				CBF9
83BE				CBFA
83BF				CBFB
83EA				CBFC
E9BA				CBFD
E9BB				CBFE
83C7		FB96	11786	CCA1
E9BC				CCA2
E9BD				CCA3
E9BE				CCA4
E9BF				CCA5
E9C0				CCA6
E9C1				CCA7
E9C2				CCA8
E9C3				CCA9
E9C4				CCAA
E9C5				CCAB
E9C6				CCAC
E9C7				CCAD
E9C8				CCAE
E9C9				CCAF
E9CA				CCB0
EBBA				CCB1
8448		FB98	11788	CCB2
E9CB				CCB3
844A				CCB4
E9CC				CCB5
844F				CCB6
845A				CCB7
E9CD				CCB8
E9CE				CCB9
E9CF				CCBA
999D				CCBB
E9D0				CCBC
E9D1				CCBD
E9D2				CCBE
E9D3				CCBF
E9D4				CCC0
E9D5				CCC1
84AF				CCC2
84B4		FB99	11789	CCC3
E9D6				CCC4
E9D7				CCC5
E9D8				CCC6
848B	3E55	8FD3	03053	CCC7
E9D9				CCC8
84A6				CCC9
84DC		FA60	11533	CCCA
E9DA				CCCB
E9DB				CCCC
E9DC				CCCD
E9DD				CCCE
E9DE				CCCF
E9DF				CCD0
E9E0				CCD1
E9E1				CCD2
E9E2				CCD3
E9E3				CCD4
E9E4				CCD5
E9E5				CCD6
E9E6				CCD7
E9E7				CCD8
84F0				CCD9
E9E8				CCDA
E9E9				CCDB
E9EA				CCDC
E9EB				CCDD
E9EC				CCDE
E9ED				CCDF
E9EE				CCE0
E9EF				CCE1
E9F0				CCE2
E9F1				CCE3
E9F2				CCE4
E9F3				CCE5
E9F4				CCE6
E9F5				CCE7
E9F6				CCE8
8562				CCE9
8559		FB9B	11791	CCEA
E9F7				CCEB
E9F8				CCEC
856B		FB9C	11792	CCED
8553		FB9A	11790	CCEE
E9F9				CCEF
E9FA				CCF0
E9FB				CCF1
E9FC				CCF2
E9FD				CCF3
E9FE				CCF4
E9FF				CCF5
EA00				CCF6
EA01				CCF7
EA02				CCF8
EA03				CCF9
EA04				CCFA
EA05				CCFB
EA06				CCFC
EA07				CCFD
858F				CCFE
EA08				CDA1
EA09				CDA2
EA0A				CDA3
EA0B				CDA4
EA0C				CDA5
EA0D				CDA6
EA0E				CDA7
EA0F				CDA8
EA10				CDA9
85CE				CDAA
85BF				CDAB
EA11				CDAC
EA12				CDAD
EA13				CDAE
EA14				CDAF
EA15				CDB0
EA16				CDB1
EA17				CDB2
EA18				CDB3
EA19				CDB4
85DC	693C	E55B	07328	CDB5
EA1A				CDB6
EA1B				CDB7
EA1C				CDB8
8610				CDB9
8612				CDBA
EA1D				CDBB
8629				CDBC
EA1E				CDBD
EA1F				CDBE
8638				CDBF
EA20				CDC0
EA21				CDC1
EA22				CDC2
EA23				CDC3
EA24				CDC4
EA25				CDC5
8652				CDC6
8657				CDC7
8656				CDC8
EB9F				CDC9
EA26				CDCA
8663				CDCB
866C				CDCC
8677				CDCD
867A				CDCE
EA27				CDCF
EA28				CDD0
EA29				CDD1
86B1				CDD2
EA2A				CDD3
732C				CDD4
86FC				CDD5
86FD				CDD6
8745				CDD7
95A9				CDD8
8719				CDD9
8758				CDDA
875D				CDDB
8749	4066	90E4	03270	CDDC
87A0				CDDD
8788				CDDE
EA2B				CDDF
87D6				CDE0
EBA0				CDE1
EBA1				CDE2
880B				CDE3
EA2C				CDE4
8801				CDE5
8828				CDE6
8832				CDE7
EA2D				CDE8
8855				CDE9
EA2E				CDEA
8860				CDEB
8864				CDEC
EA2F				CDED
EA30				CDEE
8871				CDEF
EA31				CDF0
EA32				CDF1
8898				CDF2
EA33				CDF3
88AA				CDF4
88A8				CDF5
EA34				CDF6
EA35				CDF7
88C0				CDF8
88BA				CDF9
EA36				CDFA
88CB				CDFB
88CA				CDFC
88DE				CDFD
88CE				CDFE
88DB				CEA1
EA37				CEA2
88F1				CEA3
EA38				CEA4
EA39				CEA5
891A				CEA6
EA3A				CEA7
8906				CEA8
891C		FA5D	11530	CEA9
890A	6A6F	E5ED	07479	CEAA
EBA2				CEAB
8927				CEAC
8930				CEAD
EA3B				CEAE
EA3C				CEAF
EA3D				CEB0
895B				CEB1
EA3E				CEB2
EA3F				CEB3
EA40				CEB4
EA41				CEB5
EA42				CEB6
EA43				CEB7
8994				CEB8
EA44				CEB9
EA45				CEBA
EA46				CEBB
975A				CEBC
EA47				CEBD
EA48				CEBE
89B0				CEBF
EA49				CEC0
89E5				CEC1
EA4A				CEC2
89ED				CEC3
89F9				CEC4
8A07				CEC5
8A05				CEC6
EA4B				CEC7
8A0F				CEC8
8A15				CEC9
8A12		FBA3	11805	CECA
8A11				CECB
EA4C				CECC
EA4D				CECD
8A22				CECE
8A2B				CECF
8A26				CED0
8A4D				CED1
8A4E				CED2
8A58				CED3
8A57				CED4
EA4E				CED5
8A56				CED6
8A45				CED7
8A3D				CED8
8A5D				CED9
8A37		FBA4	11806	CEDA
8A47				CEDB
8A7F				CEDC
8A61				CEDD
8A75				CEDE
8A79		FBA5	11807	CEDF
EA4F				CEE0
8A65				CEE1
8AA7		FBA6	11808	CEE2
8ABE		FBA7	11809	CEE3
8AD7				CEE4
8AB6				CEE5
EA50				CEE6
EA51				CEE7
EA52				CEE8
EA53				CEE9
EA54				CEEA
8AFC				CEEB
8ADF		FBA8	11810	CEEC
8AF6		FBAA	11812	CEED
EA55				CEEE
8AEC				CEEF
EA56				CEF0
EA57				CEF1
EA58				CEF2
8B37				CEF3
EA59				CEF4
8B43				CEF5
8B59				CEF6
8B54				CEF7
8B53		FBAB	11813	CEF8
EA5A				CEF9
8B44				CEFA
EA5B				CEFB
8B45				CEFC
EA5C				CEFD
EBC1				CEFE
91BC				CFA1
EA5D				CFA2
8C39				CFA3
EA5E				CFA4
8C45				CFA5
8C47				CFA6
8C49				CFA7
8C63				CFA8
EA5F				CFA9
EA60				CFAA
72DF				CFAB
EA61				CFAC
EA62				CFAD
8C92				CFAE
EA63				CFAF
EA64				CFB0
8CBA				CFB1
8CB9				CFB2
EA65				CFB3
EA66				CFB4
8CD6				CFB5
EBBB				CFB6
EA67				CFB7
8CE1				CFB8
8CD9				CFB9
EA68				CFBA
EBA3				CFBB
8CF0		FBAD	11815	CFBC
EBA4				CFBD
8CFE				CFBE
EA69				CFBF
EA6A				CFC0
EA6B				CFC1
8D12		FBAF	11817	CFC2
EA6C				CFC3
EA6D				CFC4
EBA5				CFC5
EA6E				CFC6
EA6F				CFC7
8D84				CFC8
EA70				CFC9
8D90				CFCA
8DAB				CFCB
EA71				CFCC
EA72				CFCD
EA73				CFCE
EA74				CFCF
8DD7				CFD0
8DD1				CFD1
8DCE				CFD2
8DDA	6C69	E6E7	07673	CFD3
EA75				CFD4
EA76				CFD5
8E20				CFD6
8E26				CFD7
8E21				CFD8
8E14				CFD9
96AE				CFDA
8E7B				CFDB
8EAF	366D	8BEB	02277	CFDC
8EB3				CFDD
8ECF		FBB2	11820	CFDE
8ED4				CFDF
EA77				CFE0
EA78				CFE1
EA79				CFE2
8EFA				CFE3
8EE8				CFE4
8F02				CFE5
8F1E				CFE6
8F18				CFE7
8F2D				CFE8
EA7A				CFE9
8F40				CFEA
EA7B				CFEB
8F54				CFEC
8F5D				CFED
8F5E				CFEE
EA7C				CFEF
8FB6				CFF0
8FB5				CFF1
EA7D				CFF2
EA7E				CFF3
8FC6				CFF4
8FBE				CFF5
8FBF	4329	9248	03509	CFF6
EA7F				CFF7
EA80				CFF8
EA81				CFF9
EA82				CFFA
8FCD				CFFB
EA83				CFFC
EA84				CFFD
8FE4				CFFE
8FEE				D0A1
8FE8				D0A2
EA85				D0A3
8FEA	6D6C	E78C	07776	D0A4
EA86				D0A5
EA87				D0A6
EA88				D0A7
8FF6				D0A8
EA89				D0A9
EA8A				D0AA
EA8B				D0AB
EA8C				D0AC
EA8D				D0AD
900C				D0AE
901D	4042	90C0	03234	D0AF
EA8E				D0B0
EA8F				D0B1
EA90				D0B2
EA91				D0B3
902D				D0B4
EA92				D0B5
9034				D0B6
902C				D0B7
EA93				D0B8
EA94				D0B9
EA95				D0BA
EA96				D0BB
EA97				D0BC
EA98				D0BD
9044				D0BE
EA99				D0BF
EA9A				D0C0
5FA7				D0C1
EA9B				D0C2
EA9C				D0C3
9065	4D5A	9779	04558	D0C4
EA9D				D0C5
EA9E				D0C6
EA9F				D0C7
EAA0				D0C8
EAA1				D0C9
906E	3C57	8ED5	02855	D0CA
EAA2				D0CB
EAA3				D0CC
EAA4				D0CD
EAA5				D0CE
EAA6				D0CF
9074				D0D0
EAA7				D0D1
907C	4E4B	97C9	04643	D0D2
EAA8				D0D3
885F				D0D4
EAA9				D0D5
EAAA				D0D6
EAAB				D0D7
9088				D0D8
EAAC				D0D9
EAAD				D0DA
EAAE				D0DB
EAAF				D0DC
EAB0				D0DD
EAB1				D0DE
EAB2				D0DF
961D				D0E0
536D				D0E1
90A2				D0E2
90A3	4661	93DF	03865	D0E3
90A0				D0E4
EAB3				D0E5
7941	3737	8C56	02323	D0E6
90B0				D0E7
90B3				D0E8
90C5				D0E9
90C7				D0EA
EAB4				D0EB
EAB5				D0EC
90DD				D0ED
EAB6				D0EE
90EF				D0EF
90F4				D0F0
90F6				D0F1
EAB7				D0F2
EBA6				D0F3
9116				D0F4
9117				D0F5
EAB8				D0F6
9129				D0F7
912F				D0F8
9127				D0F9
9131				D0FA
9134				D0FB
EBA7				D0FC
914F				D0FD
915B				D0FE
9159				D1A1
EAB9				D1A2
9164				D1A3
EABA				D1A4
916D				D1A5
9174				D1A6
9181				D1A7
9185				D1A8
EABB				D1A9
9191				D1AA
EABC				D1AB
9197	4830	94AE	04016	D1AC
919E				D1AD
EABD				D1AE
EABE				D1AF
6BC9				D1B0
91A8				D1B1
91B2				D1B2
91BB				D1B3
EABF				D1B4
EAC0				D1B5
EAC1				D1B6
91D7		FBBB	11829	D1B7
91DA		FBBA	11828	D1B8
91DE		FBBC	11830	D1B9
91EC				D1BA
91ED		FBBD	11831	D1BB
91F1				D1BC
91E5		FBC0	11834	D1BD
91E4		FBBF	11833	D1BE
EAC2				D1BF
EAC3				D1C0
91EE		FBBE	11832	D1C1
91F7				D1C2
91FB				D1C3
9210		FBC2	11836	D1C4
9207				D1C5
9206		FBC1	11835	D1C6
9228				D1C7
9205				D1C8
9216				D1C9
9218				D1CA
920A		FBC3	11837	D1CB
EAC4				D1CC
EAC5				D1CD
9201				D1CE
EBA8				D1CF
EAC6				D1D0
9251		FBC9	11843	D1D1
9238				D1D2
9239		FBCA	11844	D1D3
924F				D1D4
9243				D1D5
9240		FBC5	11839	D1D6
9260				D1D7
EAC7				D1D8
923C		FBC6	11840	D1D9
EAC8				D1DA
9259		FBC8	11842	D1DB
923A		FBC4	11838	D1DC
924E		FBC7	11841	D1DD
9235				D1DE
924A				D1DF
9233				D1E0
EAC9				D1E1
9265				D1E2
EACA				D1E3
9267		FBCB	11845	D1E4
9292				D1E5
EACB				D1E6
EACC				D1E7
EACD				D1E8
927C				D1E9
927D				D1EA
9289				D1EB
92AB				D1EC
EACE				D1ED
92A7		FBCC	11846	D1EE
9277		FBCD	11847	D1EF
92D7		FBD0	11850	D1F0
92DF				D1F1
EACF				D1F2
92BF				D1F3
92D3		FBD6	11856	D1F4
92BA				D1F5
92DD				D1F6
92D9		FBD1	11851	D1F7
92C3				D1F8
92E7		FBCF	11849	D1F9
92CE				D1FA
92BD				D1FB
92E5				D1FC
92BC				D1FD
92E0		FBD5	11855	D1FE
92B6				D2A1
92D8				D2A2
92D5		FBD4	11854	D2A3
92D0		FBD2	11852	D2A4
EAD0				D2A5
EAD1				D2A6
931F				D2A7
9327				D2A8
9321		FBD8	11858	D2A9
9315				D2AA
931E		FBDB	11861	D2AB
9311				D2AC
EBA9				D2AD
EAD2				D2AE
92FB		FBD9	11859	D2AF
EAD3				D2B0
9329				D2B1
9302		FBDE	11864	D2B2
931D		FBDD	11863	D2B3
9325		FBD7	11857	D2B4
92FF		FBDC	11862	D2B5
92F9		FA65	11538	D2B6
EAD4				D2B7
78FA				D2B8
EAD5				D2B9
EBAA				D2BA
9370		FBDF	11865	D2BB
EBAB				D2BC
9364				D2BD
EAD6				D2BE
EAD7				D2BF
EAD8				D2C0
936D				D2C1
9347				D2C2
9357		FBE0	11866	D2C3
9355				D2C4
9349				D2C5
937A				D2C6
935A				D2C7
EAD9				D2C8
EADA				D2C9
EADB				D2CA
EADC				D2CB
EADD				D2CC
EADE				D2CD
EADF				D2CE
EAE0				D2CF
EAE1				D2D0
93BA				D2D1
9388				D2D2
93A4		FBE1	11867	D2D3
93A8				D2D4
EAE2				D2D5
EAE3				D2D6
EAE4				D2D7
93DF				D2D8
93DC				D2D9
93DE		FBE3	11869	D2DA
EAE5				D2DB
93CB				D2DC
93E2				D2DD
93C7				D2DE
93C6		FBE2	11868	D2DF
EAE6				D2E0
940D				D2E1
940F				D2E2
EAE7				D2E3
EAE8				D2E4
93FB				D2E5
EAE9				D2E6
93F8		FBE4	11870	D2E7
9404				D2E8
EAEA				D2E9
93FD				D2EA
943B				D2EB
9432				D2EC
944A				D2ED
9433				D2EE
EAEB				D2EF
EAEC				D2F0
943F				D2F1
9431		FBE5	11871	D2F2
9445		FBE6	11872	D2F3
9448		FBE7	11873	D2F4
9455				D2F5
EAED				D2F6
946B				D2F7
9471				D2F8
EAEE				D2F9
EAEF				D2FA
9588				D2FB
EBAC				D2FC
958E				D2FD
958F	313C	895B	01728	D2FE
958D				D3A1
95AB				D3A2
EAF0				D3A3
95B4				D3A4
95BB	6F65	E885	07969	D3A5
95BD				D3A6
95CB				D3A7
EAF1				D3A8
EAF2				D3A9
95D0				D3AA
EAF3				D3AB
EAF4				D3AC
961E				D3AD
EAF5				D3AE
963D				D3AF
9658				D3B0
9661				D3B1
EAF6				D3B2
EAF7				D3B3
9656				D3B4
EAF8				D3B5
966E				D3B6
EAF9				D3B7
9682				D3B8
EAFA				D3B9
9696				D3BA
9699	3764	8C84	02368	D3BB
EAFB				D3BC
EAFC				D3BD
EAFD				D3BE
EAFE				D3BF
EAFF				D3C0
969D		FBEB	11877	D3C1
96A5				D3C2
96A9				D3C3
96AF		FBEC	11878	D3C4
96BA				D3C5
EB00				D3C6
EB01				D3C7
EB02				D3C8
9D70		FC47	11908	D3C9
EBB7				D3CA
9DB5				D3CB
EB03				D3CC
96E9				D3CD
EB04				D3CE
EB05				D3CF
96F1				D3D0
EB06				D3D1
9702				D3D2
9724	7045	E8C3	08037	D3D3
9733		FBED	11879	D3D4
EBAD				D3D5
9741				D3D6
9743		FBEF	11881	D3D7
974E				D3D8
EBAE				D3D9
EB07				D3DA
974F		FBF1	11883	D3DB
EBAF				D3DC
EB08				D3DD
9755		FBF3	11885	D3DE
EB09				D3DF
975B				D3E0
97CC				D3E1
6BF1				D3E2
9773				D3E3
EB0A				D3E4
EB0B				D3E5
97BC				D3E6
97C1				D3E7
EB0C				D3E8
EB0D				D3E9
97D9				D3EA
97DE				D3EB
97B8				D3EC
EB0E				D3ED
EB0F				D3EE
EB10				D3EF
EB11				D3F0
980E				D3F1
980D				D3F2
980A				D3F3
9813	465C	93DA	03860	D3F4
981E				D3F5
982B				D3F6
982C	4B4B	966A	04343	D3F7
9825				D3F8
EB12				D3F9
9847				D3FA
983B	4951	9570	04149	D3FB
9852				D3FC
9853				D3FD
EB13				D3FE
9857		FBF4	11886	D4A1
9859				D4A2
985B	453F	935E	03731	D4A3
9865		FBF5	11887	D4A4
9866				D4A5
986C				D4A6
9FA5				D4A7
EBBE				D4A8
98AD				D4A9
98B8				D4AA
98BA				D4AB
EBB0				D4AC
98C8				D4AD
EB14				D4AE
EB15				D4AF
EB16				D4B0
EB17				D4B1
98EA				D4B2
EB18				D4B3
EB19				D4B4
98F4	303B	88B9	01627	D4B5
9908				D4B6
9927		FBF8	11890	D4B7
EB1A				D4B8
EB1B				D4B9
7CE6				D4BA
9940				D4BB
994E				D4BC
EB1C				D4BD
EB1D				D4BE
EB1E				D4BF
995C				D4C0
995F				D4C1
999B				D4C2
EBC2				D4C3
EB1F				D4C4
EB20				D4C5
EB21				D4C6
99C3				D4C7
99B9				D4C8
99BA				D4C9
99DA				D4CA
99FD				D4CB
9A03				D4CC
9A02				D4CD
99FE				D4CE
EB22				D4CF
EB23				D4D0
9A28	424D	91CB	03445	D4D1
9A38				D4D2
9A2E				D4D3
9A4A				D4D4
9A41				D4D5
EB24				D4D6
EB25				D4D7
9A56				D4D8
EB26				D4D9
9AAF				D4DA
EB27				D4DB
EBB1				D4DC
EB28				D4DD
9ADC		FC40	11901	D4DE
EB29				D4DF
EB2A				D4E0
9AE9				D4E1
983E				D4E2
9B02				D4E3
9B08				D4E4
9B12				D4E5
9B1B				D4E6
EB2B				D4E7
EB2C				D4E8
EB2D				D4E9
EB2E				D4EA
9B5E				D4EB
9B5B				D4EC
9B6D				D4ED
9B73				D4EE
EBB2				D4EF
9B75		FC41	11902	D4F0
9B66				D4F1
9B79				D4F2
9B8F		FC43	11904	D4F3
9B86				D4F4
9B90				D4F5
EB2F				D4F6
EB30				D4F7
9B84				D4F8
9B87				D4F9
9B9E				D4FA
9BB1		FC44	11905	D4FB
9BB2				D4FC
EB31				D4FD
EB32				D4FE
9BC7				D5A1
EB33				D5A2
EB34				D5A3
9BD0				D5A4
9BCE				D5A5
EB35				D5A6
EB36				D5A7
9BBB		FC45	11906	D5A8
9BEB				D5A9
9BE5				D5AA
EB37				D5AB
EB38				D5AC
9BEA				D5AD
9C0B				D5AE
9C00		FC46	11907	D5AF
EB39				D5B0
9BFA				D5B1
EB3A				D5B2
9C23				D5B3
9C29				D5B4
EB3B				D5B5
9C2A				D5B6
9C31				D5B7
EB3C				D5B8
9C53				D5B9
EB3D				D5BA
9C63				D5BB
9C69				D5BC
9C6E				D5BD
9C70				D5BE
9C77				D5BF
EB3E				D5C0
9D07	463E	93BC	03830	D5C1
9D0B				D5C2
9D0E	322A	89A8	01810	D5C3
9D1E				D5C4
9D1D				D5C5
EBB8				D5C6
9D18				D5C7
EB3F				D5C8
9D34				D5C9
9D47				D5CA
EB40				D5CB
9D43				D5CC
96DE				D5CD
EB41				D5CE
9D62				D5CF
EBB3				D5D0
9D77				D5D1
9D8D				D5D2
9D7C				D5D3
9D6B		FC48	11909	D5D4
EB42				D5D5
9DA1				D5D6
9D96				D5D7
9DC3				D5D8
9DAE				D5D9
EB43				D5DA
9DEE				D5DB
9E19		FC4A	11911	D5DC
9E1C				D5DD
EB44				D5DE
9E78	3834	8CB2	02420	D5DF
EB45				D5E0
EB46				D5E1
9E8E				D5E2
9E98				D5E3
9EA4				D5E4
9EAC				D5E5
EB47				D5E6
EB48				D5E7
EBB4				D5E8
9EC8				D5E9
EB49				D5EA
EBB5				D5EB
9EDB	4263	91E1	03467	D5EC
9F09				D5ED
EB4A				D5EE
9F10				D5EF
97BA				D5F0
9F19				D5F1
9F37				D5F2
9F39				D5F3
EB4B				D5F4
9F41				D5F5
816D				D5F6
EB4C				D5F7
EB4D				D5F8
9F94				D5F9
EB4E				D5FA
9F9C	737D	EA9D	08393	D5FB
EB4F				D5FC
5516	3022	88A0	01602	D6A1
9C2F	3073	88F1	01683	D6A2
6DEB	307C	88FA	01692	D6A3
8FC2	312A	8949	01710	D6A4
6B1D	3135	8954	01721	D6A5
53A9	3139	8958	01725	D6A6
990C	3142	8961	01734	D6A7
7114	316B	898B	01775	D6A8
8956	3228	89A6	01808	D6A9
8FE6	3260	89DE	01864	D6AA
6062	327A	89F8	01890	D6AB
6666	3322	8A41	01902	D6AC
9784	3373	8A93	01983	D6AD
565B	337A	8A9A	01990	D6AE
7FF0	344D	8ACB	02045	D6AF
7FEB	3465	8AE3	02069	D6B0
5FBD	352B	8B4A	02111	D6B1
50C5	364F	8BCD	02247	D6B2
55B0	3674	8BF2	02284	D6B3
5C51	367D	8BFB	02293	D6B4
9774	3724	8C43	02304	D6B5
7E4B	3752	8C71	02350	D6B6
5026	3771	8C91	02381	D6B7
6372	377E	8C9E	02394	D6B8
8AFA	3841	8CBF	02433	D6B9
5DF7	392B	8D4A	02511	D6BA
9EB9	396D	8D8D	02577	D6BB
9D60	3974	8D94	02584	D6BC
91C7	3A53	8DD1	02651	D6BD
85A9	3B27	8E46	02707	D6BE
9BD6	3B2A	8E49	02710	D6BF
9306	3B2C	8E4B	02712	D6C0
6753	3C5D	8EDB	02861	D6C1
707C	3C5E	8EDC	02862	D6C2
914B	3D36	8F55	02922	D6C3
66D9	3D6C	8F8C	02976	D6C4
85AF	3D72	8F92	02982	D6C5
85F7	3D73	8F93	02983	D6C6
5EE0	3E33	8FB1	03019	D6C7
91A4	3E5F	8FDD	03063	D6C8
9798	3E64	8FE2	03068	D6C9
8755	3F2A	9049	03110	D6CA
976D	3F59	9078	03157	D6CB
9017	3F60	9080	03164	D6CC
647A	4022	90A0	03202	D6CD
64B0	4071	90EF	03281	D6CE
714E	4079	90F7	03289	D6D0
717D	407A	90F8	03290	D6D1
9061	414C	916B	03344	D6D3
63BB	415F	917E	03363	D6D4
75E9	4169	9189	03373	D6D5
905C	423D	91BB	03429	D6D6
817F	425C	91DA	03460	D6D7
5544	426F	91ED	03479	D6D8
86F8	427D	91FB	03493	D6D9
9C48	432D	924C	03513	D6DA
7BAA	433D	925C	03529	D6DB
8A3B	4370	9290	03580	D6DC
7026	4375	9295	03585	D6DD
51CB	437C	929C	03592	D6DE
6357	443D	92BB	03629	D6DF
69CC	4448	92C6	03640	D6E0
939A	444A	92C8	03642	D6E1
63B4	444F	92CD	03647	D6E2
912D	4522	9341	03702	D6E3
64E2	4527	9346	03707	D6E4
6EBA	452E	934D	03714	D6E5
586B	4536	9355	03722	D6E6
5835	4548	9367	03740	D6E7
5C60	454B	936A	03743	D6E8
83DF	4551	9370	03749	D6E9
8CED	4552	9371	03750	D6EA
5858	4564	9384	03768	D6EB
6D9C	4642	93C0	03834	D6EC
9041	465B	93D9	03859	D6ED
8B0E	4666	93E4	03870	D6EE
7058	4667	93E5	03871	D6EF
6962	466A	93E8	03874	D6F0
8FE9	4676	93F4	03886	D6F1
56A2	4739	9458	03925	D6F2
724C	4757	9476	03955	D6F3
9019	4767	9487	03971	D6F4
79E4	4769	9489	03973	D6F5
7BB8	4824	94A2	04004	D6F6
633D	4854	94D2	04052	D6F7
67CA	4922	9541	04102	D6F8
7A17	4923	9542	04103	D6F9
903C	492F	954E	04115	D6FA
5A9B	4932	9551	04118	D6FB
8B2C	4935	9554	04121	D6FC
5EDF	4940	955F	04132	D6FD
7015	494E	956D	04146	D6FE
853D	4A43	95C1	04235	D7A1
77A5	4A4D	95CB	04245	D7A2
5A29	4A5A	95D8	04258	D7A3
5E96	4A79	95F7	04289	D7A4
84EC	4B29	9648	04309	D7A5
9C52	4B70	9690	04380	D7A6
9EBA	4C4D	96CB	04445	D7A7
9905	4C5F	96DD	04463	D7A8
7C7E	4C62	96E0	04466	D7A9
9453	4C7A	96F8	04490	D7AA
6108	4C7C	96FA	04492	D7AB
7337	4D32	9751	04518	D7AC
7194	4D50	976F	04548	D7AD
8000	4D54	9773	04552	D7AE
6F23	4E7A	97F8	04690	D7AF
7149	4E7B	97F9	04691	D7B0
84EE	4F21	9840	04701	D7B1
6994	4F31	9850	04717	D7B2
874B	4F39	9858	04725	D7B3
5154	513D	995C	04929	D7B4
5195	514B	996A	04943	D7B5
51A4	514D	996C	04945	D7B6
5539	5330	9A4F	05116	D7B7
5533	533A	9A59	05126	D7B8
5632	535E	9A7D	05162	D7B9
56A5	536B	9A8B	05175	D7BA
580B	5444	9AC2	05236	D7BB
5ABE	553D	9B5C	05329	D7BC
5E64	5672	9BF0	05482	D7BD
6097	5824	9CA2	05604	D7BE
6369	5960	9D80	05764	D7BF
6406	596C	9D8C	05776	D7C0
689B	5B6B	9E8B	05975	D7C1
688D	5B74	9E94	05984	D7C2
6E6E	5E50	9FCE	06248	D7C3
7228	6026	E0A4	06406	D7C4
7504	612B	E14A	06511	D7C5
750D	6130	E14F	06516	D7C6
7515	6131	E150	06517	D7C7
787C	626F	E1ED	06679	D7C8
7A31	634A	E269	06742	D7C9
9F9D	6354	E273	06752	D7CA
7B99	6439	E2B7	06825	D7CB
7C90	6464	E2E2	06868	D7CC
7CAE	646E	E2EC	06878	D7CD
7D9B	6539	E358	06925	D7CE
7DAE	653B	E35A	06927	D7CF
7D9F	6546	E365	06938	D7D0
822E	6764	E484	07168	D7D1
EB50				D7D2
EB51				D7D3
8375	683B	E4B9	07227	D7D4
EB52				D7D5
EB53				D7D6
EB54				D7D7
89AF	6B32	E651	07518	D7D8
8ADE	6B66	E686	07570	D7D9
EB55				D7DA
8E09	6C74	E6F2	07684	D7DB
EB56				D7DC
9058	6E29	E7A7	07809	D7DD
6248	6E3D	E7BB	07829	D7DE
EB57				D7DF
9760	7051	E8CF	08049	D7E0
EB58				D7E1
EB59				D7E2
EB5A				D7E3
9B97	723C	E9BA	08228	D7E4
9BF2	724E	E9CC	08246	D7E5
9EAA	7351	EA70	08349	D7E6
EB5B				D7E9
EB5C				D7EA
5E30	3522	8B41	02102	D7EB
8003	394D	8D6C	02545	D7EC
62F7	3969	8D89	02573	D7ED
EB5D				D7EE
9E97	4E6F	97ED	04679	D7EF
EB5E				D7F0
5DC9	565A	9BD8	05458	D7F1
EB5F				D7F2
EB60				D7F3
6A90	5C79	9EF7	06089	D7F4
EB61				D7F5
EB62				D7F6
EB63				D7F7
7672	6221	E19F	06601	D7F8
892B	6A75	E5F3	07485	D7F9
EB64				D7FA
EB65				D7FB
EB66				D7FC
9B4D	7233	E9B1	08219	D7FD
EB68				D7FE
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	2821	849F	00801	
	282C	84AA	00812	
	2822	84A0	00802	
	282D	84AB	00813	
	2823	84A1	00803	
	282E	84AC	00814	
	2824	84A2	00804	
	282F	84AD	00815	
	2826	84A4	00806	
	2831	84AF	00817	
	2825	84A3	00805	
	2830	84AE	00816	
	2827	84A5	00807	
	283C	84BA	00828	
	2837	84B5	00823	
	2832	84B0	00818	
	2829	84A7	00809	
	283E	84BC	00830	
	2839	84B7	00825	
	2834	84B2	00820	
	2828	84A6	00808	
	2838	84B6	00824	
	283D	84BB	00829	
	2833	84B1	00819	
	282A	84A8	00810	
	283A	84B8	00826	
	283F	84BD	00831	
	2835	84B3	00821	
	282B	84A9	00811	
	283B	84B9	00827	
	2840	84BE	00832	
	2836	84B4	00822	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5029	98A7	04809	
	3557	8B76	02155	
				
				
				
				
				
				
				
				
	5034	98B2	04820	
				
				
	503F	98BD	04831	
				
	5041	98BF	04833	
				
				
				
				
				
				
				
				
				
				
				
				
	3873	8CF1	02483	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5079	98F7	04889	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	417C	919C	03392	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	3878	8CF6	02488	
				
				
				
				
				
				
				
				
				
				
	5163	9983	04967	
	4A2C	95AA	04212	
				
	516C	998C	04976	
				
				
	517B	999B	04991	
				
				
				
				
				
	5179	9999	04989	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5238	99B6	05024	
				
				
				
				
				
				
				
				
	523F	99BD	05031	
				
	485C	94DA	04060	
				
				
				
				
				
				
				
				
				
				
				
	3A35	8DB3	02621	
				
				
	5267	99E5	05071	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	527B	99F9	05091	
				
				
				
				
				
				
				
				
				
	532B	9A4A	05111	
				
				
				
				
				
				
				
				
	5337	9A56	05123	
				
				
				
				
				
	5348	9A67	05140	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5377	9A97	05187	
				
				
				
				
				
	5423	9AA1	05203	
				
				
				
				
				
				
				
	3651	8BCF	02249	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	444D	92CB	03645	
				
				
				
				
				
				
				
				
	4446	92C4	03638	
				
				
				
	3968	8D88	02572	
				
				
	546A	9AE8	05274	
				
				
				
				
				
				
		FAA0	11602	
				
				
		FAA3	11605	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	3247	89C5	01839	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5559	9B78	05357	
				
				
				
				
				
				
				
				
	3248	89C6	01840	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5638	9BB6	05424	
				
		FAAF	11617	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5726	9C45	05506	
				
	572E	9C4D	05514	
	572F	9C4E	05515	
	3276	89F4	01886	
	4736	9455	03922	
				
				
	5739	9C58	05525	
		FA67	11540	
				
	5741	9C60	05533	
				
				
				
				
				
				
	3D5B	8F7A	02959	
				
				
		FABB	11629	
				
				
				
				
				
				
				
		FABC	11630	
				
				
				
	5827	9CA5	05607	
				
				
				
				
				
	575C	9C7B	05560	
				
				
				
				
				
				
		FAC3	11637	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	5862	9CE0	05666	
				
				
				
				
				
				
				
				
				
	592D	9D4C	05713	
	592E	9D4D	05714	
				
				
				
				
				
				
				
				
				
				
				
				
				
	4472	92F0	03682	
				
				
				
				
				
				
				
				
				
				
	5969	9D89	05773	
	5964	9D84	05768	
				
				
				
		FAC9	11643	
				
				
				
				
				
		FACA	11644	
	5975	9D95	05785	
				
				
				
	597B	9D9B	05791	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	555D	9B7C	05361	
				
				
	404D	90CB	03245	
				
	4960	9580	04164	
				
				
				
				
				
				
	5A61	9DDF	05865	
	3937	8D56	02523	
				
				
				
	5A64	9DE2	05868	
				
				
				
				
				
				
				
				
				
		FADE	11664	
				
				
				
	3E73	8FF1	03083	
	3E3E	8FBC	03030	
				
				
				
				
				
				
				
				
	4073	90F1	03283	
				
				
				
				
				
				
	5B6C	9E8C	05976	
				
	393C	8D5B	02528	
				
				
				
		FAE7	11673	
				
	5C27	9EA5	06007	
				
	3F22	9041	03102	
	3371	8A91	01981	
				
				
				
	5C40	9EBE	06032	
				
	3D5D	8F7C	02961	
	364B	8BC9	02243	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	464B	93C9	03843	
				
				
				
				
				
				
	5D29	9F48	06109	
				
				
				
				
				
				
				
				
	3F23	9042	03103	
				
				
	334C	8A6B	01944	
				
				
				
				
	5D61	9F81	06165	
				
				
				
				
				
				
	3168	8988	01772	
				
				
				
				
				
				
				
				
				
				
				
				
	5E37	9FB5	06223	
				
				
		FB42	11703	
				
	4338	9257	03524	
				
	5E61	9FDF	06265	
				
		FB48	11709	
				
				
				
	4C2B	96A9	04411	
				
				
				
				
				
				
				
				
				
				
	5E78	9FF6	06288	
				
	3C3F	8EBD	02831	
				
				
				
	5F21	E040	06301	
	3443	8AC1	02035	
				
				
				
				
				
				
				
	396A	8D8A	02574	
				
				
				
				
	316A	898A	01774	
				
				
		FA62	11535	
				
				
				
				
	5F69	E089	06373	
				
				
				
	472E	944D	03914	
				
				
		FB59	11726	
				
				
				
				
	6022	E0A0	06402	
				
	4C6C	96EA	04476	
	602B	E0A9	06411	
				
				
				
				
				
				
				
				
				
				
				
	6044	E0C2	06436	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	6062	E0E0	06466	
				
	4849	94C7	04041	
				
		FB66	11739	
				
	4276	91F4	03486	
				
				
				
				
	6071	E0EF	06481	
				
		FB6B	11744	
				
				
				
				
				
				
				
				
				
	4953	9572	04151	
				
	6134	E153	06520	
				
				
				
				
				
				
				
				
				
	6159	E178	06557	
				
				
	6162	E182	06566	
				
				
	6168	E188	06572	
				
				
				
				
	6173	E193	06583	
				
				
				
	6229	E1A7	06609	
				
		FB72	11751	
				
				
	622E	E1AC	06614	
	622F	E1AD	06615	
				
	4B5F	967E	04363	
	443E	92BC	03630	
				
	3D62	8F82	02966	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	6260	E1DE	06664	
	6262	E1E0	06666	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	3F41	9060	03133	
				
	6346	E265	06738	
				
				
	3254	89D2	01852	
				
				
				
				
				
				
				
	376A	8C8A	02374	
				
				
				
	6359	E278	06757	
				
				
				
				
		FB85	11769	
				
				
				
				
				
				
				
				
				
				
				
	6425	E2A3	06805	
				
	6427	E2A5	06807	
				
				
				
				
				
				
				
				
				
				
				
	4A34	95B2	04220	
	4C30	96AE	04416	
				
	646B	E2E9	06875	
				
				
				
				
	6524	E343	06904	
				
	4A36	95B4	04222	
				
				
				
		FB8B	11775	
				
				
				
				
				
	6537	E356	06923	
				
				
	653D	E35C	06929	
				
				
	416D	918D	03377	
	6549	E368	06941	
				
				
				
	316F	898F	01779	
	4B25	9644	04305	
				
				
	6541	E360	06933	
				
				
				
				
				
				
				
				
				
				
				
	4356	9275	03554	
				
				
	3227	89A5	01807	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	665C	E3DA	07060	
				
	416F	918F	03379	
				
	6662	E3E0	07066	
	6663	E3E1	07067	
				
				
				
				
				
				
				
				
				
				
				
				
	6724	E443	07104	
				
	6727	E446	07107	
				
	4928	9547	04108	
				
				
				
				
				
				
				
				
				
	4125	9144	03305	
				
				
				
				
				
				
	6761	E481	07165	
	6768	E488	07172	
				
				
				
				
	6769	E489	07173	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	682F	E4AD	07215	
				
				
	4D69	9789	04573	
				
	365D	8BDB	02261	
				
				
				
				
				
				
				
				
				
				
				
				
				
	4C58	96D6	04456	
				
				
	686D	E4EB	07277	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	694A	E569	07342	
				
				
				
	303A	88B8	01626	
				
				
				
	6960	E580	07364	
				
				
				
	6969	E589	07373	
	6963	E583	07367	
				
				
				
				
	6A21	E59F	07401	
	6A27	E5A5	07407	
				
				
				
				
	6A2F	E5AD	07415	
				
				
				
				
				
				
				
				
				
				
	6A49	E5C7	07441	
				
				
				
				
				
				
				
	6A51	E5CF	07449	
				
				
				
				
		FBA2	11804	
				
				
				
	6B24	E643	07504	
	6B25	E644	07505	
				
				
				
	6B30	E64F	07516	
				
				
				
				
				
				
				
	3F56	9075	03154	
	6B42	E661	07534	
	6B43	E662	07535	
	3E59	8FD7	03057	
				
				
				
				
				
				
				
				
				
				
				
	434C	926B	03544	
				
				
	3F5B	907A	03159	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	6C2F	E6AD	07615	
				
				
	465A	93D8	03858	
				
	3E5D	8FDB	03061	
	396B	8D8B	02575	
				
				
				
	494F	956E	04147	
	6C44	E6C2	07636	
				
				
				
				
				
				
	6C57	E6D5	07655	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	6C7C	E6FA	07692	
				
				
				
				
				
				
				
				
	6D3A	E759	07726	
				
				
				
				
				
				
	6D54	E773	07752	
				
				
				
				
	4A56	95D4	04254	
				
				
				
				
	6D75	E795	07785	
				
				
				
				
	6E27	E7A5	07807	
				
				
		FBB5	11823	
	6E33	E7B1	07819	
				
				
				
				
				
				
				
				
				
				
	6E3C	E7BA	07828	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	3378	8A98	01988	
				
	6E60	E7DE	07864	
	6E5F	E7DD	07863	
				
				
				
				
				
	3174	8994	01784	
				
				
				
				
		FBCE	11848	
				
				
		FA5F	11532	
				
	6E74	E7F2	07884	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
		FA5E	11531	
				
				
				
				
	6F2B	E84A	07911	
				
				
	3830	8CAE	02416	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
	6F52	E871	07950	
	6F58	E877	07956	
				
				
	6F5C	E87B	07960	
				
				
				
				
				
				
				
				
				
				
				
				
	4262	91E0	03466	
				
				
	4A37	95B5	04223	
				
	703D	E8BB	08029	
				
				
	7353	EA72	08351	
	7056	E8D4	08054	
	7057	E8D5	08055	
				
	4A5C	95DA	04260	
				
				
				
				
				
	7073	E8F1	08083	
	3468	8AE6	02072	
	4852	94D0	04050	
	7075	E8F3	08085	
				
				
				
	7123	E942	08103	
				
				
				
				
	712D	E94C	08113	
				
	3B41	8E60	02733	
				
	7135	E954	08121	
				
				
				
				
				
				
				
	3642	8BC0	02234	
				
				
				
	7149	E968	08141	
	477D	949D	03993	
				
				
	714B	E96A	08143	
				
	7150	E96F	08148	
				
				
				
				
				
	715B	E97A	08159	
		FBFB	11893	
	7165	E985	08169	
				
				
				
				
				
				
				
				
				
	722D	E9AB	08213	
				
				
				
				
				
				
				
				
				
	3B2D	8E4C	02713	
	724F	E9CD	08247	
				
				
	725D	E9DB	08261	
				
				
	7268	E9E6	08272	
				
	727A	E9F8	08290	
				
				
				
				
				
	7339	EA58	08325	
				
				
				
				
				
				
				
				
				
				
				
	7367	EA87	08371	
	7368	EA88	08372	
				
				
				
				
				
	736C	EA8C	08376	
				
				
				
				
				
				
				
				
				
				
				
		FAE0	11666	
				
				
				
		FBE9	11875	
		FA90	11580	
		FA9B	11591	
		FA9C	11592	
		FAB1	11619	
		FAD8	11658	
		FAE8	11674	
		FAEA	11676	
		FB58	11725	
		FB5E	11731	
		FB75	11754	
		FB7D	11762	
		FB7E	11763	
		FB80	11764	
		FB82	11766	
		FB86	11770	
		FB89	11773	
				
		FB9D	11793	
		FB9F	11801	
		FBA0	11802	
		FBA9	11811	
		FBB1	11819	
		FBB3	11821	
		FBB4	11822	
		FBB7	11825	
		FBD3	11853	
		FBDA	11860	
		FBEA	11876	
		FBF6	11888	
		FBF7	11889	
		FBF9	11891	
		FC49	11910	
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
				
		FA57	11524	
		FA56	11523	
	215D	817C	00161	
	2141	8160	00133	
	224C	81CA	00244	
