#!/usr/bin/perl

# Copyright (C) 2012, Benjamin Drung <bdrung@debian.org>
# Copyright (C) 2016, Jonas Smedegaard <dr@jones.dk>
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

use strict;
use warnings;

use Path::Tiny;

use Test::More tests => 10;
use Test::Command;

my @COMMAND
	= ( defined( $ARGV[0] ) and $ARGV[0] eq '--installed' )
	? ('licensecheck')
	: ( 'perl', 'bin/licensecheck' );

sub license
{
	my ( $file, $stdout ) = @_;

	my $command = Test::Command->new(
		cmd => [ @COMMAND, "t/devscripts/$file" ],
	);

	$command->stdout_is_eq("t/devscripts/$file: $stdout\n");
	$command->stderr_is_eq('');
	$command->exit_is_num(0);
}

sub license2
{
	my ( $args, $file, $stdout ) = @_;

	my @args = split( ' ', $args );
	my $command = Test::Command->new(
		cmd => [ @COMMAND, @args, "t/devscripts/$file" ],
	);

	$command->stdout_is_eq("t/devscripts/$file	$stdout\n");
	$command->stderr_is_eq('');
	$command->exit_is_num(0);
}

subtest 'MultiLine declaration' => sub {

	# test copyright declared on 2 lines
	license2 '-m --copyright', 'bsd-regents.c',
		'BSD (3 clause)	1987, 1993, 1994 The Regents of the University of California. All rights reserved.';

	# or 3 lines
	license2 '-m --copyright', 'texinfo.tex',
		'GPL (v3 or later)	1985, 1986, 1988, 1990, 1991, 1992, 1993, 1994, 1995 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012 Free Software Foundation, Inc.';

	# BTS #519080
	license2 '-m --copyright', 'multi-line-copyright.c',
		'Public domain GPL (v3)	2008 Aaron Plattner, NVIDIA Corporation / 2005 Lars Knoll & Zack Rusin, Trolltech / 2000 Keith Packard, member of The XFree86 Project, Inc';
};

subtest 'Duplicated copyright' => sub {
	license2 '-m --copyright', 'duplicated-copyright.rb',
		'Apache (v2.0)	2002-2015, OpenNebula Project (OpenNebula.org), C12G Labs';
};

subtest 'Dual' => sub {
	license 'dual.c', 'Public domain GPL (v3)';
};

subtest 'Machine' => sub {
	license2 '-m',        'beerware.cpp', 'Beerware';
	license2 '--machine', 'lgpl-2.1.h',   'LGPL (v2.1)';
	license2 '-m --copyright', 'gpl-2', 'GPL (v2)	2012 Devscripts developers';
};

subtest 'Fortran comments' => sub {
	license 'bsd.f', 'BSD (2 clause)';
};

subtest 'Comments detection' => sub {
	license 'comments-detection.h',   'GPL (v3 or later)';
	license 'comments-detection.txt', '*No copyright* LGPL (v2.1 or later)';
};

subtest 'False positives' => sub {
	license2 '-m --copyright', 'false-positives',
		'Public domain	2013 Devscripts developers';
};

subtest 'Regexp killer' => sub {
	license 'regexp-killer.c', 'UNKNOWN';
};

subtest 'Encoding' => sub {
	license2 '-m --copyright --encoding iso-8859-1', 'copr-iso8859.h',
		'GPL (v2) (with incorrect FSF address)	2011 Heinrich Müller <henmull@src.gnome.org>';
	license2 '-m --copyright --encoding utf8', 'copr-utf8.h',
		"GPL (v2 or later)	2004-2015 Oliva 'f00' Oberto / 2001-2010 Paul 'bar' Stevénsön";

	# test wrong user choice and fallback
	license2 '-m --copyright --encoding utf8', 'copr-iso8859.h',
		'GPL (v2) (with incorrect FSF address)	2011 Heinrich Müller <henmull@src.gnome.org>';
};

subtest 'Info at end' => sub {
	license2 '-m --copyright', 'info-at-eof.h',
		'MIT/X11 (BSD like)	1994-2012 Lua.org, PUC-Rio. All rights reserved';
};
