package preload;

our $DATE = '2015-03-24'; # DATE
our $VERSION = '0.01'; # VERSION

use 5.014;
use strict;
use warnings;

use constant PRELOAD => $ENV{PRELOAD};

use Keyword::API;

sub import {
    no strict 'refs';

    my $class = shift;

    install_keyword(__PACKAGE__, 'load');

    # XXX can't install_keyword() twice? so currently we cheat and make preload
    # as just an "alias" to require
    my $caller = caller();
    *{"$caller\::preload"} = sub { my $mod_pm = shift; $mod_pm =~ s!::!/!g; $mod_pm .= ".pm"; require $mod_pm };
}

sub unimport { uninstall_keyword() }

sub parser {
    lex_read_space(0);
    my $module = lex_unstuff_to_ws();
    lex_stuff("unless (preload::PRELOAD) { require $module }");
};

1;
# ABSTRACT: Load and preload modules

__END__

=pod

=encoding UTF-8

=head1 NAME

preload - Load and preload modules

=head1 VERSION

This document describes version 0.01 of preload (from Perl distribution preload), released on 2015-03-24.

=head1 SYNOPSIS

 use preload;

 # Foo::Bar will be require'd when $ENV{PRELOAD} is true
 preload Foo::Bar;

 sub mysub {
     # Foo::Bar will be require'd when $ENV{PRELOAD} is false
     load Foo::Bar;
 }

=head1 DESCRIPTION

B<STATUS: Experimental, interface will likely change.>

When running a script, especially one that has to start quickly, it's desirable
to delay loading modules until it's actually used, to reduce startup overhead.

When running a (preforking) daemon, it's usually desirable to preload modules at
startup, so the daemon can then service clients without any further delay from
loading modules, and the loading before forking means child processes can share
the module code (reduced memory usage).

This pragma module tries to offer the best of both worlds. This statement:

 use preload;

will declare a constant C<PRELOAD> (currently set to C<$ENV{PRELOAD}>) and
introduce two new keywords: C<preload> and C<load>. C<preload> is defined to be:

 if (PRELOAD) { require $module }

this means it will become a no-op when PRELOAD is false. On the other hand,
C<load> is defined to be:

 unless (PRELOAD) { require $module }

this means it will become a no-op when PRELOAD is true.

With this module you can avoid run-time penalty associated with conditional
loading.

=for Pod::Coverage .+

=head1 SEE ALSO

L<prefork>

L<Dist::Zilla::Plugin::Preload>

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/preload>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-preload>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=preload>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
