/*  You may distribute under the terms of either the GNU General Public License
 *  or the Artistic License (the same terms as Perl itself)
 *
 *  (C) Paul Evans, 2014 -- leonerd@leonerd.org.uk
 */


#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include <vterm.h>

#include <string.h>

#define streq(a,b)  (!strcmp(a,b))

#define FREE_CB(v)  if(self->v) SvREFCNT_dec(self->v)

typedef struct Term__VTerm {
  VTerm *vt;

  struct {
    CV *text;
    CV *control;
    CV *escape;
    CV *csi;
    CV *osc;
    CV *dcs;
    CV *resize;
  } parser_cb;

} *Term__VTerm;

typedef VTermColor     *Term__VTerm__Color;
typedef VTermGlyphInfo *Term__VTerm__GlyphInfo;
typedef VTermLineInfo  *Term__VTerm__LineInfo;
typedef VTermPos       *Term__VTerm__Pos;
typedef VTermRect      *Term__VTerm__Rect;

typedef struct Term__VTerm__State {
  VTermState *state;
  SV         *vterm;

  struct {
    CV *putglyph;
    CV *movecursor;
    CV *scrollrect;
    CV *moverect;
    CV *erase;
    CV *initpen;
    CV *setpenattr;
    CV *settermprop;
    CV *bell;
    CV *resize;
    CV *setlineinfo;
  } cb;
} *Term__VTerm__State;

static SV *newSVcolor(VTermColor *col)
{
  VTermColor *self;
  SV *sv = newSV(0);

  Newx(self, 1, VTermColor);
  *self = *col;

  sv_setref_pv(sv, "Term::VTerm::Color", self);
  return sv;
}

static SV *newSVlineinfo(const VTermLineInfo *info)
{
  VTermLineInfo *self;
  SV *sv = newSV(0);

  Newx(self, 1, VTermLineInfo);
  *self = *info;

  sv_setref_pv(sv, "Term::VTerm::LineInfo", self);
  return sv;
}

static SV *newSVglyphinfo(VTermGlyphInfo *info)
{
  VTermGlyphInfo *self;
  SV *sv = newSV(0);
  int nchars, i;

  for(nchars = 0; info->chars[nchars]; nchars++)
    ;
  nchars++; // include the terminating NUL

  Newxc(self, sizeof(VTermGlyphInfo) + nchars * sizeof(uint32_t), char, VTermGlyphInfo);
  *self = *info;
  self->chars = (uint32_t *)(((char *)self) + sizeof(VTermGlyphInfo));

  for(i = 0; i < nchars; i++)
    // This is our own glyphinfo so we're allowed to write it. Honest gov
    ((uint32_t *)self->chars)[i] = info->chars[i];

  sv_setref_pv(sv, "Term::VTerm::GlyphInfo", self);
  return sv;
}

static SV *newSVpos(VTermPos pos)
{
  VTermPos *self;
  SV *sv = newSV(0);

  Newx(self, 1, VTermPos);
  *self = pos;

  sv_setref_pv(sv, "Term::VTerm::Pos", self);
  return sv;
}

static SV *newSVrect(VTermRect rect)
{
  VTermRect *self;
  SV *sv = newSV(0);

  Newx(self, 1, VTermRect);
  *self = rect;

  sv_setref_pv(sv, "Term::VTerm::Rect", self);
  return sv;
}

static SV *newSVvalue(VTermValue *val, VTermValueType type)
{
  switch(type) {
    case VTERM_VALUETYPE_BOOL:
      return val->boolean ? &PL_sv_yes : &PL_sv_no;
    case VTERM_VALUETYPE_INT:
      return newSViv(val->number);
    case VTERM_VALUETYPE_STRING:
      return newSVpv(val->string, 0);
    case VTERM_VALUETYPE_COLOR:
      return newSVcolor(&val->color);
  }
}

static int parser_text(const char *bytes, size_t len, void *user)
{
  dSP;
  Term__VTerm self = user;
  CV *cb = self->parser_cb.text;

  SV *str = newSVpv(bytes, len);
  if(vterm_get_utf8(self->vt))
    SvUTF8_on(str);

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 1);
  mPUSHs(str);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return len;
}

static int parser_control(unsigned char control, void *user)
{
  dSP;
  Term__VTerm self = user;
  CV *cb = self->parser_cb.control;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 1);
  mPUSHi(control);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int parser_escape(const char *bytes, size_t len, void *user)
{
  dSP;
  Term__VTerm self = user;
  CV *cb = self->parser_cb.escape;

  SV *str = newSVpv(bytes, len);

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 1);
  mPUSHs(str);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return len;
}

static int parser_csi(const char *leader, const long args[], int argcount, const char *intermed, char command, void *user)
{
  dSP;
  Term__VTerm self = user;
  int i;
  CV *cb = self->parser_cb.csi;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 2);

  if(leader && *leader)
    mPUSHp(leader, 1);
  else
    PUSHs(&PL_sv_undef);

  mPUSHp(&command, 1);

  for(i = 0; i < argcount; i++) {
    AV *av = newAV();
    for( ; i < argcount; i++) {
      av_push(av, CSI_ARG_IS_MISSING(args[i]) ?
        &PL_sv_undef :
        newSViv(CSI_ARG(args[i])));

      if(!CSI_ARG_HAS_MORE(args[i]))
        break;
    }

    mXPUSHs(newRV((SV*)av));
  }

  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int parser_osc(const char *command, size_t cmdlen, void *user)
{
  dSP;
  Term__VTerm self = user;
  CV *cb = self->parser_cb.osc;

  SV *str = newSVpv(command, cmdlen);

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 1);
  mPUSHs(str);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return cmdlen;
}

static int parser_dcs(const char *command, size_t cmdlen, void *user)
{
  dSP;
  Term__VTerm self = user;
  CV *cb = self->parser_cb.dcs;

  SV *str = newSVpv(command, cmdlen);

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 1);
  mPUSHs(str);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return cmdlen;
}

static int parser_resize(int rows, int cols, void *user)
{
  dSP;
  Term__VTerm self = user;
  CV *cb = self->parser_cb.resize;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 2);
  mPUSHi(rows);
  mPUSHi(cols);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;
}

VTermParserCallbacks parser_cbs = {
  .text    = parser_text,
  .control = parser_control,
  .escape  = parser_escape,
  .csi     = parser_csi,
  .osc     = parser_osc,
  .dcs     = parser_dcs,
  .resize  = parser_resize,
};

static int state_putglyph(VTermGlyphInfo *info, VTermPos pos, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.putglyph;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 2);
  mPUSHs(newSVglyphinfo(info));
  mPUSHs(newSVpos(pos));
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_movecursor(VTermPos pos, VTermPos oldpos, int visible, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.movecursor;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 3);
  mPUSHs(newSVpos(pos));
  mPUSHs(newSVpos(oldpos));
  mPUSHi(visible);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_scrollrect(VTermRect rect, int downward, int rightward, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.scrollrect;
  int ret;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 3);
  mPUSHs(newSVrect(rect));
  mPUSHi(downward);
  mPUSHi(rightward);
  PUTBACK;

  call_sv((SV*)cb, G_SCALAR);

  SPAGAIN;

  ret = POPi;

  PUTBACK;
  FREETMPS;
  LEAVE;

  return ret;
}

static int state_moverect(VTermRect dest, VTermRect src, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.moverect;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 2);
  mPUSHs(newSVrect(dest));
  mPUSHs(newSVrect(src));
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_erase(VTermRect rect, int selective, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.erase;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 2);
  mPUSHs(newSVrect(rect));
  mPUSHi(selective);
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_initpen(void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.initpen;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_setpenattr(VTermAttr attr, VTermValue *val, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.setpenattr;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 2);
  mPUSHi(attr);
  mPUSHs(newSVvalue(val, vterm_get_attr_type(attr)));
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_settermprop(VTermProp prop, VTermValue *val, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.settermprop;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 2);
  mPUSHi(prop);
  mPUSHs(newSVvalue(val, vterm_get_prop_type(prop)));
  PUTBACK;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_bell(void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.bell;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  call_sv((SV*)cb, G_VOID);

  FREETMPS;
  LEAVE;

  return 1;
}

static int state_setlineinfo(int row, const VTermLineInfo *info, const VTermLineInfo *oldinfo, void *user)
{
  dSP;
  Term__VTerm__State self = user;
  CV *cb = self->cb.setlineinfo;
  int ret;

  if(!cb)
    return 0;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  EXTEND(SP, 3);
  mPUSHi(row);
  mPUSHs(newSVlineinfo(info));
  mPUSHs(newSVlineinfo(oldinfo));
  PUTBACK;

  call_sv((SV*)cb, G_SCALAR);

  SPAGAIN;

  ret = POPi;

  PUTBACK;
  FREETMPS;
  LEAVE;

  return ret;
}

VTermStateCallbacks state_cbs = {
  .putglyph    = state_putglyph,
  .movecursor  = state_movecursor,
  .scrollrect  = state_scrollrect,
  .moverect    = state_moverect,
  .erase       = state_erase,
  .initpen     = state_initpen,
  .setpenattr  = state_setpenattr,
  .settermprop = state_settermprop,
  .bell        = state_bell,
  .setlineinfo = state_setlineinfo,
};

static void setup_constants(void)
{
  HV *stash  = gv_stashpvn("Term::VTerm", 11, TRUE);
  AV *export = get_av("Term::VTerm::EXPORT_OK", TRUE);

#define DO_CONSTANT(c) \
  newCONSTSUB(stash, #c+6, newSViv(c)); \
  av_push(export, newSVpv(#c+6, 0));

  DO_CONSTANT(VTERM_VALUETYPE_BOOL);
  DO_CONSTANT(VTERM_VALUETYPE_INT);
  DO_CONSTANT(VTERM_VALUETYPE_STRING);
  DO_CONSTANT(VTERM_VALUETYPE_COLOR);

  DO_CONSTANT(VTERM_ATTR_BOLD);
  DO_CONSTANT(VTERM_ATTR_UNDERLINE);
  DO_CONSTANT(VTERM_ATTR_ITALIC);
  DO_CONSTANT(VTERM_ATTR_BLINK);
  DO_CONSTANT(VTERM_ATTR_REVERSE);
  DO_CONSTANT(VTERM_ATTR_STRIKE);
  DO_CONSTANT(VTERM_ATTR_FONT);
  DO_CONSTANT(VTERM_ATTR_FOREGROUND);
  DO_CONSTANT(VTERM_ATTR_BACKGROUND);

  DO_CONSTANT(VTERM_PROP_CURSORVISIBLE);
  DO_CONSTANT(VTERM_PROP_CURSORBLINK);
  DO_CONSTANT(VTERM_PROP_ALTSCREEN);
  DO_CONSTANT(VTERM_PROP_TITLE);
  DO_CONSTANT(VTERM_PROP_ICONNAME);
  DO_CONSTANT(VTERM_PROP_REVERSE);
  DO_CONSTANT(VTERM_PROP_CURSORSHAPE);
  DO_CONSTANT(VTERM_PROP_MOUSE);

  DO_CONSTANT(VTERM_PROP_CURSORSHAPE_BLOCK);
  DO_CONSTANT(VTERM_PROP_CURSORSHAPE_UNDERLINE);
  DO_CONSTANT(VTERM_PROP_CURSORSHAPE_BAR_LEFT);

  DO_CONSTANT(VTERM_PROP_MOUSE_NONE);
  DO_CONSTANT(VTERM_PROP_MOUSE_CLICK);
  DO_CONSTANT(VTERM_PROP_MOUSE_DRAG);
  DO_CONSTANT(VTERM_PROP_MOUSE_MOVE);

  DO_CONSTANT(VTERM_MOD_SHIFT);
  DO_CONSTANT(VTERM_MOD_CTRL);
  DO_CONSTANT(VTERM_MOD_ALT);
}

MODULE = Term::VTerm        PACKAGE = Term::VTerm

Term::VTerm
_new(package,rows,cols)
  char *package
  int   rows
  int   cols
  INIT:
    VTerm *vt;
  CODE:
    vt = vterm_new(rows, cols);
    if(!vt)
      XSRETURN_UNDEF;

    Newxz(RETVAL, 1, struct Term__VTerm);
    RETVAL->vt = vt;

  OUTPUT:
    RETVAL

void
DESTROY(self)
  Term::VTerm self
  INIT:
    struct ParserCallbackData *pcbdata;
  CODE:
    FREE_CB(parser_cb.text);
    FREE_CB(parser_cb.control);
    FREE_CB(parser_cb.escape);
    FREE_CB(parser_cb.csi);
    FREE_CB(parser_cb.osc);
    FREE_CB(parser_cb.dcs);
    FREE_CB(parser_cb.resize);

    vterm_free(self->vt);
    Safefree(self);

void
get_size(self)
  Term::VTerm self
  INIT:
    int rows, cols;
  PPCODE:
    vterm_get_size(self->vt, &rows, &cols);

    EXTEND(SP, 2);
    mPUSHi(rows);
    mPUSHi(cols);
    XSRETURN(2);

void
set_size(self,rows,cols)
  Term::VTerm self
  int         rows
  int         cols
  CODE:
    vterm_set_size(self->vt, rows, cols);

int
get_utf8(self)
  Term::VTerm self
  CODE:
    RETVAL = vterm_get_utf8(self->vt);
  OUTPUT:
    RETVAL

void
set_utf8(self,utf8)
  Term::VTerm self
  int         utf8
  CODE:
    vterm_set_utf8(self->vt, utf8);

size_t
input_write(self,str)
  Term::VTerm  self
  SV          *str
  CODE:
    RETVAL = vterm_input_write(self->vt, SvPVutf8_nolen(str), SvCUR(str));
  OUTPUT:
    RETVAL

size_t
output_read(self,buffer,len)
  Term::VTerm  self
  SV          *buffer
  size_t       len
  CODE:
    sv_grow(buffer, len);
    RETVAL = vterm_output_read(self->vt, SvPVX(buffer), len);
    if(RETVAL > 0) {
      SvPOK_on(buffer);
      SvCUR_set(buffer, RETVAL);
    }
    else
      SvCUR_set(buffer, 0);
  OUTPUT:
    RETVAL

void
keyboard_unichar(self,c,mod=&PL_sv_undef)
  Term::VTerm  self
  int          c
  SV          *mod
  INIT:
    VTermModifier m = 0;
  CODE:
    if(SvOK(mod))
      m = SvIV(mod);
    m &= VTERM_MOD_SHIFT|VTERM_MOD_CTRL|VTERM_MOD_ALT;
    vterm_keyboard_unichar(self->vt, c, m);

void
mouse_move(self,row,col,mod=&PL_sv_undef)
  Term::VTerm  self
  int          row
  int          col
  SV          *mod
  INIT:
    VTermModifier m = 0;
  CODE:
    if(SvOK(mod))
      m = SvIV(mod);
    m &= VTERM_MOD_SHIFT|VTERM_MOD_CTRL|VTERM_MOD_ALT;
    vterm_mouse_move(self->vt, row, col, m);

void
mouse_button(self,button,pressed,mod=&PL_sv_undef)
  Term::VTerm  self
  int          button
  bool         pressed
  SV          *mod
  INIT:
    VTermModifier m = 0;
  CODE:
    if(SvOK(mod))
      m = SvIV(mod);
    m &= VTERM_MOD_SHIFT|VTERM_MOD_CTRL|VTERM_MOD_ALT;
    vterm_mouse_button(self->vt, button, pressed, m);

void
parser_set_callbacks(self,...)
  Term::VTerm  self
  INIT:
    int i;
  CODE:
    vterm_parser_set_callbacks(self->vt, &parser_cbs, self);

    for(i = 1; i < items; i++) {
      char *name = SvPV_nolen(ST(i));
      SV *newcb;
      CV **cvp;
      i++;

      if     (streq(name, "on_text"   )) cvp = &self->parser_cb.text;
      else if(streq(name, "on_control")) cvp = &self->parser_cb.control;
      else if(streq(name, "on_escape"))  cvp = &self->parser_cb.escape;
      else if(streq(name, "on_csi"))     cvp = &self->parser_cb.csi;
      else if(streq(name, "on_osc"))     cvp = &self->parser_cb.osc;
      else if(streq(name, "on_dcs"))     cvp = &self->parser_cb.dcs;
      else if(streq(name, "on_resize"))  cvp = &self->parser_cb.resize;
      else
        croak("Unrecognised parser callback name '%s'", name);

      if(*cvp)
        SvREFCNT_dec(*cvp);

      if(i < items && (newcb = ST(i)) && SvOK(newcb))
        *cvp = (CV *)SvREFCNT_inc(newcb);
      else
        *cvp = NULL;
    }

Term::VTerm::State
obtain_state(self)
  Term::VTerm self
  INIT:
    VTermState *state;
  CODE:
    state = vterm_obtain_state(self->vt);
    if(!state)
      XSRETURN_UNDEF;

    Newxz(RETVAL, 1, struct Term__VTerm__State);
    RETVAL->state = state;
    RETVAL->vterm = SvREFCNT_inc(ST(0));

  OUTPUT:
    RETVAL

int
get_attr_type(attr)
  int attr
  CODE:
    RETVAL = vterm_get_attr_type(attr);
  OUTPUT:
    RETVAL

int
get_prop_type(prop)
  int prop
  CODE:
    RETVAL = vterm_get_prop_type(prop);
  OUTPUT:
    RETVAL


MODULE = Term::VTerm        PACKAGE = Term::VTerm::Color

void
DESTROY(self)
  Term::VTerm::Color self
  CODE:
    Safefree(self);

int
red(self)
  Term::VTerm::Color self
  ALIAS:
    red   = 0
    green = 1
    blue  = 2
  CODE:
    switch(ix) {
      case 0: RETVAL = self->red;   break;
      case 1: RETVAL = self->green; break;
      case 2: RETVAL = self->blue;  break;
    }
  OUTPUT:
    RETVAL


MODULE = Term::VTerm        PACKAGE = Term::VTerm::GlyphInfo

void
DESTROY(self)
  Term::VTerm::GlyphInfo self
  CODE:
    Safefree(self);

void
chars(self)
  Term::VTerm::GlyphInfo self
  INIT:
    int i;
  PPCODE:
    for(i = 0; self->chars[i]; i++)
      mXPUSHi(self->chars[i]);
    XSRETURN(i);

int
width(self)
  Term::VTerm::GlyphInfo self
  ALIAS:
    width = 0
    dhl   = 1
  CODE:
    switch(ix) {
      case 0: RETVAL = self->width; break;
      case 1: RETVAL = self->dhl;   break;
    }
  OUTPUT:
    RETVAL

bool
protected_cell(self)
  Term::VTerm::GlyphInfo self
  ALIAS:
    protected_cell = 0
    dwl            = 1
  CODE:
    switch(ix) {
      case 0: RETVAL = self->protected_cell; break;
      case 1: RETVAL = self->dwl;            break;
    }
  OUTPUT:
    RETVAL


MODULE = Term::VTerm        PACKAGE = Term::VTerm::LineInfo

void
DESTROY(self)
  Term::VTerm::LineInfo self
  CODE:
    Safefree(self);

int
doublewidth(self)
  Term::VTerm::LineInfo self
  ALIAS:
    doublewidth  = 0
    doubleheight = 1
  CODE:
    switch(ix) {
      case 0: RETVAL = self->doublewidth;  break;
      case 1: RETVAL = self->doubleheight; break;
    }
  OUTPUT:
    RETVAL


MODULE = Term::VTerm        PACKAGE = Term::VTerm::Pos

void
DESTROY(self)
  Term::VTerm::Pos self
  CODE:
    Safefree(self);

int
row(self)
  Term::VTerm::Pos self
  CODE:
    RETVAL = self->row;
  OUTPUT:
    RETVAL

int
col(self)
  Term::VTerm::Pos self
  CODE:
    RETVAL = self->col;
  OUTPUT:
    RETVAL


MODULE = Term::VTerm        PACKAGE = Term::VTerm::Rect

void
DESTROY(self)
  Term::VTerm::Rect self
  CODE:
    Safefree(self);

int
start_row(self)
  Term::VTerm::Rect self
  ALIAS:
    start_row = 0
    end_row   = 1
    start_col = 2
    end_col   = 3
  CODE:
    switch(ix) {
      case 0: RETVAL = self->start_row; break;
      case 1: RETVAL = self->end_row;   break;
      case 2: RETVAL = self->start_col; break;
      case 3: RETVAL = self->end_col;   break;
    }
  OUTPUT:
    RETVAL


MODULE = Term::VTerm        PACKAGE = Term::VTerm::State

void
DESTROY(self)
  Term::VTerm::State self
  CODE:
    FREE_CB(cb.putglyph);
    FREE_CB(cb.movecursor);
    FREE_CB(cb.scrollrect);
    FREE_CB(cb.moverect);
    FREE_CB(cb.erase);
    FREE_CB(cb.initpen);
    FREE_CB(cb.setpenattr);
    FREE_CB(cb.settermprop);
    FREE_CB(cb.bell);
    FREE_CB(cb.resize);
    FREE_CB(cb.setlineinfo);

    SvREFCNT_dec(self->vterm);
    Safefree(self);

void
reset(self,hard=&PL_sv_undef)
  Term::VTerm::State  self
  SV                 *hard
  CODE:
    vterm_state_reset(self->state, SvOK(hard) ? SvIV(hard) : 0);

Term::VTerm::Pos
get_cursorpos(self)
  Term::VTerm::State self
  CODE:
    Newx(RETVAL, 1, VTermPos);
    vterm_state_get_cursorpos(self->state, RETVAL);
  OUTPUT:
    RETVAL

void
get_default_colors(self)
  Term::VTerm::State self
  INIT:
    VTermColor fg, bg;
  PPCODE:
    vterm_state_get_default_colors(self->state, &fg, &bg);
    EXTEND(SP, 2);
    mPUSHs(newSVcolor(&fg));
    mPUSHs(newSVcolor(&bg));
    XSRETURN(2);

SV *
get_palette_color(self,index)
  Term::VTerm::State self
  int                index
  INIT:
    VTermColor col;
  CODE:
    vterm_state_get_palette_color(self->state, index, &col);
    RETVAL = newSVcolor(&col);
  OUTPUT:
    RETVAL

SV *
get_penattr(self,attr)
  Term::VTerm::State self
  int                attr
  INIT:
    VTermValue val;
  CODE:
    vterm_state_get_penattr(self->state, attr, &val);
    RETVAL = newSVvalue(&val, vterm_get_attr_type(attr));
  OUTPUT:
    RETVAL

void
set_callbacks(self,...)
  Term::VTerm::State self
  INIT:
    int i;
  CODE:
    vterm_state_set_callbacks(self->state, &state_cbs, self);

    for(i = 1; i < items; i++) {
      char *name = SvPV_nolen(ST(i));
      SV *newcb;
      CV **cvp;
      i++;

      if     (streq(name, "on_putglyph"   )) cvp = &self->cb.putglyph;
      else if(streq(name, "on_movecursor" )) cvp = &self->cb.movecursor;
      else if(streq(name, "on_scrollrect" )) cvp = &self->cb.scrollrect;
      else if(streq(name, "on_moverect"   )) cvp = &self->cb.moverect;
      else if(streq(name, "on_erase"      )) cvp = &self->cb.erase;
      else if(streq(name, "on_initpen"    )) cvp = &self->cb.initpen;
      else if(streq(name, "on_setpenattr" )) cvp = &self->cb.setpenattr;
      else if(streq(name, "on_settermprop")) cvp = &self->cb.settermprop;
      else if(streq(name, "on_bell"       )) cvp = &self->cb.bell;
      else if(streq(name, "on_setlineinfo")) cvp = &self->cb.setlineinfo;
      else
        croak("Unrecognised state callback name '%s'", name);

      if(*cvp)
        SvREFCNT_dec(*cvp);

      if(i < items && (newcb = ST(i)) && SvOK(newcb))
        *cvp = (CV *)SvREFCNT_inc(newcb);
      else
        *cvp = NULL;
    }


BOOT:
  setup_constants();
