package Auth::Kokolores::Plugin::SqlRetrieve;

use Moose;
use DBI;

# ABSTRACT: kokolores plugin for retrieving users from SQL databases
our $VERSION = '1.00'; # VERSION

extends 'Auth::Kokolores::Plugin';


has 'select' => (
  is => 'ro', isa => 'Str',
  default => 'SELECT username, method, salt, cost, hash FROM users WHERE username = ?',
);

has 'user' => ( is => 'ro', isa => 'Str', default => '' );
has 'password' => ( is => 'ro', isa => 'Str', default => '' );

has 'dsn' => (
  is => 'ro', isa => 'Str',
  default => 'dbi:SQLite:dbname=/var/lib/saslauthd/saslauth.sqlite',
);

has 'dbh' => (
  is => 'ro', isa => 'DBI::db',
  lazy => 1,
  default => sub {
    my $self = shift;
    $self->log(1, 'connecting to '.$self->dsn.'...');
    my $dbh = DBI->connect(
      $self->dsn, $self->user, $self->password,
      { RaiseError => 1 },
    );
    return $dbh;
  },
);

has 'sth' => (
  is => 'ro', isa => 'DBI::st', lazy => 1,
  default => sub {
    my $self = shift;
    my $dbh = $self->dbh;
    $self->log(1, 'preparing statement '.$self->select.'...');
    return $dbh->prepare($self->select);
  },
);

sub authenticate {
  my ( $self, $r ) = @_;
  my $sth = $self->sth;
  
  $sth->execute( $r->username );
  my $fields = $sth->fetchrow_hashref;
  $sth->finish;

  if( ! defined $fields ) {
    $r->log(1, 'could not find user '.$r->username);
    return 0;
  }

  foreach my $field ( keys %$fields ) {
    if( ! defined $fields->{$field} ) {
      next;
    }
    $r->log(1, 'retrieved userinfo '.$field.'='.$fields->{$field});
    $r->set_info( $field, $fields->{$field} );
  }

  return 1;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Auth::Kokolores::Plugin::SqlRetrieve - kokolores plugin for retrieving users from SQL databases

=head1 VERSION

version 1.00

=head1 DESCRIPTION

Retrieve a user from a SQL database.

Will fail if no user is found.

If a user is found all fields of the record will be stored in the
current requests userinfo field.

=head1 EXAMPLE

  <Plugin retrieve-user>
    module = "SqlRetrieve"
    dsn = "dbi:SQLite:dbname=/var/lib/kokolores/users.sqlite"
    select = "SELECT * FROM passwd WHERE username = ?"
  </Plugin>

=head1 MODULE PARAMETERS

=head2 select (default: SELECT username, method, salt, cost, hash FROM users WHERE username = ?)

Define a SQL SELECT query to retrieve the user from the SQL database.

=head2 user (default: '')

User used to connect to SQL database.

=head2 password (default: '')

Password used to connect to SQL database.

=head2 dsn (default: 'dbi:SQLite:dbname=/var/lib/saslauthd/saslauth.sqlite')

A connection string for the database in perl-DBI format.

=head1 AUTHOR

Markus Benning <ich@markusbenning.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by Markus Benning <ich@markusbenning.de>.

This is free software, licensed under:

  The GNU General Public License, Version 2, June 1991

=cut
