########################################################################
# housekeeping
########################################################################

use v6.c;

# module & EXPORT close over these.

my $Script;
my $bin_from;

my $real_bin    = False;
my $verbose     = False;

########################################################################
# module makes FindBin::Bin & ::Script available as late calls in 
# in the code available.
########################################################################

module FindBin:ver<0.1.6>:auth<CPAN:lembark>
{
    INIT given $*PROGRAM-NAME
    {
        when '-e' | '-' | 'interactive'
        {
            $Script     = $_.Str;
            $bin_from   = $*CWD;
        }

        default
        {
            # Note: can't take dirname here since the 
            # basename might be the symlink.

            $Script     = $*PROGRAM.basename;
            $bin_from   = $*PROGRAM;
        }
    }

    our sub Script  
    {
        # this cannot change during execution.
        # starts out as a str, no logic necessary.

        $Script
    }

    our sub Bin ( $use_real = $real_bin --> Str )
    {
        # simplest solution: resolve the path as 
        # required and then check for it being a 
        # directory ($*CWD) or file ($*PROGRAM).
        #
        # path has to be resolved prior to dirname
        # since the basename may be our simlink.

        my $path
        = $use_real
        ?? $bin_from.resolve
        !! $bin_from.absolute.IO 
        ;

        if $verbose
        {
            say "# Use real: '$use_real'";
            say "# Bin from: '$bin_from'";
            say "# Path is:  '$path'";
        }

        $path.d
        ?? $path.Str
        !! $path.dirname
    }

    our sub verbose ( Bool $new? --> Bool )
    {
        $new
        // return $verbose;

        $verbose    = $new.Bool
    }
}

########################################################################
# handle the use argumens.
########################################################################

our sub EXPORT( *@argz )
{
    my %varz;
    
    # RealBin sets the defaut in Bin for everyone using
    # the module. This is intentional. used early in the
    # #! code it allows universal use of resolved paths
    # regardless of where else the module is used.
    #
    # process flags first to get consistent effects from
    # $Bin & $Script
    #
    # notice that there is no '!RealBin' and verbosity
    # takes precidence over silence.

    for @argz
    {
        when 'RealBin'  { $real_bin = True  }
        when 'Verbose'  { $verbose  = True  }
    }

    # internal flags are set, export values will
    # handled correctly.
    #
    # no reason &Bin & &Script couldn't have been 
    # handled above, handling them here seems cleaner.

    for @argz
    {
        when '$Bin'     { %varz< $Bin    > =  FindBin::Bin()    }
        when  'Bin'     { %varz< &Bin    > = &FindBin::Bin      }

        when '$Script'  { %varz< $Script > =  FindBin::Script() }
        when  'Script'  { %varz< &Script > = &FindBin::Script   }
    }

    %varz
}

=finish

=begin pod

=head1 SYNOPSIS

    use FindBin <$Bin>;   

    my $dir_from    = $Bin;
    my $script_base = $Script;

    # export the Bin path after resolving any
    # symlinks by default (see argument to Bin,
    # below).

    use FindBin <$Bin RealBin>;

    my $dir_from    = $Bin;
    my $script_base = $Script;

    # export &Bin (vs. $Bin).
    # default to not resolving relative paths.
    
    use FindBin <Bin>;

    my $dir_from    = Bin;

    # default Bin to returning the 
    # resolved path without exporting
    # anything at all.

    use FindBin <RealBin>;

    my $path    = FindBin::Bin();

    # default to using the directory as-is,
    # without resolving any symlinks; override
    # default with argument to Bin call.

    use FindBin <Bin>;

    my $resolved    = Bin( True );
    my $symlinked   = Bin;

    # as above with default of resolving the
    # symlinks.

    use FindBin <Bin RealBin>;

    my $resolved    = Bin;
    my $symlinked   = Bin( False );

    # determine if the current executable is running
    # from a symlinked path.

    use FindBin <Bin>;

    my $resolved    = Bin( False );
    my $program     = Bin( True  );

    $resolved ne $program
    and say "There is a symlink in '$program'";

    # messasges might be useful for dealing with 
    # stray symlinks in filesystem.

    use FindBin <Verbose>;

    # Stdout now gets messages like:
    #
    #   Bin from '/foo/bar/bim/bam.t'
    #   Bin is   '/foo/bar/blort';
    #
    # indicating that bim or bam.t are
    # symlinks.

    FindBin::verbose( True  );  # turn on verbosity
    FindBin::verbose( False );  # turn off verbosity

    my $is-verbose  = FindBin::verbose;

    # verbose returns the value it was set to, makes
    # it easy to pass the value through and get it back
    # in one call.

    my $status      = FindBin::verbose( $^a );


=head1 DESCRIPTION

This module is used to locate the currently running Perl program and
the diretory it is running from. Command-line use of "perl6 -", 
"perl6 -e" or interactive use of the REPL will have a script of 
"-", "-e", or "interactive" and $Bin from $*CWD. Otherwise $Script 
will contain the basename of the running program and $Bin will have 
the direname the program was run from taken from $*PROGRAM.

The option 'RealBin' will return $Bin after procesing with "resolve"
to convert any symbolic links in the path to an absolte path.

The options '$Bin', 'Bin', '$Script', 'Script' will export the 
$Bin or $Script variables or their corresponding subs at startup.

The 'Bin' function takes an optional argument of True or False to
override the default set by RealBin. 


=head2 Notes

=item RealBin RealBin is package flag.

Any module setting will provide a resolved value for all all uses; if 
RealBin needs to be turned on and off during execution then exporting 
Bin and calling it with the appropriate value will be more useful than 
just exporting $Bin at startup.

The module provides no way to alter this after setting it via use.
This is intentional in order to avoid used modules overriding the
#! code's view of the world.

=item Tests use symlinks

To see how symlinks are handled do an "ls -l" on ./t and look at
the tests symlinked to ./symlinks or ./bin.


=head1 SEE ALSO

Perl variables $*PROGRAM, $*PROGRAM-NAME:

    https://docs.perl6.org/language/variables#index-entry-%24%2APROGRAM

Class which implements the dirname, basename, and absolute methods 
use to determine the absolute path of $Bin.

    https://docs.perl6.org/type/IO::Path
    
=head1 AUTHOR

Steven Lembark <lembark@wrkhors.com>

=head1 COPYRIGHT AND LICENSE

Copyright 2018 lembark

This library is free software; you can redistribute it and/or modify 
it under the Artistic License 2.0.

=end pod

