# ABSTRACT: Data::Object Prototype-based Programming
package Data::Object::Prototype;

use 5.10.0;

use strict;
use warnings;

use Carp qw(croak);
use Clone qw(clone);
use Data::Object qw(deduce_type);
use Scalar::Util qw(blessed);

our $VERSION = '0.02'; # VERSION

sub import {
    my $class  = shift;
    my $target = caller;

    no strict 'refs';

    *{"${target}::extend"} = $class->can('build_clone');
    *{"${target}::object"} = $class->can('build_object');

    return;
}

my $serial = 0;
sub build_class (@) {
    my $type = shift;
    my $base = shift;

    my $mappings = {
        'ARRAY'     =>  'Data::Object::Array',
        'HASH'      =>  'Data::Object::Hash',
        'CODE'      =>  'Data::Object::Code',
        'FLOAT'     =>  'Data::Object::Float',
        'NUMBER'    =>  'Data::Object::Number',
        'INTEGER'   =>  'Data::Object::Integer',
        'STRING'    =>  'Data::Object::String',
        'SCALAR'    =>  'Data::Object::Scalar',
        'REGEXP'    =>  'Data::Object::Regexp',
        'UNDEF'     =>  'Data::Object::Undef',
        'UNIVERSAL' =>  'Data::Object::Universal',
    };

    my $class = join '::', __PACKAGE__, 'Instance';

    $type = $mappings->{$type} // 'Data::Object::Universal';
    $base = $class unless $base;

    my $format  = '%s::__ANON__::%04d';
    my $package = sprintf $format, $class, ++$serial;
    my @supers  = ("use base '$type'", "use base '$base'");

    eval join '; ', ("package $package", @supers);

    croak $@ if $@;

    return $package;
}

sub build_clone (@) {
    my $class = shift;
    my $args  = shift;

    $args = clone $class->data if not defined $args && ref $class;

    my $type  = deduce_type $class;
    my $clone = build_class $type, ref($class) || $class;

    $args = blessed $args ? $args->data : $args;

    return $clone->new($args);
}

sub build_object (@) {
    my $args = shift;

    my $type  = deduce_type $args;
    my $class = build_class $type;

    $args = blessed $args ? $args->data : $args;

    return $class->new($args);
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Data::Object::Prototype - Data::Object Prototype-based Programming

=head1 VERSION

version 0.02

=head1 SYNOPSIS

    use Data::Object::Prototype;

    my $bear = object {
        name     => 'bear',
        type     => 'black bear',
        attitude => 'indifferent',
    };

    $bear->package->install(responds => sub {
        'Roarrrr'
    });

    $bear->package->install(succeeds => sub {
        shift->isa(ref shift)
    });

    my $papa = extend $bear => {
        name     => 'papa bear',
        type     => 'great big papa bear',
        attitude => 'agitated',
    };

    $papa->package->install(responds => sub {
        "Who's been eating my porridge?"
    });

    my $baby = extend $papa => {
        name     => 'baby bear',
        type     => 'tiny little baby bear',
        attitude => 'baby',
    };

    $baby->package->install(responds => sub {
        "Who's eaten up all my porridge?"
    });

    my $mama = extend $bear => {
        name     => 'mama bear',
        type     => 'middle-sized mama bear',
        attitude => 'confused',
    };

    $mama->package->install(responds => sub {
        "Who's been eating my porridge?"
    });

    if ($papa && $mama && $baby && $baby->succeeds($papa)) {
        my $statement = "The %s said, '%s'\n";

        printf $statement, $papa->get('name'), $papa->responds;
        printf $statement, $mama->get('name'), $mama->responds;
        printf $statement, $baby->get('name'), $baby->responds;

        # The Papa Bear said, "Who's been eating my porridge?"
        # The Mama Bear said, "Who's been eating my porridge?"
        # The Baby Bear said, "Who's eaten up all my porridge?"
    }

=head1 DESCRIPTION

Data::Object::Prototype implements a thin prototype-like layer on top of the
L<Data::Object> type-object framework. This module allows you to develop using
a prototype-based style in Perl, giving you the ability to create, mutate,
extend, mixin, and destroy anonymous Data::Object type classes, ad hoc and with
very little code.

Prototype-based programming is a style of object-oriented programming in which
classes are not present, and behavior reuse (known as inheritance in class-based
languages) is performed via a process of cloning existing objects that serve as
prototypes. Due to familiarity with class-based languages such as Java, many
programmers assume that object-oriented programming is synonymous with
class-based programming.

However, class-based programming is just one kind of object-oriented programming
style, and other varieties exist such as role-oriented, aspect-oriented and
prototype-based programming. A prominent example of a prototype-based
programming language is ECMAScript (a.k.a. JavaScript or JScript). B<Note: This
is an early release available for testing and feedback and as such is subject to
change.>

=head1 AUTHOR

Al Newkirk <anewkirk@ana.io>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Al Newkirk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
