#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "ppport.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <assert.h>
#include <inttypes.h>
#include <stdint.h>

#include "spvm_compiler.h"
#include "spvm_hash.h"
#include "spvm_list.h"
#include "spvm_util_allocator.h"
#include "spvm_runtime.h"
#include "spvm_op.h"
#include "spvm_sub.h"
#include "spvm_package.h"
#include "spvm_sub.h"
#include "spvm_my.h"
#include "spvm_type.h"
#include "spvm_field.h"
#include "spvm_object.h"
#include "spvm_api.h"
#include "spvm_opcode_builder.h"
#include "spvm_jitcode_builder.h"
#include "spvm_list.h"
#include "spvm_jitcode_builder.h"
#include "spvm_string_buffer.h"

static SPVM_API_VALUE call_sub_args[255];

SPVM_API* SPVM_XS_UTIL_get_api() {
  
  SV* sv_api = get_sv("SPVM::API", 0);
  
  SPVM_API* api = (SPVM_API*)SvIV(SvRV(sv_api));
  
  return api;
}

SV* SPVM_XS_UTIL_new_sv_object(SPVM_OBJECT* object, const char* package) {
  // Create object
  size_t iv_object = PTR2IV(object);
  SV* sviv_object = sv_2mortal(newSViv(iv_object));
  SV* sv_object = sv_2mortal(newRV_inc(sviv_object));
  HV* hv_class = gv_stashpv(package, 0);
  sv_bless(sv_object, hv_class);
  
  return sv_object;
}

SPVM_OBJECT* SPVM_XS_UTIL_get_object(SV* sv_object) {
  
  if (SvOK(sv_object)) {
    size_t iv_object = SvIV(SvRV(sv_object));
    SPVM_OBJECT* object = INT2PTR(SPVM_OBJECT*, iv_object);
    
    return object;
  }
  else {
    return NULL;
  }
}

int SPVM_XS_UTIL_compile_jit_sub(SPVM_API* api, int32_t sub_id) {
  dSP;
  
  (void)api;

  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  SPVM_OP* op_sub = SPVM_LIST_fetch(compiler->op_subs, sub_id);
  SPVM_SUB* sub = op_sub->uv.sub;
  
  if (sub->is_jit || sub->disable_jit) {
    return 1;
  }
  
  // String buffer for jitcode
  SPVM_STRING_BUFFER* string_buffer = SPVM_STRING_BUFFER_new(0);
  
  // Build sub jitcode
  SPVM_JITCODE_BUILDER_build_sub_jitcode(string_buffer, sub_id);
  
  SV* sv_jitcode_source = sv_2mortal(newSVpv(string_buffer->buffer, string_buffer->length));
  
  SV* sv_sub_id = sv_2mortal(newSViv(sub_id));
  
  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  XPUSHs(sv_sub_id);
  XPUSHs(sv_jitcode_source);
  PUTBACK;

  call_pv("SPVM::compile_jit_sub", G_SCALAR);

  SPAGAIN;

  int32_t success = POPi;

  PUTBACK;
  FREETMPS;
  LEAVE;
  
  SPVM_STRING_BUFFER_free(string_buffer);
  
  return success;
}

MODULE = SPVM::Core::Object		PACKAGE = SPVM::Core::Object

SV*
DESTROY(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_object = ST(0);
  
  assert(SvOK(sv_object));
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* object = SPVM_XS_UTIL_get_object(sv_object);
  
  assert(api->get_ref_count(api, object));
  
  // Decrement reference count
  api->dec_ref_count(api, object);
  
  XSRETURN(0);
}

MODULE = SPVM::Core::Object::Package::String		PACKAGE = SPVM::Core::Object::Package::String

SV*
new_string(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_chars = ST(1);
  int32_t length = sv_len(sv_chars);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // New string
  SPVM_API_OBJECT* string =  api->new_string(api, SvPV_nolen(sv_chars), length);
  
  // Increment reference count
  api->inc_ref_count(api, string);
  
  // New sv string
  SV* sv_string = SPVM_XS_UTIL_new_sv_object(string, "SPVM::Core::Object::Package::String");
  
  XPUSHs(sv_string);
  XSRETURN(1);
}

SV*
to_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_string = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* string = SPVM_XS_UTIL_get_object(sv_string);
  
  int32_t string_length = api->get_string_length(api, string);
  
  char* chars = api->get_string_chars(api, string);
  
  SV* sv_data = sv_2mortal(newSVpvn(chars, string_length));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array::Byte		PACKAGE = SPVM::Core::Object::Array::Byte

SV*
new_len(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_length = ST(1);
  
  int32_t length = (int32_t)SvIV(sv_length);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // New array
  SPVM_API_OBJECT* array =  api->new_byte_array(api, length);
  
  // Increment reference count
  api->inc_ref_count(api, array);
  
  // New sv array
  SV* sv_byte_array = SPVM_XS_UTIL_new_sv_object(array, "SPVM::Core::Object::Array::Byte");
  
  XPUSHs(sv_byte_array);
  XSRETURN(1);
}

SV*
set_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_values = ST(1);
  
  if (!(SvROK(sv_values) && sv_derived_from(sv_values, "ARRAY"))) {
    croak("Values must be array refenrece(SPVM::Core::Object::Array::Byte::set_elements())");
  }
  
  AV* av_values = (AV*)SvRV(sv_values);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int8_t* elements = api->get_byte_array_elements(api, array);

  // Check length
  if (av_len(av_values) + 1 != length) {
    croak("Elements length must be same as array length(SPVM::Core::Object::Array::Byte::set_elements())");
  }
  
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[i] = (int8_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_values = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Byte::set_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Byte::set_elements_range())");
  }
  
  // Check if sv values is array reference
  if (!(SvROK(sv_values) && sv_derived_from(sv_values, "ARRAY"))) {
    croak("Values must be array refenrece(SPVM::Core::Object::Array::Byte::set_elements_range())");
  }
  
  AV* av_values = (AV*)SvRV(sv_values);
  
  // Check elements length
  if (av_len(av_values) + 1 != count) {
    croak("Elements length must be same as count argument(SPVM::Core::Object::Array::Byte::set_elements_range())");
  }
  
  // Elements
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  // Set element value
  {
    int32_t i;
    
    for (i = 0; i < count; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[index + i] = (int8_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_data = ST(1);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  // Check range
  if ((int32_t)sv_len(sv_data) != length) {
    croak("Data total byte size must be same as array length(SPVM::Core::Object::Array::Byte::set_data())");
  }
  
  if (length > 0) {
    memcpy(elements, SvPV_nolen(sv_data), length);
  }
  
  XSRETURN(0);
}

SV*
set_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_data = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Byte::set_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Byte::set_data_range())");
  }
  
  // Check data byte size
  int32_t data_byte_size = (int32_t)sv_len(sv_data);
  
  if (data_byte_size != count) {
    croak("Data byte size must be same as count argument(SPVM::Core::Object::Array::Byte::set_data_range())");
  }
  
  // Elements
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  // Copy data
  if (count > 0) {
    memcpy(elements + index, SvPV_nolen(sv_data), count);
  }
  
  XSRETURN(0);
}

SV*
set(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_value = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Byte::set())");
  }
  
  // Value
  int8_t value = (int8_t)SvIV(sv_value);
  
  // Set element
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  elements[index] = value;
  
  XSRETURN(0);
}

SV*
get(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Byte::set())");
  }
  
  // Get element
  int8_t* elements = api->get_byte_array_elements(api, array);
  int8_t value = elements[index];
  SV* sv_value = sv_2mortal(newSViv(value));
  
  XPUSHs(sv_value);
  XSRETURN(1);
}

SV*
get_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
get_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Byte::get_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Byte::get_elements_range())");
  }
  
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = index; i < index + count; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
to_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)elements, length));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

SV*
to_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Byte::to_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Byte::to_data_range())");
  }
  
  int8_t* elements = api->get_byte_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)(elements + index), count));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array::Short		PACKAGE = SPVM::Core::Object::Array::Short

SV*
new_len(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_length = ST(1);
  
  int32_t length = (int32_t)SvIV(sv_length);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // New array
  SPVM_API_OBJECT* array =  api->new_short_array(api, length);
  
  // Increment reference count
  api->inc_ref_count(api, array);
  
  // New sv array
  SV* sv_array = SPVM_XS_UTIL_new_sv_object(array, "SPVM::Core::Object::Array::Short");
  
  XPUSHs(sv_array);
  XSRETURN(1);
}

SV*
set_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_values = ST(1);
  AV* av_values = (AV*)SvRV(sv_values);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int16_t* elements = api->get_short_array_elements(api, array);

  // Check range
  if (av_len(av_values) + 1 != length) {
    croak("Elements length must be same as array length(SPVM::Core::Object::Array::Short::set_elements())");
  }
  
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[i] = (int16_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_values = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Short::set_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Short::set_elements_range())");
  }
  
  // Check if sv values is array reference
  if (!(SvROK(sv_values) && sv_derived_from(sv_values, "ARRAY"))) {
    croak("Values must be array refenrece(SPVM::Core::Object::Array::Short::set_elements_range())");
  }
  
  AV* av_values = (AV*)SvRV(sv_values);
  
  // Check elements length
  if (av_len(av_values) + 1 != count) {
    croak("Elements length must be same as count argument(SPVM::Core::Object::Array::Short::set_elements_range())");
  }
  
  // Elements
  int16_t* elements = api->get_short_array_elements(api, array);
  
  // Set element value
  {
    int32_t i;
    
    for (i = 0; i < count; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[index + i] = (int16_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_data = ST(1);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int16_t* elements = api->get_short_array_elements(api, array);
  
  // Check range
  if ((int32_t)sv_len(sv_data) != length * 2) {
    croak("Data total byte size must be same as array length * 2(SPVM::Core::Object::Array::Short::set_data())");
  }
  
  if (length > 0) {
    memcpy(elements, SvPV_nolen(sv_data), length * 2);
  }
  
  XSRETURN(0);
}

SV*
set_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_data = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Short::set_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Short::set_data_range())");
  }
  
  // Check data short size
  int32_t data_short_size = (int32_t)sv_len(sv_data);
  
  if (data_short_size != count * 2) {
    croak("Data byte size must be same as count argument * 2(SPVM::Core::Object::Array::Short::set_data_range())");
  }
  
  // Elements
  int16_t* elements = api->get_short_array_elements(api, array);
  
  // Copy data
  if (count > 0) {
    memcpy(elements + index, SvPV_nolen(sv_data), count * 2);
  }
  
  XSRETURN(0);
}

SV*
set(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_value = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Short::set())");
  }
  
  // Value
  int16_t value = (int16_t)SvIV(sv_value);
  
  // Set element
  int16_t* elements = api->get_short_array_elements(api, array);
  
  elements[index] = value;
  
  XSRETURN(0);
}

SV*
get(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Short::set())");
  }
  
  // Get element
  int16_t* elements = api->get_short_array_elements(api, array);
  int16_t value = elements[index];
  SV* sv_value = sv_2mortal(newSViv(value));
  
  XPUSHs(sv_value);
  XSRETURN(1);
}

SV*
get_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int16_t* elements = api->get_short_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
get_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Short::get_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Short::get_elements_range())");
  }
  
  int16_t* elements = api->get_short_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = index; i < index + count; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
to_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int16_t* elements = api->get_short_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)elements, length * 2));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

SV*
to_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Short::to_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Short::to_data_range())");
  }
  
  int16_t* elements = api->get_short_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)(elements + index), count * 2));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array::Int		PACKAGE = SPVM::Core::Object::Array::Int

SV*
new_len(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_length = ST(1);
  
  int32_t length = (int32_t)SvIV(sv_length);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // New array
  SPVM_API_OBJECT* array =  api->new_int_array(api, length);

  // Increment reference count
  api->inc_ref_count(api, array);
  
  // New sv array
  SV* sv_array = SPVM_XS_UTIL_new_sv_object(array, "SPVM::Core::Object::Array::Int");
  
  XPUSHs(sv_array);
  XSRETURN(1);
}

SV*
set_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_values = ST(1);
  AV* av_values = (AV*)SvRV(sv_values);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int32_t* elements = api->get_int_array_elements(api, array);

  // Check range
  if (av_len(av_values) + 1 != length) {
    croak("Elements length must be same as array length(SPVM::Core::Object::Array::Int::set_elements())");
  }
  
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[i] = (int32_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_values = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Int::set_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Int::set_elements_range())");
  }
  
  // Check if sv values is array reference
  if (!(SvROK(sv_values) && sv_derived_from(sv_values, "ARRAY"))) {
    croak("Values must be array refenrece(SPVM::Core::Object::Array::Int::set_elements_range())");
  }
  
  AV* av_values = (AV*)SvRV(sv_values);
  
  // Check elements length
  if (av_len(av_values) + 1 != count) {
    croak("Elements length must be same as count argument(SPVM::Core::Object::Array::Int::set_elements_range())");
  }
  
  // Elements
  int32_t* elements = api->get_int_array_elements(api, array);
  
  // Set element value
  {
    int32_t i;
    
    for (i = 0; i < count; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[index + i] = (int32_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_data = ST(1);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int32_t* elements = api->get_int_array_elements(api, array);
  
  // Check range
  if ((int32_t)sv_len(sv_data) != length * 4) {
    croak("Data total byte size must be same as array length * 4(SPVM::Core::Object::Array::Int::set_data())");
  }
  
  if (length > 0) {
    memcpy(elements, SvPV_nolen(sv_data), length * 4);
  }
  
  XSRETURN(0);
}

SV*
set_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_data = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Int::set_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Int::set_data_range())");
  }
  
  // Check data int size
  int32_t data_int_size = (int32_t)sv_len(sv_data);
  
  if (data_int_size != count * 4) {
    croak("Data byte size must be same as count argument * 4(SPVM::Core::Object::Array::Int::set_data_range())");
  }
  
  // Elements
  int32_t* elements = api->get_int_array_elements(api, array);
  
  // Copy data
  if (count > 0) {
    memcpy(elements + index, SvPV_nolen(sv_data), count * 4);
  }
  
  XSRETURN(0);
}

SV*
set(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_value = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Int::set())");
  }
  
  // Value
  int32_t value = (int32_t)SvIV(sv_value);
  
  // Set element
  int32_t* elements = api->get_int_array_elements(api, array);
  
  elements[index] = value;
  
  XSRETURN(0);
}

SV*
get(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Int::set())");
  }
  
  // Get element
  int32_t* elements = api->get_int_array_elements(api, array);
  int32_t value = elements[index];
  SV* sv_value = sv_2mortal(newSViv(value));
  
  XPUSHs(sv_value);
  XSRETURN(1);
}

SV*
get_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int32_t* elements = api->get_int_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
get_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Int::get_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Int::get_elements_range())");
  }
  
  int32_t* elements = api->get_int_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = index; i < index + count; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
to_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int32_t* elements = api->get_int_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)elements, length * 4));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

SV*
to_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Int::to_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Int::to_data_range())");
  }
  
  int32_t* elements = api->get_int_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)(elements + index), count * 4));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array::Long		PACKAGE = SPVM::Core::Object::Array::Long

SV*
new_len(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_length = ST(1);
  
  int32_t length = (int32_t)SvIV(sv_length);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // New array
  SPVM_API_OBJECT* array =  api->new_long_array(api, length);
  
  // Increment reference count
  api->inc_ref_count(api, array);
  
  // New sv array
  SV* sv_array = SPVM_XS_UTIL_new_sv_object(array, "SPVM::Core::Object::Array::Long");
  
  XPUSHs(sv_array);
  XSRETURN(1);
}

SV*
set_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_values = ST(1);
  AV* av_values = (AV*)SvRV(sv_values);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int64_t* elements = api->get_long_array_elements(api, array);

  // Check range
  if (av_len(av_values) + 1 != length) {
    croak("Elements length must be same as array length(SPVM::Core::Object::Array::Long::set_elements())");
  }
  
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[i] = (int64_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_values = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Long::set_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Long::set_elements_range())");
  }
  
  // Check if sv values is array reference
  if (!(SvROK(sv_values) && sv_derived_from(sv_values, "ARRAY"))) {
    croak("Values must be array refenrece(SPVM::Core::Object::Array::Long::set_elements_range())");
  }
  
  AV* av_values = (AV*)SvRV(sv_values);
  
  // Check elements length
  if (av_len(av_values) + 1 != count) {
    croak("Elements length must be same as count argument(SPVM::Core::Object::Array::Long::set_elements_range())");
  }
  
  // Elements
  int64_t* elements = api->get_long_array_elements(api, array);
  
  // Set element value
  {
    int32_t i;
    
    for (i = 0; i < count; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[index + i] = (int64_t)SvIV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_data = ST(1);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();

  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int64_t* elements = api->get_long_array_elements(api, array);
  
  // Check range
  if ((int32_t)sv_len(sv_data) != length * 8) {
    croak("Data total byte size must be same as array length * 8(SPVM::Core::Object::Array::Long::set_data())");
  }
  
  if (length > 0) {
    memcpy(elements, SvPV_nolen(sv_data), length * 8);
  }
  
  XSRETURN(0);
}

SV*
set_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_data = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Long::set_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Long::set_data_range())");
  }
  
  // Check data long size
  int32_t data_long_size = (int32_t)sv_len(sv_data);
  
  if (data_long_size != count * 8) {
    croak("Data byte size must be same as count argument * 8(SPVM::Core::Object::Array::Long::set_data_range())");
  }
  
  // Elements
  int64_t* elements = api->get_long_array_elements(api, array);
  
  // Copy data
  if (count > 0) {
    memcpy(elements + index, SvPV_nolen(sv_data), count * 8);
  }
  
  XSRETURN(0);
}

SV*
set(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_value = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Long::set())");
  }
  
  // Value
  int64_t value = (int64_t)SvIV(sv_value);
  
  // Set element
  int64_t* elements = api->get_long_array_elements(api, array);
  
  elements[index] = value;
  
  XSRETURN(0);
}

SV*
get(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Long::set())");
  }
  
  // Get element
  int64_t* elements = api->get_long_array_elements(api, array);
  int64_t value = elements[index];
  SV* sv_value = sv_2mortal(newSViv(value));
  
  XPUSHs(sv_value);
  XSRETURN(1);
}

SV*
get_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int64_t* elements = api->get_long_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
get_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Long::get_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Long::get_elements_range())");
  }
  
  int64_t* elements = api->get_long_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = index; i < index + count; i++) {
      SV* sv_value = sv_2mortal(newSViv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
to_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  int64_t* elements = api->get_long_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)elements, length * 8));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

SV*
to_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Long::to_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Long::to_data_range())");
  }
  
  int64_t* elements = api->get_long_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)(elements + index), count * 8));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array::Float		PACKAGE = SPVM::Core::Object::Array::Float

SV*
new_len(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_length = ST(1);
  
  int32_t length = (int32_t)SvIV(sv_length);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // New array
  SPVM_API_OBJECT* array =  api->new_float_array(api, length);
  
  // Increment reference count
  api->inc_ref_count(api, array);
  
  // New sv array
  SV* sv_array = SPVM_XS_UTIL_new_sv_object(array, "SPVM::Core::Object::Array::Float");
  
  XPUSHs(sv_array);
  XSRETURN(1);
}

SV*
set_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_values = ST(1);
  AV* av_values = (AV*)SvRV(sv_values);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  float* elements = api->get_float_array_elements(api, array);

  // Check range
  if (av_len(av_values) + 1 != length) {
    croak("Elements length must be same as array length(SPVM::Core::Object::Array::Float::set_elements())");
  }
  
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[i] = (float)SvNV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_values = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Float::set_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Float::set_elements_range())");
  }
  
  // Check if sv values is array reference
  if (!(SvROK(sv_values) && sv_derived_from(sv_values, "ARRAY"))) {
    croak("Values must be array refenrece(SPVM::Core::Object::Array::Float::set_elements_range())");
  }
  
  AV* av_values = (AV*)SvRV(sv_values);
  
  // Check elements length
  if (av_len(av_values) + 1 != count) {
    croak("Elements length must be same as count argument(SPVM::Core::Object::Array::Float::set_elements_range())");
  }
  
  // Elements
  float* elements = api->get_float_array_elements(api, array);
  
  // Set element value
  {
    int32_t i;
    
    for (i = 0; i < count; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[index + i] = (float)SvNV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_data = ST(1);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();

  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  float* elements = api->get_float_array_elements(api, array);
  
  // Check range
  if ((int32_t)sv_len(sv_data) != length * 4) {
    croak("Data total byte size must be same as array length * 4(SPVM::Core::Object::Array::Float::set_data())");
  }
  
  if (length > 0) {
    memcpy(elements, SvPV_nolen(sv_data), length * 4);
  }
  
  XSRETURN(0);
}

SV*
set_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_data = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Float::set_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Float::set_data_range())");
  }
  
  // Check data float size
  int32_t data_float_size = (int32_t)sv_len(sv_data);
  
  if (data_float_size != count * 4) {
    croak("Data byte size must be same as count argument * 4(SPVM::Core::Object::Array::Float::set_data_range())");
  }
  
  // Elements
  float* elements = api->get_float_array_elements(api, array);
  
  // Copy data
  if (count > 0) {
    memcpy(elements + index, SvPV_nolen(sv_data), count * 4);
  }
  
  XSRETURN(0);
}

SV*
set(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_value = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Float::set())");
  }
  
  // Value
  float value = (float)SvNV(sv_value);
  
  // Set element
  float* elements = api->get_float_array_elements(api, array);
  
  elements[index] = value;
  
  XSRETURN(0);
}

SV*
get(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Float::set())");
  }
  
  // Get element
  float* elements = api->get_float_array_elements(api, array);
  float value = elements[index];
  SV* sv_value = sv_2mortal(newSVnv(value));
  
  XPUSHs(sv_value);
  XSRETURN(1);
}

SV*
get_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  float* elements = api->get_float_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV* sv_value = sv_2mortal(newSVnv((NV)elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
get_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Float::get_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Float::get_elements_range())");
  }
  
  float* elements = api->get_float_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = index; i < index + count; i++) {
      SV* sv_value = sv_2mortal(newSVnv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
to_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  float* elements = api->get_float_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)elements, length * 4));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

SV*
to_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Float::to_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Float::to_data_range())");
  }
  
  float* elements = api->get_float_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)(elements + index), count * 4));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array::Double		PACKAGE = SPVM::Core::Object::Array::Double

SV*
new_len(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_length = ST(1);
  
  int32_t length = (int32_t)SvIV(sv_length);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // New array
  SPVM_API_OBJECT* array =  api->new_double_array(api, length);
  
  // Increment reference count
  api->inc_ref_count(api, array);
  
  // New sv array
  SV* sv_array = SPVM_XS_UTIL_new_sv_object(array, "SPVM::Core::Object::Array::Double");
  
  XPUSHs(sv_array);
  XSRETURN(1);
}

SV*
set_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_values = ST(1);
  AV* av_values = (AV*)SvRV(sv_values);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  double* elements = api->get_double_array_elements(api, array);

  // Check range
  if (av_len(av_values) + 1 != length) {
    croak("Elements length must be same as array length(SPVM::Core::Object::Array::Double::set_elements())");
  }
  
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[i] = (double)SvNV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_values = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Double::set_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Double::set_elements_range())");
  }
  
  // Check if sv values is array reference
  if (!(SvROK(sv_values) && sv_derived_from(sv_values, "ARRAY"))) {
    croak("Values must be array refenrece(SPVM::Core::Object::Array::Double::set_elements_range())");
  }
  
  AV* av_values = (AV*)SvRV(sv_values);
  
  // Check elements length
  if (av_len(av_values) + 1 != count) {
    croak("Elements length must be same as count argument(SPVM::Core::Object::Array::Double::set_elements_range())");
  }
  
  // Elements
  double* elements = api->get_double_array_elements(api, array);
  
  // Set element value
  {
    int32_t i;
    
    for (i = 0; i < count; i++) {
      SV** sv_value_ptr = av_fetch(av_values, i, 0);
      SV* sv_value = sv_value_ptr ? *sv_value_ptr : &PL_sv_undef;
      elements[index + i] = (double)SvNV(sv_value);
    }
  }
  
  XSRETURN(0);
}

SV*
set_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_data = ST(1);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();

  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  double* elements = api->get_double_array_elements(api, array);
  
  // Check range
  if ((int32_t)sv_len(sv_data) != length * 8) {
    croak("Data total byte size must be same as array length * 8(SPVM::Core::Object::Array::Double::set_data())");
  }
  
  if (length > 0) {
    memcpy(elements, SvPV_nolen(sv_data), length * 8);
  }
  
  XSRETURN(0);
}

SV*
set_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  SV* sv_data = ST(3);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Double::set_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Double::set_data_range())");
  }
  
  // Check data double size
  int32_t data_double_size = (int32_t)sv_len(sv_data);
  
  if (data_double_size != count * 8) {
    croak("Data byte size must be same as count argument * 8(SPVM::Core::Object::Array::Double::set_data_range())");
  }
  
  // Elements
  double* elements = api->get_double_array_elements(api, array);
  
  // Copy data
  if (count > 0) {
    memcpy(elements + index, SvPV_nolen(sv_data), count * 8);
  }
  
  XSRETURN(0);
}

SV*
set(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_value = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Double::set())");
  }
  
  // Value
  double value = (double)SvNV(sv_value);
  
  // Set element
  double* elements = api->get_double_array_elements(api, array);
  
  elements[index] = value;
  
  XSRETURN(0);
}

SV*
get(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);

  // Array
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check range
  if (index < 0 || index > length - 1) {
    croak("Out of range(SPVM::Core::Object::Array::Double::set())");
  }
  
  // Get element
  double* elements = api->get_double_array_elements(api, array);
  double value = elements[index];
  SV* sv_value = sv_2mortal(newSVnv(value));
  
  XPUSHs(sv_value);
  XSRETURN(1);
}

SV*
get_elements(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  double* elements = api->get_double_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = 0; i < length; i++) {
      SV* sv_value = sv_2mortal(newSVnv((NV)elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
get_elements_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Double::get_elements_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Double::get_elements_range())");
  }
  
  double* elements = api->get_double_array_elements(api, array);
  
  AV* av_values = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t i;
    for (i = index; i < index + count; i++) {
      SV* sv_value = sv_2mortal(newSVnv(elements[i]));
      av_push(av_values, SvREFCNT_inc(sv_value));
    }
  }
  SV* sv_values = sv_2mortal(newRV_inc((SV*)av_values));
  
  XPUSHs(sv_values);
  XSRETURN(1);
}

SV*
to_data(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);

  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  int32_t length = api->get_array_length(api, array);
  
  double* elements = api->get_double_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)elements, length * 8));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

SV*
to_data_range(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_count = ST(2);
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  // Count
  int32_t count = (int32_t)SvIV(sv_count);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get object
  SPVM_API_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Length
  int32_t length = api->get_array_length(api, array);
  
  // Check index
  if (index < 0 || index > length - 1) {
    croak("Index is out of range(SPVM::Core::Object::Array::Double::to_data_range())");
  }
  
  // Check count
  if (count < 0 || index + count > length - 1) {
    croak("Index + count is out of range(SPVM::Core::Object::Array::Double::to_data_range())");
  }
  
  double* elements = api->get_double_array_elements(api, array);
  
  SV* sv_data = sv_2mortal(newSVpvn((char*)(elements + index), count * 8));
  
  XPUSHs(sv_data);
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array::Object		PACKAGE = SPVM::Core::Object::Array::Object

SV*
new_len(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_class = ST(0);
  (void)sv_class;
  
  SV* sv_element_type_name = ST(1);
  SV* sv_length = ST(2);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  int32_t length = (int32_t)SvIV(sv_length);
  
  // Element type id
  const char* element_type_name = SvPV_nolen(sv_element_type_name);
  int32_t element_type_id = api->get_type_id(api, element_type_name);
  
  // New array
  SPVM_API_OBJECT* array =  api->new_object_array(api, element_type_id, length);
  
  // Fix type name(int[] -> int[][]);
  SV* sv_type_name = sv_2mortal(newSVsv(sv_element_type_name));
  sv_catpv(sv_type_name, "[]");
  
  // Type id
  const char* type_name = SvPV_nolen(sv_type_name);
  
  int32_t type_id = api->get_type_id(api, type_name);
  
  if (type_id < 0) {
    croak("Unknown type %s. Type must be used in SPVM module at least one(SPVM::Core::Object::Array::Object::new())", type_name);
  }
  if (type_id >= SPVM_TYPE_C_ID_BYTE && type_id <= SPVM_TYPE_C_ID_DOUBLE) {
    croak("Type is not object array %s(SPVM::Core::Object::Array::Object::new())", type_name);
  }
  
  // Increment reference count
  api->inc_ref_count(api, array);
  
  // New sv array
  SV* sv_array = SPVM_XS_UTIL_new_sv_object(array, "SPVM::Core::Object::Array::Object");
  
  XPUSHs(sv_array);
  XSRETURN(1);
}

SV*
set(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  SV* sv_object = ST(2);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Get array
  SPVM_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Runtime
  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  // Array type id
  int32_t array_type_id = array->type_id;
  
  // Array type
  SPVM_TYPE* array_type = SPVM_LIST_fetch(compiler->types, array_type_id);

  // Get object
  SPVM_OBJECT* object = SPVM_XS_UTIL_get_object(sv_object);
  
  // Object type id
  int32_t object_type_id = object->type_id;

  // Object type
  SPVM_TYPE* object_type = SPVM_LIST_fetch(compiler->types, object_type_id);

  if (strncmp(array_type->name, object_type->name, strlen(array_type->name - 2)) != 0) {
    croak("Invalid type %s is set to object array %s(SPVM::Core::Object::Array::Object::set())", object_type->name, array_type->name);
  }
  
  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  
  api->set_object_array_element(api, array, index, (SPVM_API_OBJECT*)object);
  
  XSRETURN(0);
}

SV*
get(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_array = ST(0);
  SV* sv_index = ST(1);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  // Runtime
  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  // Get array
  SPVM_OBJECT* array = SPVM_XS_UTIL_get_object(sv_array);
  
  // Array type id
  int32_t array_type_id = array->type_id;
  
  // Array type
  SPVM_TYPE* array_type = SPVM_LIST_fetch(compiler->types, array_type_id);
  
  // Element type name sv
  SV* sv_element_type_name = sv_2mortal(newSVpvn(array_type->name, strlen(array_type->name) - 2));
  const char* element_type_name = SvPV_nolen(sv_element_type_name);
  
  // Element type id
  int32_t element_type_id = api->get_type_id(api, element_type_name);
  SPVM_TYPE* element_type = SPVM_LIST_fetch(compiler->types, element_type_id);

  // Index
  int32_t index = (int32_t)SvIV(sv_index);
  SPVM_API_OBJECT* base_object = api->get_object_array_element(api, array, index);
  if (base_object != NULL) {
    api->inc_ref_count(api, base_object);
  }
  
  SV* sv_base_object;
  switch (element_type->id) {
    case SPVM_TYPE_C_ID_BYTE_ARRAY :
      sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Array::Byte");
      break;
    case SPVM_TYPE_C_ID_SHORT_ARRAY :
      sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Array::Short");
      break;
    case SPVM_TYPE_C_ID_INT_ARRAY :
      sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Array::Int");
      break;
    case SPVM_TYPE_C_ID_LONG_ARRAY :
      sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Array::Long");
      break;
    case SPVM_TYPE_C_ID_FLOAT_ARRAY :
      sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Array::Float");
      break;
    case SPVM_TYPE_C_ID_DOUBLE_ARRAY :
      sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Array::Double");
      break;
    case SPVM_TYPE_C_ID_STRING :
      sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Package::String");
      break;
    default : {
      if (element_type->dimension > 0) {
        sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, "SPVM::Core::Object::Array::Object");
      }
      else {
        SV* sv_element_type_name = sv_2mortal(newSVpv("SPVM::", 0));
        sv_catpv(sv_element_type_name, element_type->name);
        
        sv_base_object = SPVM_XS_UTIL_new_sv_object(base_object, SvPV_nolen(sv_element_type_name));
      }
    }
  }
  
  XPUSHs(sv_base_object);
  
  XSRETURN(1);
}

MODULE = SPVM::Core::Object::Array		PACKAGE = SPVM::Core::Object::Array


MODULE = SPVM		PACKAGE = SPVM

SV*
POSITIVE_INFINITY(...)
  PPCODE :
{
  (void)RETVAL;

  uint64_t positive_infinity_bits = 0x7ff0000000000000L;
  
  double positive_infinity;
  
  memcpy((void*)&positive_infinity, (void*)&positive_infinity_bits, sizeof(double));
  
  SV* sv_positive_infinity = sv_2mortal(newSVnv((NV)positive_infinity));
  
  XPUSHs(sv_positive_infinity);
  XSRETURN(1);
}

SV*
NEGATIVE_INFINITY(...)
  PPCODE :
{
  (void)RETVAL;

  uint64_t negative_infinity_bits = 0xfff0000000000000L;
  
  double negative_infinity;
  
  memcpy((void*)&negative_infinity, (void*)&negative_infinity_bits, sizeof(double));

  SV* sv_negative_infinity = sv_2mortal(newSVnv((NV)negative_infinity));

  XPUSHs(sv_negative_infinity);
  XSRETURN(1);
}

SV*
NaN(...)
  PPCODE :
{
  (void)RETVAL;

  uint64_t nan_bits = 0x7ff8000000000000L;
  
  double nan;
  
  memcpy((void*)&nan, (void*)&nan_bits, sizeof(double));
  
  SV* sv_nan = sv_2mortal(newSVnv((NV)nan));
  
  XPUSHs(sv_nan);
  XSRETURN(1);
}

SV*
get_objects_count(...)
  PPCODE:
{
  (void)RETVAL;
  
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  int32_t objects_count = api->get_objects_count(api);
  SV* sv_objects_count = sv_2mortal(newSViv(objects_count));
  
  XPUSHs(sv_objects_count);
  XSRETURN(1);
}

SV*
compile(...)
  PPCODE:
{
  (void)RETVAL;
  
  // Create compiler
  SPVM_COMPILER* compiler = SPVM_COMPILER_new();
  
  // Add package
  AV* av_package_infos = get_av("SPVM::PACKAGE_INFOS", 0);
  int32_t av_package_infos_length = (int32_t)av_len(av_package_infos) + 1;
  {
    int32_t i;
    for (i = 0; i < av_package_infos_length; i++) {
      SV** sv_package_info_ptr = av_fetch(av_package_infos, i, 0);
      SV* sv_package_info = sv_package_info_ptr ? *sv_package_info_ptr : &PL_sv_undef;
      HV* hv_package_info = (HV*)SvRV(sv_package_info);
      
      // Name
      SV** sv_name_ptr = hv_fetch(hv_package_info, "name", strlen("name"), 0);
      SV* sv_name = sv_name_ptr ? *sv_name_ptr : &PL_sv_undef;
      const char* name = SvPV_nolen(sv_name);
      
      // File
      SV** sv_file_ptr = hv_fetch(hv_package_info, "file", strlen("file"), 0);
      SV* sv_file = sv_file_ptr ? *sv_file_ptr : &PL_sv_undef;
      const char* file = SvPV_nolen(sv_file);
      
      // Line
      SV** sv_line_ptr = hv_fetch(hv_package_info, "line", strlen("line"), 0);
      SV* sv_line = sv_line_ptr ? *sv_line_ptr : &PL_sv_undef;
      int32_t line = (int32_t)SvIV(sv_line);
      
      // push package to compiler use stack
      SPVM_OP* op_use_package = SPVM_OP_new_op_use_from_package_name(compiler, name, file, line);
      SPVM_LIST_push(compiler->op_use_stack, op_use_package);
      SPVM_HASH_insert(compiler->op_use_symtable, name, strlen(name), op_use_package);
    }
  }
  
  // Add include paths
  AV* av_include_paths = get_av("main::INC", 0);;
  int32_t av_include_paths_length = (int32_t)av_len(av_include_paths) + 1;
  {
    int32_t i;
    for (i = 0; i < av_include_paths_length; i++) {
      SV** sv_include_path_ptr = av_fetch(av_include_paths, i, 0);
      SV* sv_include_path = sv_include_path_ptr ? *sv_include_path_ptr : &PL_sv_undef;
      char* include_path = SvPV_nolen(sv_include_path);
      SPVM_LIST_push(compiler->include_pathes, include_path);
    }
  }
  
  // Set compiler
  size_t iv_compiler = PTR2IV(compiler);
  SV* sviv_compiler = sv_2mortal(newSViv(iv_compiler));
  SV* sv_compiler = sv_2mortal(newRV_inc(sviv_compiler));
  sv_setsv(get_sv("SPVM::COMPILER", 0), sv_compiler);

  // Compile SPVM
  SPVM_COMPILER_compile(compiler);
  SV* sv_compile_success;
  
  if (compiler->error_count > 0) {
    sv_compile_success = sv_2mortal(newSViv(0));
  }
  else {
    sv_compile_success = sv_2mortal(newSViv(1));
  }
  
  XPUSHs(sv_compile_success);
  
  XSRETURN(1);
}

SV*
build_opcode(...)
  PPCODE:
{
  (void)RETVAL;

  // Get compiler
  SPVM_COMPILER* compiler = (SPVM_COMPILER*)SvIV(SvRV(get_sv("SPVM::COMPILER", 0)));
  
  // Build opcode
  SPVM_OPCODE_BUILDER_build_opcode_array(compiler);
  
  XSRETURN(0);
}

SV*
get_sub_name(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_sub_id = ST(0);
  int32_t sub_id = (int32_t)SvIV(sv_sub_id);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  SPVM_OP* op_sub = SPVM_LIST_fetch(compiler->op_subs, sub_id);
  SPVM_SUB* sub = op_sub->uv.sub;

  const char* sub_name = sub->abs_name;
  
  SV* sv_sub_name = sv_2mortal(newSVpvn(sub_name, strlen(sub_name)));
  
  XPUSHs(sv_sub_name);
  XSRETURN(1);
}

SV*
get_sub_names(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  AV* av_sub_names = (AV*)sv_2mortal((SV*)newAV());
  
  {
    int32_t sub_index;
    for (sub_index = 0; sub_index < compiler->op_subs->length; sub_index++) {
      SPVM_OP* op_sub = SPVM_LIST_fetch(compiler->op_subs, sub_index);
      SPVM_SUB* sub = op_sub->uv.sub;

      const char* sub_name = sub->abs_name;
      
      SV* sv_sub_name = sv_2mortal(newSVpvn(sub_name, strlen(sub_name)));
      av_push(av_sub_names, SvREFCNT_inc(sv_sub_name));
    }
  }
  
  SV* sv_sub_names = sv_2mortal(newRV_inc((SV*)av_sub_names));
  
  XPUSHs(sv_sub_names);
  XSRETURN(1);
}

SV*
get_native_sub_names(...)
  PPCODE:
{
  (void)RETVAL;
  
  // Get compiler
  SPVM_COMPILER* compiler = (SPVM_COMPILER*)SvIV(SvRV(get_sv("SPVM::COMPILER", 0)));
  
  SPVM_LIST* op_subs = compiler->op_subs;
  
  AV* av_sub_names = (AV*)sv_2mortal((SV*)newAV());
  {
    int32_t sub_index;
    for (sub_index = 0; sub_index < op_subs->length; sub_index++) {
      SPVM_OP* op_sub = SPVM_LIST_fetch(op_subs, sub_index);
      SPVM_SUB* sub = op_sub->uv.sub;
      
      if (sub->is_native) {
        const char* sub_name = sub->abs_name;
        SV* sv_sub_name = sv_2mortal(newSVpvn(sub_name, strlen(sub_name)));
        av_push(av_sub_names, SvREFCNT_inc(sv_sub_name));
      }
    }
  }
  
  SV* sv_sub_names = sv_2mortal(newRV_inc((SV*)av_sub_names));
  
  XPUSHs(sv_sub_names);
  XSRETURN(1);
}

SV*
get_no_native_sub_names(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  AV* av_sub_names = (AV*)sv_2mortal((SV*)newAV());
  
  {
    int32_t sub_index;
    for (sub_index = 0; sub_index < compiler->op_subs->length; sub_index++) {
      SPVM_OP* op_sub = SPVM_LIST_fetch(compiler->op_subs, sub_index);
      SPVM_SUB* sub = op_sub->uv.sub;

      if (!sub->is_native) {
        const char* sub_name = sub->abs_name;
        
        SV* sv_sub_name = sv_2mortal(newSVpvn(sub_name, strlen(sub_name)));
        av_push(av_sub_names, SvREFCNT_inc(sv_sub_name));
      }
    }
  }
  
  SV* sv_sub_names = sv_2mortal(newRV_inc((SV*)av_sub_names));
  
  XPUSHs(sv_sub_names);
  XSRETURN(1);
}

SV*
get_package_load_path(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();

  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;


  SV* sv_package_name = ST(0);
  const char* package_name = SvPV_nolen(sv_package_name);
  

  // Subroutine information
  SPVM_OP* op_package = SPVM_HASH_search(compiler->op_package_symtable, package_name, strlen(package_name));;
  SPVM_PACKAGE* package = op_package->uv.package;
  
  const char* package_load_path = package->load_path;
  
  SV* sv_package_load_path = sv_2mortal(newSVpvn(package_load_path, strlen(package_load_path)));
  
  XPUSHs(sv_package_load_path);
  
  XSRETURN(1);
}

SV*
bind_native_sub(...)
  PPCODE:
{
  (void)RETVAL;
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();

  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  SV* sv_native_sub_name = ST(0);
  SV* sv_native_address = ST(1);
  
  // Native subroutine name
  const char* native_sub_name = SvPV_nolen(sv_native_sub_name);
  
  // Native address
  IV native_address = SvIV(sv_native_address);
  
  // Set native address to subroutine
  SPVM_OP* op_sub = SPVM_HASH_search(compiler->op_sub_symtable, native_sub_name, strlen(native_sub_name));
  SPVM_SUB* sub = op_sub->uv.sub;
  
  sub->native_address = (void*)native_address;
  
  XSRETURN(0);
}

SV*
bind_jitcode_sub(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_sub_abs_name = ST(0);
  SV* sv_sub_native_address = ST(1);
  
  const char* sub_abs_name = SvPV_nolen(sv_sub_abs_name);
  void* sub_jit_address = (void*)SvIV(sv_sub_native_address);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();
  
  int32_t sub_id = api->get_sub_id(api, sub_abs_name);

  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  // Subroutine information
  SPVM_OP* op_sub = SPVM_LIST_fetch(compiler->op_subs, sub_id);
  SPVM_SUB* sub = op_sub->uv.sub;
  
  sub->jit_address = sub_jit_address;
  sub->is_jit = 1;
  
  XSRETURN(0);
}

SV*
build_field_symtable(...)
  PPCODE:
{
  (void)RETVAL;
  
  // Get compiler
  SPVM_COMPILER* compiler = (SPVM_COMPILER*)SvIV(SvRV(get_sv("SPVM::COMPILER", 0)));
  
  // Field symbol table
  HV* hv_field_symtable = get_hv("SPVM::FIELD_SYMTABLE", 0);
  
  // name, arg_types, return_type
  SPVM_LIST* op_packages = compiler->op_packages;
  {
    int32_t package_index;
    for (package_index = 0; package_index < op_packages->length; package_index++) {
      SPVM_OP* op_package = SPVM_LIST_fetch(op_packages, package_index);
      const char* package_name = op_package->uv.package->op_name->uv.name;
      
      HV* hv_package_info = (HV*)sv_2mortal((SV*)newHV());
      
      SPVM_LIST* op_fields = op_package->uv.package->op_fields;
      {
        int32_t field_index;
        for (field_index = 0; field_index < op_fields->length; field_index++) {
          SPVM_OP* op_field = SPVM_LIST_fetch(op_fields, field_index);
          SPVM_FIELD* field = op_field->uv.field;
          const char* field_name = field->op_name->uv.name;
          
          // Field type id
          int32_t field_type_id = field->op_type->uv.type->id;
          SV* sv_field_type_id = sv_2mortal(newSViv(field_type_id));

          // Field id
          int32_t field_id = field->index;
          SV* sv_field_id = sv_2mortal(newSViv(field_id));
          
          HV* hv_field = (HV*)sv_2mortal((SV*)newHV());
          (void)hv_store(hv_field, "id", strlen("id"), SvREFCNT_inc(sv_field_id), 0);
          (void)hv_store(hv_field, "id", strlen("id"), SvREFCNT_inc(sv_field_id), 0);
          (void)hv_store(hv_field, "type_id", strlen("type_id"), SvREFCNT_inc(sv_field_type_id), 0);
          SV* sv_field = sv_2mortal(newRV_inc((SV*)hv_field));
          
          (void)hv_store(hv_package_info, field_name, strlen(field_name), SvREFCNT_inc(sv_field), 0);
        }
      }
      
      SV* sv_package_info = sv_2mortal(newRV_inc((SV*)hv_package_info));
      (void)hv_store(hv_field_symtable, package_name, strlen(package_name), SvREFCNT_inc(sv_package_info), 0);
    }
  }
  
  XSRETURN(0);
}

SV*
build_runtime(...)
  PPCODE:
{
  (void)RETVAL;
  
  // Get compiler
  SPVM_COMPILER* compiler = (SPVM_COMPILER*)SvIV(SvRV(get_sv("SPVM::COMPILER", 0)));
  
  // Create run-time
  SPVM_RUNTIME* runtime = SPVM_COMPILER_new_runtime(compiler);
  
  // Set API
  SPVM_API* api = runtime->api;
  size_t iv_api = PTR2IV(api);
  SV* sviv_api = sv_2mortal(newSViv(iv_api));
  SV* sv_api = sv_2mortal(newRV_inc(sviv_api));
  sv_setsv(get_sv("SPVM::API", 0), sv_api);
  
  api->compile_jit_sub = &SPVM_XS_UTIL_compile_jit_sub;
  
  // JIT mode
  HV* hv_env = get_hv("ENV", 0);
  SV** sv_jit_mode_ptr = hv_fetch(hv_env, "SPVM_JIT_MODE", strlen("SPVM_JIT_MODE"), 0);
  const char* pv_jit_mode;
  if (sv_jit_mode_ptr) {
    pv_jit_mode = SvPV_nolen(*sv_jit_mode_ptr);
  }
  else {
    pv_jit_mode = "auto";
  }
  
  if (strcmp(pv_jit_mode, "auto") == 0) {
    runtime->jit_mode = SPVM_RUNTIME_C_JIT_MODE_AUTO;
  }
  else if (strcmp(pv_jit_mode, "all") == 0) {
    runtime->jit_mode = SPVM_RUNTIME_C_JIT_MODE_ALL;
  }
  else if (strcmp(pv_jit_mode, "none") == 0) {
    runtime->jit_mode = SPVM_RUNTIME_C_JIT_MODE_NONE;
  }
  else {
    croak("Unknown jit mode");
  }
  
  XSRETURN(0);
}

SV*
free_compiler(...)
  PPCODE:
{
  (void)RETVAL;
  
  // Get compiler
  SPVM_COMPILER* compiler = (SPVM_COMPILER*)SvIV(SvRV(get_sv("SPVM::COMPILER", 0)));
  
  // Free compiler
  SPVM_COMPILER_free(compiler);
  
  // Set undef to compiler
  sv_setsv(get_sv("SPVM::COMPILER", 0), &PL_sv_undef);
  
  XSRETURN(0);
}

SV*
call_sub(...)
  PPCODE:
{
  (void)RETVAL;
  
  SV* sv_sub_abs_name = ST(0);
  
  // API
  SPVM_API* api = SPVM_XS_UTIL_get_api();

  const char* sub_abs_name = SvPV_nolen(sv_sub_abs_name);
  int32_t sub_id = api->get_sub_id(api, sub_abs_name);
  
  SPVM_RUNTIME* runtime = (SPVM_RUNTIME*)api->get_runtime(api);
  SPVM_COMPILER* compiler = runtime->compiler;
  
  // Subroutine information
  SPVM_OP* op_sub = SPVM_LIST_fetch(compiler->op_subs, sub_id);
  SPVM_SUB* sub = op_sub->uv.sub;
  
  // Arguments
  {
    int32_t arg_index;
    // Check argument count
    if (items - 1 != sub->op_args->length) {
      croak("Argument count is defferent");
    }
    
    for (arg_index = 0; arg_index < sub->op_args->length; arg_index++) {
      SV* sv_value = ST(arg_index + 1);
      
      SPVM_OP* op_arg = SPVM_LIST_fetch(sub->op_args, arg_index);
      SPVM_TYPE* arg_type = op_arg->uv.my->op_type->uv.type;
      
      switch (arg_type->id) {
        case SPVM_TYPE_C_ID_BYTE : {
          int8_t value = (int8_t)SvIV(sv_value);
          call_sub_args[arg_index].byte_value = value;
          break;
        }
        case  SPVM_TYPE_C_ID_SHORT : {
          int16_t value = (int16_t)SvIV(sv_value);
          call_sub_args[arg_index].short_value = value;
          break;
        }
        case  SPVM_TYPE_C_ID_INT : {
          int32_t value = (int32_t)SvIV(sv_value);
          call_sub_args[arg_index].int_value = value;
          break;
        }
        case  SPVM_TYPE_C_ID_LONG : {
          int64_t value = (int64_t)SvIV(sv_value);
          call_sub_args[arg_index].long_value = value;
          break;
        }
        case  SPVM_TYPE_C_ID_FLOAT : {
          float value = (float)SvNV(sv_value);
          call_sub_args[arg_index].float_value = value;
          break;
        }
        case  SPVM_TYPE_C_ID_DOUBLE : {
          double value = (double)SvNV(sv_value);
          call_sub_args[arg_index].double_value = value;
          break;
        }
        default :
          if (!SvOK(sv_value)) {
            call_sub_args[arg_index].object_value = NULL;
          }
          else {
            if (sv_isobject(sv_value)) {
              SV* sv_base_object = sv_value;
              if (sv_derived_from(sv_base_object, "SPVM::Core::Object")) {
                
                SPVM_OBJECT* base_object = SPVM_XS_UTIL_get_object(sv_base_object);
                
                int32_t base_object_type_id = base_object->type_id;
                
                SPVM_TYPE* base_object_type = SPVM_LIST_fetch(compiler->types, base_object_type_id);
                
                if (base_object_type->id != arg_type->id) {
                  croak("Argument base_object type need %s, but %s", arg_type->name, base_object_type->name);
                }
                
                call_sub_args[arg_index].object_value = base_object;
              }
              else {
                croak("Object must be derived from SPVM::Core::Object");
              }
            }
            else {
              croak("Argument must be numeric value or SPVM::Core::Object subclass");
            }
          }
      }
    }
  }
  
  // Return type id
  SPVM_TYPE* return_type = sub->op_return_type->uv.type;
  int32_t return_type_id = return_type->id;
  
  // Return count
  int32_t return_count;
  switch (return_type_id) {
    case SPVM_TYPE_C_ID_VOID:  {
      api->call_void_sub(api, sub_id, call_sub_args);
      return_count = 0;
      break;
    }
    case SPVM_TYPE_C_ID_BYTE: {
      int8_t return_value = api->call_byte_sub(api, sub_id, call_sub_args);
      SV* sv_return_value = sv_2mortal(newSViv(return_value));
      XPUSHs(sv_return_value);
      return_count = 1;
      break;
    }
    case SPVM_TYPE_C_ID_SHORT: {
      int16_t return_value = api->call_short_sub(api, sub_id, call_sub_args);
      SV* sv_return_value = sv_2mortal(newSViv(return_value));
      XPUSHs(sv_return_value);
      return_count = 1;
      break;
    }
    case SPVM_TYPE_C_ID_INT: {
      int32_t return_value = api->call_int_sub(api, sub_id, call_sub_args);
      SV* sv_return_value = sv_2mortal(newSViv(return_value));
      XPUSHs(sv_return_value);
      return_count = 1;
      break;
    }
    case SPVM_TYPE_C_ID_LONG: {
      int64_t return_value = api->call_long_sub(api, sub_id, call_sub_args);
      SV* sv_return_value = sv_2mortal(newSViv(return_value));
      XPUSHs(sv_return_value);
      return_count = 1;
      break;
    }
    case SPVM_TYPE_C_ID_FLOAT: {
      float return_value = api->call_float_sub(api, sub_id, call_sub_args);
      SV* sv_return_value = sv_2mortal(newSVnv(return_value));
      XPUSHs(sv_return_value);
      return_count = 1;
      break;
    }
    case SPVM_TYPE_C_ID_DOUBLE: {
      double return_value = api->call_double_sub(api, sub_id, call_sub_args);
      SV* sv_return_value = sv_2mortal(newSVnv(return_value));
      XPUSHs(sv_return_value);
      return_count = 1;
      break;
    }
    default: {
      SPVM_API_OBJECT* return_value = api->call_object_sub(api, sub_id, call_sub_args);
      SV* sv_return_value = NULL;
      if (return_value != NULL) {
        api->inc_ref_count(api, return_value);
        
        switch(return_type_id) {
          case SPVM_TYPE_C_ID_BYTE_ARRAY :
            sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Array::Byte");
            break;
          case SPVM_TYPE_C_ID_SHORT_ARRAY :
            sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Array::Short");
            break;
          case SPVM_TYPE_C_ID_INT_ARRAY :
            sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Array::Int");
            break;
          case SPVM_TYPE_C_ID_LONG_ARRAY :
            sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Array::Long");
            break;
          case SPVM_TYPE_C_ID_FLOAT_ARRAY :
            sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Array::Float");
            break;
          case SPVM_TYPE_C_ID_DOUBLE_ARRAY :
            sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Array::Double");
            break;
          case SPVM_TYPE_C_ID_STRING :
            sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Package::String");
            break;
          default : {
            if (return_type->dimension > 0) {
              sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, "SPVM::Core::Object::Array::Object");
            }
            else {
              SV* sv_return_type_name = sv_2mortal(newSVpv("SPVM::", 0));
              sv_catpv(sv_return_type_name, return_type->name);
              
              sv_return_value = SPVM_XS_UTIL_new_sv_object(return_value, SvPV_nolen(sv_return_type_name));
            }
          }
        }
      }
      else {
        sv_return_value = &PL_sv_undef;
      }
      XPUSHs(sv_return_value);
      return_count = 1;
    }
  }
  SPVM_API_OBJECT* exception = api->get_exception(api);
  if (exception) {
    int32_t length = api->get_string_length(api, exception);
    char* exception_chars = (char*)api->get_string_chars(api, exception);
    SV* sv_exception = sv_2mortal(newSVpvn(exception_chars, length));
    croak("%s", SvPV_nolen(sv_exception));
  }
  
  XSRETURN(return_count);
}
