﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/directory-service-data/DirectoryServiceDataRequest.h>
#include <aws/directory-service-data/DirectoryServiceData_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DirectoryServiceData {
namespace Model {

/**
 */
class DeleteGroupRequest : public DirectoryServiceDataRequest {
 public:
  AWS_DIRECTORYSERVICEDATA_API DeleteGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteGroup"; }

  AWS_DIRECTORYSERVICEDATA_API Aws::String SerializePayload() const override;

  AWS_DIRECTORYSERVICEDATA_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p> A unique and case-sensitive identifier that you provide to make sure the
   * idempotency of the request, so multiple identical calls have the same effect as
   * one single call. </p> <p> A client token is valid for 8 hours after the first
   * request that uses it completes. After 8 hours, any request with the same client
   * token is treated as a new request. If the request succeeds, any future uses of
   * that token will be idempotent for another 8 hours. </p> <p> If you submit a
   * request with the same client token but change one of the other parameters within
   * the 8-hour idempotency window, Directory Service Data returns an
   * <code>ConflictException</code>. </p>  <p> This parameter is optional when
   * using the CLI or SDK. </p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  DeleteGroupRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The identifier (ID) of the directory that's associated with the group. </p>
   */
  inline const Aws::String& GetDirectoryId() const { return m_directoryId; }
  inline bool DirectoryIdHasBeenSet() const { return m_directoryIdHasBeenSet; }
  template <typename DirectoryIdT = Aws::String>
  void SetDirectoryId(DirectoryIdT&& value) {
    m_directoryIdHasBeenSet = true;
    m_directoryId = std::forward<DirectoryIdT>(value);
  }
  template <typename DirectoryIdT = Aws::String>
  DeleteGroupRequest& WithDirectoryId(DirectoryIdT&& value) {
    SetDirectoryId(std::forward<DirectoryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the group. </p>
   */
  inline const Aws::String& GetSAMAccountName() const { return m_sAMAccountName; }
  inline bool SAMAccountNameHasBeenSet() const { return m_sAMAccountNameHasBeenSet; }
  template <typename SAMAccountNameT = Aws::String>
  void SetSAMAccountName(SAMAccountNameT&& value) {
    m_sAMAccountNameHasBeenSet = true;
    m_sAMAccountName = std::forward<SAMAccountNameT>(value);
  }
  template <typename SAMAccountNameT = Aws::String>
  DeleteGroupRequest& WithSAMAccountName(SAMAccountNameT&& value) {
    SetSAMAccountName(std::forward<SAMAccountNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_directoryId;
  bool m_directoryIdHasBeenSet = false;

  Aws::String m_sAMAccountName;
  bool m_sAMAccountNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace DirectoryServiceData
}  // namespace Aws
