/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.search.pipeline.common;

import java.nio.charset.StandardCharsets;
import java.time.DayOfWeek;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.WeekFields;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import org.apache.lucene.search.BooleanClause;
import org.opensearch.action.search.SearchRequest;
import org.opensearch.common.hash.MurmurHash3;
import org.opensearch.common.time.DateFormatter;
import org.opensearch.common.time.DateFormatters;
import org.opensearch.core.common.Strings;
import org.opensearch.index.query.QueryBuilder;
import org.opensearch.index.query.QueryBuilderVisitor;
import org.opensearch.index.query.RangeQueryBuilder;
import org.opensearch.ingest.ConfigurationUtils;
import org.opensearch.search.pipeline.AbstractProcessor;
import org.opensearch.search.pipeline.Processor;
import org.opensearch.search.pipeline.SearchRequestProcessor;

public class TemporalRoutingSearchProcessor
extends AbstractProcessor
implements SearchRequestProcessor {
    public static final String TYPE = "temporal_routing_search";
    private static final String DEFAULT_FORMAT = "strict_date_optional_time";
    private final String timestampField;
    private final Granularity granularity;
    private final DateFormatter dateFormatter;
    private final boolean enableAutoDetection;
    private final boolean hashBucket;

    TemporalRoutingSearchProcessor(String tag, String description, boolean ignoreFailure, String timestampField, Granularity granularity, String format, boolean enableAutoDetection, boolean hashBucket) {
        super(tag, description, ignoreFailure);
        this.timestampField = timestampField;
        this.granularity = granularity;
        this.dateFormatter = DateFormatter.forPattern((String)format);
        this.enableAutoDetection = enableAutoDetection;
        this.hashBucket = hashBucket;
    }

    public String getType() {
        return TYPE;
    }

    public SearchRequest processRequest(SearchRequest request) throws Exception {
        if (request.routing() != null && !request.routing().isEmpty()) {
            return request;
        }
        HashSet<String> routingValues = new HashSet<String>();
        if (request.source() != null && request.source().query() != null) {
            TemporalRangeExtractionVisitor visitor = new TemporalRangeExtractionVisitor(routingValues);
            request.source().query().visit((QueryBuilderVisitor)visitor);
        }
        if (!routingValues.isEmpty()) {
            HashSet<String> computedRouting = new HashSet<String>();
            for (String temporalBucket : routingValues) {
                if (this.hashBucket) {
                    String routingValue = this.hashTemporalBucket(temporalBucket);
                    computedRouting.add(routingValue);
                    continue;
                }
                computedRouting.add(temporalBucket);
            }
            if (!computedRouting.isEmpty()) {
                String routing = String.join((CharSequence)",", computedRouting);
                request.routing(routing);
            }
        }
        return request;
    }

    private ZonedDateTime parseTimestamp(String timestamp) {
        TemporalAccessor accessor = this.dateFormatter.parse(timestamp);
        return DateFormatters.from((TemporalAccessor)accessor, (Locale)Locale.ROOT, (ZoneId)ZoneOffset.UTC);
    }

    private ZonedDateTime truncateToGranularity(ZonedDateTime dateTime) {
        switch (this.granularity.ordinal()) {
            case 0: {
                return dateTime.withMinute(0).withSecond(0).withNano(0);
            }
            case 1: {
                return dateTime.withHour(0).withMinute(0).withSecond(0).withNano(0);
            }
            case 2: {
                ZonedDateTime dayTruncated = dateTime.withHour(0).withMinute(0).withSecond(0).withNano(0);
                return dayTruncated.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY));
            }
            case 3: {
                return dateTime.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            }
        }
        throw new IllegalArgumentException("Unsupported granularity: " + String.valueOf((Object)this.granularity));
    }

    private ZonedDateTime incrementByGranularity(ZonedDateTime dateTime) {
        switch (this.granularity.ordinal()) {
            case 0: {
                return dateTime.plusHours(1L);
            }
            case 1: {
                return dateTime.plusDays(1L);
            }
            case 2: {
                return dateTime.plusWeeks(1L);
            }
            case 3: {
                return dateTime.plusMonths(1L);
            }
        }
        throw new IllegalArgumentException("Unsupported granularity: " + String.valueOf((Object)this.granularity));
    }

    private String createTemporalBucket(ZonedDateTime dateTime) {
        ZonedDateTime truncated = this.truncateToGranularity(dateTime);
        switch (this.granularity.ordinal()) {
            case 0: {
                return truncated.getYear() + "-" + String.format(Locale.ROOT, "%02d", truncated.getMonthValue()) + "-" + String.format(Locale.ROOT, "%02d", truncated.getDayOfMonth()) + "T" + String.format(Locale.ROOT, "%02d", truncated.getHour());
            }
            case 1: {
                return truncated.getYear() + "-" + String.format(Locale.ROOT, "%02d", truncated.getMonthValue()) + "-" + String.format(Locale.ROOT, "%02d", truncated.getDayOfMonth());
            }
            case 2: {
                int weekOfYear = truncated.get(WeekFields.ISO.weekOfWeekBasedYear());
                int weekYear = truncated.get(WeekFields.ISO.weekBasedYear());
                return weekYear + "-W" + String.format(Locale.ROOT, "%02d", weekOfYear);
            }
            case 3: {
                return truncated.getYear() + "-" + String.format(Locale.ROOT, "%02d", truncated.getMonthValue());
            }
        }
        throw new IllegalArgumentException("Unsupported granularity: " + String.valueOf((Object)this.granularity));
    }

    private String hashTemporalBucket(String temporalBucket) {
        byte[] bucketBytes = temporalBucket.getBytes(StandardCharsets.UTF_8);
        long hash = MurmurHash3.hash128((byte[])bucketBytes, (int)0, (int)bucketBytes.length, (long)0L, (MurmurHash3.Hash128)new MurmurHash3.Hash128()).h1;
        return String.valueOf(hash == Long.MIN_VALUE ? 0L : (hash < 0L ? -hash : hash));
    }

    public static enum Granularity {
        HOUR(ChronoUnit.HOURS),
        DAY(ChronoUnit.DAYS),
        WEEK(ChronoUnit.WEEKS),
        MONTH(ChronoUnit.MONTHS);

        private final ChronoUnit chronoUnit;

        private Granularity(ChronoUnit chronoUnit) {
            this.chronoUnit = chronoUnit;
        }

        public ChronoUnit getChronoUnit() {
            return this.chronoUnit;
        }

        public static Granularity fromString(String value) {
            try {
                return Granularity.valueOf(value.toUpperCase(Locale.ROOT));
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException("Invalid granularity: " + value + ". Supported values are: hour, day, week, month");
            }
        }
    }

    private class TemporalRangeExtractionVisitor
    implements QueryBuilderVisitor {
        private final Set<String> temporalBuckets;

        TemporalRangeExtractionVisitor(Set<String> temporalBuckets) {
            this.temporalBuckets = temporalBuckets;
        }

        public void accept(QueryBuilder qb) {
            RangeQueryBuilder rangeQuery;
            if (qb instanceof RangeQueryBuilder && TemporalRoutingSearchProcessor.this.timestampField.equals((rangeQuery = (RangeQueryBuilder)qb).fieldName())) {
                this.extractTemporalBucketsFromRange(rangeQuery);
            }
        }

        public QueryBuilderVisitor getChildVisitor(BooleanClause.Occur occur) {
            if (occur == BooleanClause.Occur.MUST || occur == BooleanClause.Occur.FILTER) {
                return this;
            }
            return QueryBuilderVisitor.NO_OP_VISITOR;
        }

        private void extractTemporalBucketsFromRange(RangeQueryBuilder rangeQuery) {
            try {
                Object from = rangeQuery.from();
                Object to = rangeQuery.to();
                if (from != null && to != null) {
                    ZonedDateTime fromDate = TemporalRoutingSearchProcessor.this.parseTimestamp(from.toString());
                    ZonedDateTime toDate = TemporalRoutingSearchProcessor.this.parseTimestamp(to.toString());
                    this.generateTemporalBucketsInRange(fromDate, toDate);
                } else if (from != null) {
                    ZonedDateTime fromDate = TemporalRoutingSearchProcessor.this.parseTimestamp(from.toString());
                    String bucket = TemporalRoutingSearchProcessor.this.createTemporalBucket(fromDate);
                    this.temporalBuckets.add(bucket);
                } else if (to != null) {
                    ZonedDateTime toDate = TemporalRoutingSearchProcessor.this.parseTimestamp(to.toString());
                    String bucket = TemporalRoutingSearchProcessor.this.createTemporalBucket(toDate);
                    this.temporalBuckets.add(bucket);
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
        }

        private void generateTemporalBucketsInRange(ZonedDateTime from, ZonedDateTime to) {
            ZonedDateTime current = TemporalRoutingSearchProcessor.this.truncateToGranularity(from);
            ZonedDateTime end = TemporalRoutingSearchProcessor.this.truncateToGranularity(to);
            int maxBuckets = 100;
            for (int bucketCount = 0; !current.isAfter(end) && bucketCount < maxBuckets; ++bucketCount) {
                String bucket = TemporalRoutingSearchProcessor.this.createTemporalBucket(current);
                this.temporalBuckets.add(bucket);
                current = TemporalRoutingSearchProcessor.this.incrementByGranularity(current);
            }
        }
    }

    public static final class Factory
    implements Processor.Factory<SearchRequestProcessor> {
        public TemporalRoutingSearchProcessor create(Map<String, Processor.Factory<SearchRequestProcessor>> processorFactories, String tag, String description, boolean ignoreFailure, Map<String, Object> config, Processor.PipelineContext pipelineContext) throws Exception {
            Granularity granularity;
            String timestampField = ConfigurationUtils.readStringProperty((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, config, (String)"timestamp_field");
            String granularityStr = ConfigurationUtils.readStringProperty((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, config, (String)"granularity");
            String format = ConfigurationUtils.readOptionalStringProperty((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, config, (String)"format");
            boolean enableAutoDetection = ConfigurationUtils.readBooleanProperty((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, config, (String)"enable_auto_detection", (boolean)true);
            boolean hashBucket = ConfigurationUtils.readBooleanProperty((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, config, (String)"hash_bucket", (boolean)false);
            if (format == null) {
                format = TemporalRoutingSearchProcessor.DEFAULT_FORMAT;
            }
            if (Strings.isNullOrEmpty((String)timestampField)) {
                throw ConfigurationUtils.newConfigurationException((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, (String)"timestamp_field", (String)"cannot be null or empty");
            }
            if (Strings.isNullOrEmpty((String)granularityStr)) {
                throw ConfigurationUtils.newConfigurationException((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, (String)"granularity", (String)"cannot be null or empty");
            }
            try {
                granularity = Granularity.fromString(granularityStr);
            }
            catch (IllegalArgumentException e) {
                throw ConfigurationUtils.newConfigurationException((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, (String)"granularity", (String)e.getMessage());
            }
            try {
                DateFormatter.forPattern((String)format);
            }
            catch (Exception e) {
                throw ConfigurationUtils.newConfigurationException((String)TemporalRoutingSearchProcessor.TYPE, (String)tag, (String)"format", (String)("invalid date format: " + e.getMessage()));
            }
            return new TemporalRoutingSearchProcessor(tag, description, ignoreFailure, timestampField, granularity, format, enableAutoDetection, hashBucket);
        }
    }
}

